#include <iostream>
#include "TObjArray.h"
#include "TCanvas.h"

#include "rootfilehandler.h"
#include "pixel.h" // class implemented
using namespace std;

/////////////////////////////// PUBLIC ///////////////////////////////////////

//============================= LIFECYCLE ====================================

Pixel::Pixel(
        int         pixelID,
        int         maxPulsorder,
        int         verbosityLevel,
        bool        stats,
        TFile*      filename
        )
{
    mConstructorType        = 1;
    mChid                   = pixelID;
    mStats                  = stats;
    mMaxPulseOrder          = maxPulsorder;
    mVerbosityLevel         = verbosityLevel;
    mRootFile               = filename;

    hMaxOverlay  = new TH2F*[mMaxPulseOrder];
    hEdgeOverlay = new TH2F*[mMaxPulseOrder];
    hMaxProfile  = new TProfile*[mMaxPulseOrder];
    hEdgeProfile = new TProfile*[mMaxPulseOrder];

    hPixelMax  = new TH1F*[mMaxPulseOrder];
    hPixelMedian = new TH1F*[mMaxPulseOrder];
    hPixelMean  = new TH1F*[mMaxPulseOrder];

    hPixelEdgeMax  = new TH1F*[mMaxPulseOrder];
    hPixelEdgeMedian = new TH1F*[mMaxPulseOrder];
    hPixelEdgeMean  = new TH1F*[mMaxPulseOrder];

    hList           = new TObjArray;

    LoadPulseHistos( );
    BookTemplateHistos();
}

Pixel::Pixel(
        int         pixelID,
        int         maxPulsorder,
        int         verbosityLevel,
        bool        stats,
        TFile*      filename,
        int         pixelOverlayXaxisLeft,
        int         pixelOverlayXaxisRight,
        float       bSLMean,
        float       gainMean,
        TString     options
        )
{
    mConstructorType        = 0;
    mChid                   = pixelID;
    mStats                  = stats;        ///TODO: HANDOVER THE VALUE
    mMaxPulseOrder          = maxPulsorder;
    mVerbosityLevel         = verbosityLevel;
    mOptions                = options;
    mPixelOverlayXaxisLeft  = pixelOverlayXaxisLeft;
    mPixelOverlayXaxisRight = pixelOverlayXaxisRight;
    mBSLMean                = bSLMean;
    mGainMean               = gainMean;
    mRootFile               = filename;

    hMaxOverlay  = new TH2F*[mMaxPulseOrder];
    hEdgeOverlay = new TH2F*[mMaxPulseOrder];
    hMaxProfile  = new TProfile*[mMaxPulseOrder];
    hEdgeProfile = new TProfile*[mMaxPulseOrder];

    if (mOptions.Contains("S") )
    {
        hSlopeRisingEdge = new TH1F*[mMaxPulseOrder];
    }

    if (mOptions.Contains("R") )
    {
        hRisingEdgeToMax = new TH1I*[mMaxPulseOrder];
    }

    if (mOptions.Contains("M") )
    {
        hPosOfMax = new TH1I*[mMaxPulseOrder];
    }

    hList           = new TObjArray;

    BookPixelHistos();
    BookDistributionHistos();
}

Pixel::~Pixel()
{
    if (mVerbosityLevel > 1) cout << endl << "...delete histograms of pixel " << mChid ;

    DeletePixelHistos();
    if (mConstructorType == 0)
    {
        DeleteDistributionHistos();
    }

    if (mConstructorType == 1)
    {
        DeleteTemplateHistos();
    }
    delete hList;
    hList = NULL;

}// ~Pixel


//============================= OPERATORS ====================================

//XX&
//XX::operator=(const XX&);
//{
//    return *this;

//}// =

//============================= OPERATIONS ===================================
void
Pixel::BookPixelHistos()
{
    if (mVerbosityLevel > 2) cout << endl << "...book pixel histograms" << endl;
    for (int order = 0; order < mMaxPulseOrder; order++)
    {
        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hMaxOverlay", order) << endl;
        hMaxOverlay[order]=new TH2F(
                    HistoName("hMaxOverlay", order),
                    HistoTitle("Overlay at peak maximum of detected pulses of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5 ,
                    512,
                    -55.5,
                    200.5
                    );

        hMaxOverlay[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hMaxOverlay[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hMaxOverlay[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hMaxOverlay[order]->SetStats(mStats);
        hList->Add( hMaxOverlay[order] );

//------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hEdgeOverlay", order) << endl;
        hEdgeOverlay[order] = new TH2F(
                    HistoName("hEdgeOverlay", order),
                    HistoTitle("Overlay at rising edge of detected pulses of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5 ,
                    512,
                    -55.5,
                    200.5
                    );

        hEdgeOverlay[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hEdgeOverlay[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hEdgeOverlay[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        hEdgeOverlay[order]->SetStats(mStats);
        hList->Add( hEdgeOverlay[order] );

    //------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hMaxProfile", order) << endl;
        hMaxProfile[order] = new TProfile(
                    HistoName("hMaxProfile", order),
                    HistoTitle("Mean value of each slice in overlay plot (Tprofile)", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,//nbinsx
                    (-1*mPixelOverlayXaxisLeft)-0.5,                 //xlow
                    mPixelOverlayXaxisRight-0.5 ,                    //xup
    //                512,                                            //nbinsy
                    -55.5,                                          //ylow
                    300.5,                                          //yup
                    "s");                                           //option
        hMaxProfile[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hMaxProfile[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hMaxProfile[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hMaxProfile[order]->SetStats(mStats);
        hList->Add( hMaxProfile[order] );


    //------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hEdgeProfile", order) << endl;
        hEdgeProfile[order] = new TProfile(
                    HistoName("hEdgeProfile", order),
                    HistoTitle("Mean value of each slice in overlay plot (Tprofile)", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,//nbinsx
                    (-1*mPixelOverlayXaxisLeft)-0.5,                 //xlow
                    mPixelOverlayXaxisRight-0.5 ,                    //xup
    //                512,                                            //nbinsy
                    -55.5,                                          //ylow
                    300.5,                                          //yup
                    "s");                                           //option
        hEdgeProfile[order]->SetLineColor(kRed);
        hEdgeProfile[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hEdgeProfile[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hEdgeProfile[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hEdgeProfile[order]->SetStats(mStats);
        hList->Add( hEdgeProfile[order] );

    }
    if (mVerbosityLevel > 2) cout << "...done" << endl;
}
//end of BookPixelHistos
//----------------------------------------------------------------------------

void
Pixel::BookDistributionHistos( )
{
    if (!mOptions.IsNull() )
    {
        int x_min = 0;
        int x_max = 0;

        for (int order =0; order < mMaxPulseOrder; order++)
        {
            if (mVerbosityLevel > 2) cout << endl
                                          << "...book distribution histograms"
                                          << endl;

            if (mOptions.Contains("S"))
            {
                if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hSlopeRisingEdge", order) << endl;
                hSlopeRisingEdge[order] = new TH1F (
                            HistoName("hSlopeRisingEdge", order),
                            HistoTitle("Distribution of rising edge's slope", order),
                            600,
                            -10.1,
                            10.1
                            );
                hSlopeRisingEdge[order]->SetAxisRange(
                                    -1,
                                    7,
                                    "X");
                hSlopeRisingEdge[order]->GetXaxis()->SetTitle( "Slope Amplitude/time [mV/timeslices]" );
                hSlopeRisingEdge[order]->GetYaxis()->SetTitle( "counts" );
                hSlopeRisingEdge[order]->SetStats(mStats);
                hList->Add( hSlopeRisingEdge[order] );
            }

            if (mOptions.Contains("R"))
            {
                x_min = -15;
                x_max = 35;

                if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hRisingEdgeToMax", order) << endl;
                hRisingEdgeToMax[order] = new TH1I (
                            HistoName("hRisingEdgeToMax", order),
                            HistoTitle("Distribution of distance between rising edge and pulse's maximum", order),
                            x_max -x_min,
                            x_min,
                            x_max
                            );
                hSlopeRisingEdge[order]->SetAxisRange(
                                    -5,
                                    25,
                                    "X");
                hRisingEdgeToMax[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
                hRisingEdgeToMax[order]->GetYaxis()->SetTitle( "counts" );
                hRisingEdgeToMax[order]->SetStats(mStats);
                hList->Add( hRisingEdgeToMax[order] );
            }

            if (mOptions.Contains("M"))
            {
                x_min = 10;
                x_max = 290;
                if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPosOfMax", order) << endl;
                hPosOfMax[order] = new TH1I (
                            HistoName("hPosOfMax", order),
                            HistoTitle("Distribution of pulse's maximum's positon", order),
                            x_max - x_min,
                            x_min,
                            x_max
                            );
                hPosOfMax[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
                hPosOfMax[order]->GetYaxis()->SetTitle( "counts" );
                hRisingEdgeToMax[order]->SetStats(mStats);
                hList->Add( hPosOfMax[order] );
            }

            if (mVerbosityLevel > 2) cout << "...done" << endl;
        }
    }
}
//end of BookDistributionHistos
//----------------------------------------------------------------------------

void
Pixel::BookTemplateHistos()
{
    if (mVerbosityLevel > 2) cout << endl << "...book pixel histograms" << endl;
    for (int order = 0; order < mMaxPulseOrder; order++)
    {
        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPixelMax", order) << endl;
        hPixelMax[order]=new TH1F(
                    HistoName("hPixelMax", order),
                    HistoTitle("Maximum value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5
                    );

        hPixelMax[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelMax[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelMax[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hPixelMax[order]->SetStats(mStats);
        hList->Add( hPixelMax[order] );

//------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPixelMean", order) << endl;
        hPixelMean[order] = new TH1F(
                    HistoName("hPixelMean", order),
                    HistoTitle("Mean value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5
                    );
        hPixelMean[order]->SetLineColor(kBlue);
        hPixelMean[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelMean[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelMean[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        hPixelMean[order]->SetStats(mStats);
        hList->Add( hPixelMean[order] );

 //------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPixelMedian", order) << endl;
        hPixelMedian[order] = new TH1F(
                    HistoName("hPixelMedian", order),
                    HistoTitle("Median value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,//nbinsx
                    (-1*mPixelOverlayXaxisLeft)-0.5,                 //xlow
                    mPixelOverlayXaxisRight-0.5                     //xup
                    );
        hPixelMedian[order]->SetLineColor(kRed);
        hPixelMedian[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelMedian[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelMedian[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hPixelMedian[order]->SetStats(mStats);
        hList->Add( hPixelMedian[order] );

    }
    if (mVerbosityLevel > 2) cout << "...done" << endl;
}
//end of BookTemplateHistos
//----------------------------------------------------------------------------

void
Pixel::BookEdgeTemplateHistos()
{
    if (mVerbosityLevel > 2) cout << endl << "...book pixel histograms" << endl;
    for (int order = 0; order < mMaxPulseOrder; order++)
    {
        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPixelEdgeMax", order) << endl;
        hPixelEdgeMax[order]=new TH1F(
                    HistoName("hPixelEdgeMax", order),
                    HistoTitle("Maximum value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5
                    );

        hPixelEdgeMax[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelEdgeMax[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelEdgeMax[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hPixelEdgeMax[order]->SetStats(mStats);
        hList->Add( hPixelEdgeMax[order] );

//------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPixelEdgeMean", order) << endl;
        hPixelEdgeMean[order] = new TH1F(
                    HistoName("hPixelEdgeMean", order),
                    HistoTitle("Mean value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5
                    );
        hPixelEdgeMean[order]->SetLineColor(kBlue);
        hPixelEdgeMean[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelEdgeMean[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelEdgeMean[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        hPixelEdgeMean[order]->SetStats(mStats);
        hList->Add( hPixelEdgeMean[order] );

 //------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPixelEdgeMedian", order) << endl;
        hPixelEdgeMedian[order] = new TH1F(
                    HistoName("hPixelEdgeMedian", order),
                    HistoTitle("Median value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,//nbinsx
                    (-1*mPixelOverlayXaxisLeft)-0.5,                 //xlow
                    mPixelOverlayXaxisRight-0.5                     //xup
                    );
        hPixelEdgeMedian[order]->SetLineColor(kRed);
        hPixelEdgeMedian[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelEdgeMedian[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelEdgeMedian[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hPixelEdgeMedian[order]->SetStats(mStats);
        hList->Add( hPixelEdgeMedian[order] );

    }
    if (mVerbosityLevel > 2) cout << "...done" << endl;
}
//end of BookEdgeTemplateHistos
//----------------------------------------------------------------------------


void
Pixel::DrawOverlayHistograms(
        TCanvas**    pixelCanvas,
        int*        histoFrameNr
        )
{
    if (mVerbosityLevel > 2) cout << endl << "...drawing pulse histograms" ;
    for (int pulse_order = 0; pulse_order < mMaxPulseOrder; pulse_order++)
    {
        pixelCanvas[pulse_order]->cd( histoFrameNr[0] );
        hMaxOverlay[pulse_order]->Draw("COLZ");
        pixelCanvas[pulse_order]->cd( histoFrameNr[2] );
        hMaxProfile[pulse_order]->Draw();
        hEdgeProfile[pulse_order]->Draw("SAME");

        pixelCanvas[pulse_order]->cd( histoFrameNr[1] );
        hEdgeOverlay[pulse_order]->Draw("COLZ");
    }
}
// end of DrawOverlayHistograms
//----------------------------------------------------------------------------

void
Pixel::DrawDistributionHistograms(
        TCanvas**    pixelCanvas,
        int*        histoFrameNr
        )
{
    if (mVerbosityLevel > 2) cout << endl << "...drawing distribution histograms" ;
    bool nothing_to_fill = true;
    for (int pulse_order = 0; pulse_order < mMaxPulseOrder; pulse_order++)
    {
        if (mOptions.Contains("S") )
        {
        pixelCanvas[pulse_order]->cd( histoFrameNr[0] );
        hSlopeRisingEdge[pulse_order]->Draw();
        nothing_to_fill = false;
        }

        if (mOptions.Contains("R") )
        {
        pixelCanvas[pulse_order]->cd( histoFrameNr[1] );
        hRisingEdgeToMax[pulse_order]->Draw();
        nothing_to_fill = false;
        }

        if (mOptions.Contains("M") )
        {
        pixelCanvas[pulse_order]->cd( histoFrameNr[2] );
        hPosOfMax[pulse_order]->Draw();
        nothing_to_fill = false;
        }

    }
    if (nothing_to_fill)
    {
        cout << endl << "there were NO DISTRIBUTION HISTOGRAMS to fill" << endl;
    }
}
// end of DrawDistributionHistograms
//----------------------------------------------------------------------------

void
Pixel::DrawTemplateHistograms(
        TCanvas**    pixelCanvas,
        int*        histoFrameNr
        )
{
    if (mVerbosityLevel > 2) cout << endl << "...drawing Template histograms" ;
    for (int pulse_order = 0; pulse_order < mMaxPulseOrder; pulse_order++)
    {
        pixelCanvas[pulse_order]->cd( histoFrameNr[0] );
        hPixelMax[pulse_order]->Draw();

        pixelCanvas[pulse_order]->cd( histoFrameNr[1] );
        hPixelMedian[pulse_order]->Draw();

        pixelCanvas[pulse_order]->cd( histoFrameNr[2] );
        hPixelMean[pulse_order]->Draw();

    }
}
// end of DrawTemplateHistograms
//----------------------------------------------------------------------------

void
Pixel::DrawEdgeTemplateHistograms(
        TCanvas**    pixelCanvas,
        int*        histoFrameNr
        )
{
    if (mVerbosityLevel > 2) cout << endl << "...drawing Template histograms" ;
    for (int pulse_order = 0; pulse_order < mMaxPulseOrder; pulse_order++)
    {
        pixelCanvas[pulse_order]->cd( histoFrameNr[0] );
        hPixelEdgeMax[pulse_order]->Draw();

        pixelCanvas[pulse_order]->cd( histoFrameNr[1] );
        hPixelEdgeMedian[pulse_order]->Draw();

        pixelCanvas[pulse_order]->cd( histoFrameNr[2] );
        hPixelEdgeMean[pulse_order]->Draw();

    }
}
// end of DrawTemplateHistograms
//----------------------------------------------------------------------------

void
Pixel::DeletePixelHistos()
{
    if (mVerbosityLevel > 2)
    {
        cout << endl
             << "\t...delete current overlay histograms of Pixel# " << mChid;
    }
    for (int order = 0;
         order < mMaxPulseOrder;
         order ++)
    {
        if (mVerbosityLevel > 3)
        cout << endl << "\t\t...deleting hMaxOverlay"
                                      << mChid << "_" << order
                                      << " hMaxOverlay[order] adr " << hMaxOverlay[order]
                                      << " hMaxOverlay adr " << hMaxOverlay
                                         ;

        delete hMaxOverlay[order];
        hMaxOverlay[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hEdgeOverlay"
                                      << mChid << "_" << order ;
        delete hEdgeOverlay[order];
        hEdgeOverlay[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hMaxProfile"
                                      << mChid << "_" << order ;
        delete hMaxProfile[order];
        hMaxProfile[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hMaxProfile2"
                                      << mChid << "_" << order ;
        delete hEdgeProfile[order];
        hEdgeProfile[order] = NULL;
    }
    if (mVerbosityLevel > 3) cout << endl << "\t...deleting histogram Arrays";

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hMaxOverlay";
    delete[] hMaxOverlay;
    hMaxOverlay = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hEdgeOverlay";
    delete[] hEdgeOverlay;
    hEdgeOverlay = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hMaxProfile";
    delete[] hMaxProfile;
    hMaxProfile = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hEdgeProfile";
    delete[] hEdgeProfile;
    hEdgeProfile = NULL;
}
// end of DeletePixelHistos
//----------------------------------------------------------------------------

void
Pixel::DeleteDistributionHistos()
{
    if (mVerbosityLevel > 2)
    {
        cout << endl
             << "\t...delete current distribution histograms" ;
    }

    for (int order = 0;
         order < mMaxPulseOrder;
         order ++)
    {
        if (mOptions.Contains("S"))
        {
            if (mVerbosityLevel > 3) cout << endl
                                          << "\t\t...deleting hSlopeRisingEdge"
                                          << mChid << "_" << order ;
            delete hSlopeRisingEdge[order];
        }

        if (mOptions.Contains("R"))
        {
            if (mVerbosityLevel > 3) cout << endl
                                          << "\t\t...deleting hRisingEdgeToMax"
                                          << mChid << "_" << order ;
            delete hRisingEdgeToMax[order];
        }

        if (mOptions.Contains("M"))
        {
            if (mVerbosityLevel > 3) cout << endl
                                          << "\t\t...deleting hPosOfMax"
                                          << mChid << "_" << order ;

            delete hPosOfMax[order];
        }

    }
    if (mVerbosityLevel > 3) cout << endl << "\t...deleting histogram Arrays";

    if (mOptions.Contains("S"))
    {
        if (mVerbosityLevel > 3)
            cout << endl << "\t\t...deleting hSlopeRisingEdge";
        delete[] hSlopeRisingEdge;
        hSlopeRisingEdge = NULL;
    }

    if (mOptions.Contains("R"))
    {
        if (mVerbosityLevel > 3)
            cout << endl << "\t\t...deleting hRisingEdgeToMax";
        delete[] hRisingEdgeToMax;
        hRisingEdgeToMax = NULL;
    }

    if (mOptions.Contains("M"))
    {
        if (mVerbosityLevel > 3)
            cout << endl << "\t\t...deleting hPosOfMax";
        delete[] hPosOfMax;
        hPosOfMax = NULL;
    }
}
// end of DeletePixelHistos
//----------------------------------------------------------------------------

void
Pixel::DeleteTemplateHistos()
{
    if (mVerbosityLevel > 2)
    {
        cout << endl
             << "\t...delete current template histograms of Pixel# " << mChid;
    }
    for (int order = 0;
         order < mMaxPulseOrder;
         order ++)
    {
        if (mVerbosityLevel > 3)
        cout << endl << "\t\t...deleting hPixelMax"
                                      << mChid << "_" << order
                                      << " hPixelMax[order] adr " << hPixelMax[order]
                                      << " hPixelMax adr " << hPixelMax
                                         ;

        delete hPixelMax[order];
        hPixelMax[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelMedian"
                                      << mChid << "_" << order ;
        delete hPixelMedian[order];
        hPixelMedian[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelMean"
                                      << mChid << "_" << order ;
        delete hPixelMean[order];
        hPixelMean[order] = NULL;

    }
    if (mVerbosityLevel > 3) cout << endl << "\t...deleting histogram Arrays";

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelMax";
    delete[] hPixelMax;
    hPixelMax = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelMedian";
    delete[] hPixelMedian;
    hPixelMedian = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelMean";
    delete[] hPixelMean;
    hPixelMean = NULL;
}
// end of DeleteTemplateHistos
//----------------------------------------------------------------------------

void
Pixel::DeleteEdgeTemplateHistos()
{
    if (mVerbosityLevel > 2)
    {
        cout << endl
             << "\t...delete current template histograms of Pixel# " << mChid;
    }
    for (int order = 0;
         order < mMaxPulseOrder;
         order ++)
    {
        if (mVerbosityLevel > 3)
        cout << endl << "\t\t...deleting hPixelEdgeMax"
                                      << mChid << "_" << order
                                      << " hPixelEdgeMax[order] adr " << hPixelEdgeMax[order]
                                      << " hPixelEdgeMax adr " << hPixelEdgeMax
                                         ;

        delete hPixelEdgeMax[order];
        hPixelEdgeMax[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelEdgeMedian"
                                      << mChid << "_" << order ;
        delete hPixelEdgeMedian[order];
        hPixelEdgeMedian[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelEdgeMean"
                                      << mChid << "_" << order ;
        delete hPixelEdgeMean[order];
        hPixelEdgeMean[order] = NULL;

    }
    if (mVerbosityLevel > 3) cout << endl << "\t...deleting histogram Arrays";

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelEdgeMax";
    delete[] hPixelEdgeMax;
    hPixelEdgeMax = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelEdgeMedian";
    delete[] hPixelEdgeMedian;
    hPixelEdgeMedian = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelEdgeMean";
    delete[] hPixelEdgeMean;
    hPixelEdgeMean = NULL;
}
// end of DeleteTemplateHistos
//----------------------------------------------------------------------------

void
Pixel::SavePixelHistograms(
        const char* outRootFileName,
        bool        saveResults
        )
{
    if (mVerbosityLevel > 2) cout << endl << "Saving histograms of Pixel# " << mChid;
    if (!saveResults) return;
    SaveHistograms(
            outRootFileName,
            CreateSubDirName( mChid ),
            hList,
            saveResults,
            mVerbosityLevel
            );
}
// end of SavePixelHistograms
//----------------------------------------------------------------------------

void
Pixel::LoadPulseHistos()
{
    mRootFile->cd();
    mRootFile->cd(CreateSubDirName( mChid ));
    for (int order = 0; order < mMaxPulseOrder; order++)
    {
        hMaxOverlay[order]  = (TH2F*)mRootFile->Get( HistoName("hMaxOverlay", order) );
        hEdgeOverlay[order] = (TH2F*)mRootFile->Get( HistoName("hEdgeOverlay", order) );
        hMaxProfile[order]  = (TProfile*)mRootFile->Get( HistoName("hMaxProfile", order) );
        hEdgeProfile[order] = (TProfile*)mRootFile->Get( HistoName("hEdgeProfile", order) );
    }
}
// end of LoadPulseHistos
//----------------------------------------------------------------------------

TString
Pixel::HistoName(
        TString     histoname,
        int         order
        )
{
    histoname +=   mChid;
    histoname +=   "_";
    histoname +=   order;
    return histoname;
}

TString
Pixel::HistoTitle(
        TString     histo_title,
        int         order
        )
{
    histo_title +=   " [Pixel_Order] ";
    histo_title +=   mChid;
    histo_title +=   "_";
    histo_title +=   order;
    return histo_title;
}



//============================= ACESS      ===================================
//============================= INQUIRY    ===================================
/////////////////////////////// PROTECTED  ///////////////////////////////////

/////////////////////////////// PRIVATE    ///////////////////////////////////








