#include "Dim.h"
#include "Event.h"
#include "Shell.h"
#include "StateMachineDim.h"
#include "Connection.h"
#include "Configuration.h"
#include "Console.h"
#include "Converter.h"
#include "DimServiceInfoList.h"

#include "tools.h"

#include "LocalControl.h"

#include "HeadersFTM.h"
#include "HeadersFAD.h"


namespace ba    = boost::asio;
namespace bs    = boost::system;
namespace dummy = ba::placeholders;

using namespace std;

// ------------------------------------------------------------------------

#include "DimDescriptionService.h"
#include "DimState.h"

// ------------------------------------------------------------------------

class StateMachineMCP : public StateMachineDim
{
private:
    enum states_t
    {
        kStateDimNetworkNA = 1,
        kStateDisconnected,
        kStateConnecting,
        kStateConnected,
        kStateIdle,
        kStateDummy, // Doesn't exist, kept to keep the numbers
        kStateConfiguring1,
        kStateConfiguring2,
        kStateConfiguring3,
        kStateConfigured,
        kStateTriggerOn,
        kStateTakingData,
//        kStateRunInProgress,
    };

    DimVersion fDim;
    DimDescribedState fDimFTM;
    DimDescribedState fDimFAD;
    DimDescribedState fDimLog;
    DimDescribedState fDimRC;

    DimDescribedService fService;

    int Print() const
    {
        Out() << fDim << endl;
        Out() << fDimFTM << endl;
        Out() << fDimFAD << endl;
        Out() << fDimLog << endl;
        Out() << fDimRC << endl;

        return GetCurrentState();
    }

    int GetReady()
    {
        return GetCurrentState();
    }

    int StopRun(const EventImp &)
    {
	if (fDimFTM.state()==FTM::kTriggerOn)
	{
            Message("Stopping FTM");
	    Dim::SendCommand("FTM_CONTROL/STOP_TRIGGER");
	}

        // FIXME: Do step 2 only when FTM is stopped
        if (fDimFAD.state()==FAD::kConnected)
        {
            //Dim::SendCommand("FAD_CONTROL/ENABLE_TRIGGER_LINE",      bool(false));
	    Message("Stopping FAD");
            Dim::SendCommand("FAD_CONTROL/ENABLE_CONTINOUS_TRIGGER", bool(false));
        }

        return GetCurrentState();
    }

    int Reset(const EventImp &)
    {
        fRunType = "";
	Message("Reseting configuration states of FAD and FTM");
	Dim::SendCommand("FTM_CONTROL/RESET_CONFIGURE");
	Dim::SendCommand("FAD_CONTROL/RESET_CONFIGURE");
        Update(kStateIdle);
        return kStateIdle;
        /*
        // FIMXE: Handle error states!
        if (fDimLog.state()>=20)//kSM_NightlyOpen
            Dim::SendCommand("DATA_LOGGER/STOP");

        if (fDimLog.state()==0)
            Dim::SendCommand("DATA_LOGGER/WAIT_FOR_RUN_NUMBER");

        if (fDimFAD.state()==FAD::kConnected)
        {
            Dim::SendCommand("FAD_CONTROL/ENABLE_TRIGGER_LINE", bool(false));
            Dim::SendCommand("FAD_CONTROL/ENABLE_CONTINOUS_TRIGGER", bool(false));
        }

        if (fDimFTM.state()==FTM::kTakingData)
            Dim::SendCommand("FTM_CONTROL/STOP");

        return GetCurrentState(); */
    }

    int64_t fMaxTime;
    int64_t fNumEvents;
    string  fRunType;

    int StartRun(const EventImp &evt)
    {
        if (fDimFTM.state()==-2)
        {
            Error("No connection to ftmcontrol (see PRINT).");
            return GetCurrentState();
        }
        if (fDimFAD.state()==-2)
        {
            Warn("No connection to fadcontrol (see PRINT).");
            return GetCurrentState();
        }
        if (fDimLog.state()==-2)
        {
            Warn("No connection to datalogger (see PRINT).");
            return GetCurrentState();
        }
        if (fDimRC.state()==-2)
        {
            Warn("No connection to ratecontrol (see PRINT).");
            return GetCurrentState();
        }

        fMaxTime   = evt.Get<int64_t>();
        fNumEvents = evt.Get<int64_t>(8);
        fRunType   = evt.Ptr<char>(16);

        ostringstream str;
        str << "Starting configuration '" << fRunType << "' for new run";
        if (fNumEvents>0 || fMaxTime>0)
            str << " [";
        if (fNumEvents>0)
            str << fNumEvents << " events";
        if (fNumEvents>0 && fMaxTime>0)
            str << " / ";
        if (fMaxTime>0)
            str << fMaxTime << "s";
        if (fNumEvents>0 || fMaxTime>0)
            str << "]";
        Message(str);

        Update(kStateConfiguring1);

        return kStateConfiguring1;
    }

    struct Value
    {
        uint64_t time;
        uint64_t nevts;
        char type[];
    };

    Value *GetBuffer()
    {
        const size_t len = sizeof(Value)+fRunType.length()+1;

        char *buf = new char[len];

        Value *val = reinterpret_cast<Value*>(buf);

        val->time  = fMaxTime;
        val->nevts = fNumEvents;

        strcpy(val->type, fRunType.c_str());

        return val;
    }

    void Update(int newstate)
    {
        Value *buf = GetBuffer();
        fService.setQuality(newstate);
        fService.setData(buf, sizeof(Value)+fRunType.length()+1);
        fService.Update();
        delete buf;
    }

    void ConfigureFAD()
    {
        Value *buf = GetBuffer();

	Message("Configuring FAD");
        Dim::SendCommand("FAD_CONTROL/CONFIGURE", buf, sizeof(Value)+fRunType.length()+1);

        delete buf;
    }

    int Execute()
    {
        // Dispatch (execute) at most one handler from the queue. In contrary
        // to run_one(), it doesn't wait until a handler is available
        // which can be dispatched, so poll_one() might return with 0
        // handlers dispatched. The handlers are always dispatched/executed
        // synchronously, i.e. within the call to poll_one()
        //poll_one();

        if (!fDim.online())
            return kStateDimNetworkNA;

        if (fDimFTM.state() >= FTM::kConnected &&
            fDimFAD.state() >= FAD::kConnected &&
            fDimLog.state() >= kSM_Ready)
        {
            if (GetCurrentState()==kStateConfiguring1)
            {
		if (fDimLog.state()<30/*kSM_WaitForRun*/)
		{
		    Message("Starting datalogger");
                    Dim::SendCommand("DATA_LOGGER/START_RUN_LOGGING");
		}
		Message("Configuring Trigger (FTM)");
                Dim::SendCommand("FTM_CONTROL/CONFIGURE", fRunType);
                Update(kStateConfiguring2);
                return kStateConfiguring2;
            }

            if (GetCurrentState()==kStateConfiguring2)
            {
                // FIMXE: Reset in case of error
                if ((/*fDimFTM.state() != FTM::kConfiguring2 &&*/
                     fDimFTM.state() != FTM::kConfigured) ||
                    fDimLog.state()<30 || fDimLog.state()>0xff)
                    return GetCurrentState();

                // FIMXE: This is to make sure that the rate control
                // has received the correct trigger setup already...
                //usleep(1000000);

                Message("Starting Rate Control");
                Dim::SendCommand("RATE_CONTROL/CALIBRATE");

                ConfigureFAD();
                Update(kStateConfiguring3);
                return kStateConfiguring3;
            }

            if (GetCurrentState()==kStateConfiguring3)
            {
                if (fDimFTM.state() != FTM::kConfigured ||
                    fDimFAD.state() != FAD::kConfigured ||
                    fDimRC.state()  < 6)
                    return GetCurrentState();

                Message("Starting Trigger (FTM)");
                Dim::SendCommand("FTM_CONTROL/START_TRIGGER");
                Update(kStateConfigured);
                return kStateConfigured;
            }

            if (GetCurrentState()==kStateConfigured)
            {
                if (fDimFTM.state() != FTM::kTriggerOn)
                    return GetCurrentState();

                Update(kStateTriggerOn);

                return kStateTriggerOn;
            }

            if (GetCurrentState()==kStateTriggerOn)
            {
                if (fDimFAD.state() != FAD::kWritingData)
                    return GetCurrentState();

                Update(kStateTakingData);

                return kStateTakingData;
            }

            if (GetCurrentState()==kStateTakingData)
            {
                if (fDimFTM.state()==FTM::kTriggerOn &&
                    fDimFAD.state()==FAD::kWritingData)
                    return kStateTakingData;

                Update(kStateIdle);
            }

            return kStateIdle;
        }

        /*
        if (fDimFTM.state() >= FTM::kConnected &&
            fDimFAD.state() >= FAD::kConnected &&
            fDimLog.state() >= kSM_Ready)
            return kStateIdle;
         */
        if (fDimFTM.state() >-2 &&
            fDimFAD.state() >-2 &&
            fDimLog.state() >-2 &&
            fDimRC.state() >-2)
            return kStateConnected;

        if (fDimFTM.state() >-2 ||
            fDimFAD.state() >-2 ||
            fDimLog.state() >-2 ||
            fDimRC.state() >-2)
            return kStateConnecting;

        return kStateDisconnected;
    }

public:
    StateMachineMCP(ostream &out=cout) : StateMachineDim(out, "MCP"),
        fDimFTM("FTM_CONTROL"),
        fDimFAD("FAD_CONTROL"),
        fDimLog("DATA_LOGGER"),
        fDimRC("RATE_CONTROL"),
        fService("MCP/CONFIGURATION", "X:1;X:1;C", "Run configuration information"
                 "|MaxTime[s]:Maximum time before the run gets stopped"
                 "|MaxEvents[num]:Maximum number of events before the run gets stopped"
                 "|Name[text]:Name of the chosen configuration")
    {
        // ba::io_service::work is a kind of keep_alive for the loop.
        // It prevents the io_service to go to stopped state, which
        // would prevent any consecutive calls to run()
        // or poll() to do nothing. reset() could also revoke to the
        // previous state but this might introduce some overhead of
        // deletion and creation of threads and more.

        fDim.Subscribe(*this);
        fDimFTM.Subscribe(*this);
        fDimFAD.Subscribe(*this);
        fDimLog.Subscribe(*this);
        fDimRC.Subscribe(*this);

        // State names
        AddStateName(kStateDimNetworkNA, "DimNetworkNotAvailable",
                     "DIM dns server not available.");

        AddStateName(kStateDisconnected, "Disconnected",
                     "Neither ftmctrl, fadctrl, datalogger nor rate control online.");

        AddStateName(kStateConnecting, "Connecting",
                     "Either ftmctrl, fadctrl, datalogger or rate control not online.");

        AddStateName(kStateConnected, "Connected",
                     "All needed subsystems online.");

        AddStateName(kStateIdle, "Idle",
                     "Waiting for next configuration command");

        AddStateName(kStateConfiguring1, "Configuring1",
                     "Starting configuration procedure, checking Datalogger state");

        AddStateName(kStateConfiguring2, "Configuring2",
                     "Waiting for FTM and Datalogger to get ready");

        AddStateName(kStateConfiguring3, "Configuring3",
                     "Waiting for FADs and rate control to get ready");

        AddStateName(kStateConfigured, "Configured",
                     "Everything is configured, trigger will be switched on now");

        AddStateName(kStateTriggerOn, "TriggerOn",
                     "The trigger is switched on, waiting for FAD to receive data");

        AddStateName(kStateTakingData, "TakingData",
                     "The trigger is switched on, FADs are sending data");


        AddEvent("START", "X:2;C")//, kStateIdle)
            (bind(&StateMachineMCP::StartRun, this, placeholders::_1))
            ("Start the configuration and data taking for a run-type of a pre-defined setup"
             "|TimeMax[s]:Maximum number of seconds before the run will be closed automatically"
             "|NumMax[count]:Maximum number events before the run will be closed automatically"
             "|Name[text]:Name of the configuration to be used for taking data");

        AddEvent("STOP")
            (bind(&StateMachineMCP::StopRun, this, placeholders::_1))
            ("Stops the trigger (either disables the FTM trigger or the internal DRS trigger)");

        AddEvent("RESET", kStateConfiguring1, kStateConfiguring2, kStateConfiguring3, kStateConfigured)
            (bind(&StateMachineMCP::Reset, this, placeholders::_1))
            ("If a configuration blockes because a system cannot configure itself properly, "
             "this command can be called to leave the configuration procedure. The command "
             "is also propagated to FTM and FAD");

        AddEvent("PRINT")
            (bind(&StateMachineMCP::Print, this))
            ("Print the states and connection status of all systems connected to the MCP.");
    }

    int EvalOptions(Configuration &)
    {
        return -1;
    }
};

// ------------------------------------------------------------------------

#include "Main.h"

template<class T>
int RunShell(Configuration &conf)
{
    return Main::execute<T, StateMachineMCP>(conf);
}

/*
 Extract usage clause(s) [if any] for SYNOPSIS.
 Translators: "Usage" and "or" here are patterns (regular expressions) which
 are used to match the usage synopsis in program output.  An example from cp
 (GNU coreutils) which contains both strings:
  Usage: cp [OPTION]... [-T] SOURCE DEST
    or:  cp [OPTION]... SOURCE... DIRECTORY
    or:  cp [OPTION]... -t DIRECTORY SOURCE...
 */
void PrintUsage()
{
    cout <<
        "The ftmctrl controls the FSC (FACT Slow Control) board.\n"
        "\n"
        "The default is that the program is started without user intercation. "
        "All actions are supposed to arrive as DimCommands. Using the -c "
        "option, a local shell can be initialized. With h or help a short "
        "help message about the usuage can be brought to the screen.\n"
        "\n"
        "Usage: fscctrl [-c type] [OPTIONS]\n"
        "  or:  fscctrl [OPTIONS]\n";
    cout << endl;
}

void PrintHelp()
{
    Main::PrintHelp<StateMachineMCP>();

    /* Additional help text which is printed after the configuration
     options goes here */

    /*
     cout << "bla bla bla" << endl << endl;
     cout << endl;
     cout << "Environment:" << endl;
     cout << "environment" << endl;
     cout << endl;
     cout << "Examples:" << endl;
     cout << "test exam" << endl;
     cout << endl;
     cout << "Files:" << endl;
     cout << "files" << endl;
     cout << endl;
     */
}

int main(int argc, const char* argv[])
{
    Configuration conf(argv[0]);
    conf.SetPrintUsage(PrintUsage);
    Main::SetupConfiguration(conf);

    if (!conf.DoParse(argc, argv, PrintHelp))
        return -1;

    //try
    {
        // No console access at all
        if (!conf.Has("console"))
        {
//            if (conf.Get<bool>("no-dim"))
//                return RunShell<LocalStream, StateMachine, ConnectionFSC>(conf);
//            else
                return RunShell<LocalStream>(conf);
        }
        // Cosole access w/ and w/o Dim
/*        if (conf.Get<bool>("no-dim"))
        {
            if (conf.Get<int>("console")==0)
                return RunShell<LocalShell, StateMachine, ConnectionFSC>(conf);
            else
                return RunShell<LocalConsole, StateMachine, ConnectionFSC>(conf);
        }
        else
*/        {
            if (conf.Get<int>("console")==0)
                return RunShell<LocalShell>(conf);
            else
                return RunShell<LocalConsole>(conf);
        }
    }
    /*catch (std::exception& e)
    {
        cerr << "Exception: " << e.what() << endl;
        return -1;
    }*/

    return 0;
}
