/* ============================================================ 

Edd - Evidence Data Display

Qt-based graphical user interface for the Evidence contron system

December 2009, Oliver Grimm

============================================================ */

#include "Edd.h"

Qt::GlobalColor LineColors[] = {Qt::black, Qt::blue, Qt::red, Qt::green, Qt::white,
   	Qt::darkRed, Qt::darkGreen, Qt::darkBlue, Qt::cyan,
    	Qt::darkCyan, Qt::magenta, Qt::darkMagenta,
	Qt::gray, Qt::darkGray, Qt::lightGray};

//////////////////////////////////////////
// Text display for arbitary DIM service//
//////////////////////////////////////////

// Constructor
Edd_Indicator::Edd_Indicator(char *DIMService, QWidget *P): QLineEdit(P) {

  ServiceName = new char [strlen(DIMService)+1];
  strcpy(ServiceName, DIMService);
  
  // Widget properties
  setReadOnly(true);
  setMaximumWidth(100);
  connect(this, SIGNAL(YEP(QString)), this, SLOT(setText(QString)));

  // DIM client
  Data = new DimStampedInfo(DIMService, INT_MAX, (char *) NO_LINK, this);
}

// Destructor
Edd_Indicator::~Edd_Indicator() {
  delete Data;
  delete[] ServiceName;
}

// Handling of DIM service update
void Edd_Indicator::infoHandler() {

  // Check if service available
  QPalette Pal = palette();  
  if (getInfo()->getSize() == strlen(NO_LINK)+1 && strcmp(getInfo()->getString(), NO_LINK) == 0) {
	emit(YEP(QString("n/a")));
	setStatusTip(QString("%1:  unavailable").arg(ServiceName));
    Pal.setColor(backgroundRole(), Qt::red);
    setPalette(Pal);
	return;
  }
  Pal.setColor(backgroundRole(), Qt::white);
  setPalette(Pal);

  // Translate data into ASCII
  char *Text = EvidenceServer::ToString(getInfo());
  QString S;

  if (Text != NULL) {
    QTextStream(&S) << Text;
	free(Text);
  }
  else QTextStream(&S) << "Cannot interpret format identifier";
    
  if (strlen(getInfo()->getFormat()) > 1) {
    QTextStream(&S) << " (DIM format string longer)";
  }
  
  // Trigger display update
  emit(YEP(S));
  
  // Update status tip
  QDateTime Time = QDateTime::fromTime_t(getInfo()->getTimestamp()); 
  setStatusTip(QString("%1:  Last update %2  Format '%3'").arg(ServiceName, Time.toString()).arg(getInfo()->getFormat()));
}

// Open plot if mouse release within widget
void Edd_Indicator::mouseReleaseEvent(QMouseEvent *Event) {

  if (Event->button()==Qt::LeftButton && contentsRect().contains(Event->pos())) {
    Edd_Plot *Graph = new Edd_Plot(ServiceName, NULL);
    Graph->show();
  }
}

// Handling of mouse press event: Register start position for drag
void Edd_Indicator::mousePressEvent(QMouseEvent *Event) {

  if (Event->button() == Qt::LeftButton) dragStart = Event->pos();
}

// Handling of dragging (Drag and MimeData will be deleted by Qt)
void Edd_Indicator::mouseMoveEvent(QMouseEvent *Event) {

  if ((Event->buttons() & Qt::LeftButton) == 0) return;
  if ((Event->pos()-dragStart).manhattanLength() < QApplication::startDragDistance()) return;

  QDrag *Drag = new QDrag(this);
  QMimeData *MimeData = new QMimeData;
  MimeData->setText(QString(ServiceName));
  Drag->setMimeData(MimeData);
  Drag->exec();
}


//////////////////////////////////
// History plot for DIM service //
//////////////////////////////////

//
// Constructor
//
Edd_Plot::Edd_Plot(char *DIMService, QWidget *P): QwtPlot(P) {

  setAcceptDrops(true);

  // Graph properties
  QwtText XAxisTitle("Time (RJD-55000)");
  XAxisTitle.setFont(QFont("Helvetica", 10));
  setAxisTitle(QwtPlot::xBottom, XAxisTitle);
  setAutoReplot(false);
  setCanvasBackground(QColor(Qt::yellow));
  
  Zoomer = new QwtPlotZoomer(QwtPlot::xBottom,QwtPlot::yLeft,canvas());
  connect(Zoomer, SIGNAL(zoomed(const QwtDoubleRect &)), this, SLOT(HandleZoom(const QwtDoubleRect &)));
  Panner = new QwtPlotPanner(canvas());
  Panner->setMouseButton(Qt::LeftButton, Qt::ShiftModifier);
  Grid = new QwtPlotGrid;
  Grid->setMajPen(QPen(Qt::gray, 0, Qt::DotLine));
  Grid->attach(this);
  Legend = new QwtLegend();
  insertLegend(Legend, QwtPlot::TopLegend);
    
  // Threads may not call replot directly, but only through this signal
  connect(this, SIGNAL(YEP()), this, SLOT(UpdatePlot()));

  // Context menu
  Menu = new QMenu(this);
  YLogAction = Menu->addAction("y scale log", this, SLOT(UpdatePlot()));
  YLogAction->setCheckable(true);
  NormAction = Menu->addAction("Normalize", this, SLOT(UpdatePlot()));
  NormAction->setCheckable(true);
  Menu->addAction("Zoom out", this, SLOT(MenuZoomOut()));
  Menu->addAction("Single trace", this, SLOT(MenuSingleTrace()));
  Menu->addSeparator();
  Menu->addAction("Save plot", this, SLOT(MenuSave()));
  Menu->addAction("Print plot", this, SLOT(MenuPrint()));

  // DIM client
  AddService(DIMService);
}

//
// Destructor (items with parent widget are automatically deleted)
//
Edd_Plot::~Edd_Plot() {

  for (int i=0; i<Items.size(); i++) {
    delete Items[i].Data;
    delete Items[i].Signal;
    delete[] Items[i].x;
    delete[] Items[i].y;
  }  
  delete Grid;
}

//
// Add history service to plot
//
void Edd_Plot::AddService(char *DIMService) {

  // Generate name of DIM service
  QString Name(DIMService);
  Name.append(".hist");

  // Lock this thread (because infoHandler() might be called before list insertion)
  QMutexLocker Locker(&Mutex);

  // Generate new curve and subscribe to service
  struct PlotItem New;
  New.Signal = new QwtPlotCurve;
  New.Signal->attach(this);
  New.Signal->setTitle(Name);
  New.Signal->setPen(QColor(LineColors[Items.size()%(sizeof(LineColors)/sizeof(Qt::GlobalColor))]));
  New.x = NULL;
  New.y = NULL;
  New.Data = new DimStampedInfo(Name.toAscii(), NO_LINK, this); 

  Items.append(New);
}

//
// Handle update of DIM service
//
void Edd_Plot::infoHandler() {

  // Check if service available
  if (getInfo()->getSize() == strlen(NO_LINK)+1 && strcmp(getInfo()->getString(), NO_LINK) == 0) {
    setStatusTip(QString("%1:  unavailable").arg(getInfo()->getName()));
    return;
  }

  // Lock this thread (see AddService())
  QMutexLocker Locker(&Mutex);

  // Determine which plot item this call belongs to
  int ItemNo;
  for (ItemNo=0; ItemNo<Items.size(); ItemNo++) if (Items[ItemNo].Data == getInfo()) break;
  if (ItemNo == Items.size()) return;  // safety check
  
  EvidenceHistoryItem *Curr = (EvidenceHistoryItem *) getInfo()->getData();
  int Count=0, DataPoints = getInfo()->getSize()/sizeof(struct EvidenceHistoryItem);

  delete[] Items[ItemNo].x;
  delete[] Items[ItemNo].y;
  Items[ItemNo].x = new double [DataPoints];
  Items[ItemNo].y = new double [DataPoints];

  // Find oldest item
  int Oldest;
  for (Oldest=1; Oldest<DataPoints; Oldest++) {
    if (Curr[Oldest].Seconds < Curr[Oldest-1].Seconds) break;
  }

  // Plot data starting with oldest value
  for (int i=0; i<DataPoints; i++) {
    Items[ItemNo].x[Count] = (double) Curr[(i+Oldest)%DataPoints].Seconds;
    Items[ItemNo].y[Count] = Curr[(i+Oldest)%DataPoints].Value;
    if (Items[ItemNo].x[Count] != 0) Count++;
  }
  
  // Find smallest and largest item
  double Smallest = DBL_MAX, Largest = DBL_MIN;
  for (int i=0; i<Count; i++) {
    if (Largest < Items[ItemNo].y[i]) Largest = Items[ItemNo].y[i];
    if (Smallest > Items[ItemNo].y[i]) Smallest = Items[ItemNo].y[i];
  }
  Items[ItemNo].Smallest = Smallest;
  Items[ItemNo].Largest = Largest;
  Items[ItemNo].Count = Count;
  
  // Update status tip
  QDateTime Time = QDateTime::fromTime_t(getInfo()->getTimestamp()); 
  setStatusTip(QString("%1:  Last update %2  Format '%3'").arg(getInfo()->getName(), Time.toString()).arg(getInfo()->getFormat()));

  emit(YEP());
}


//
// Update all curves in plot
//
void Edd_Plot::UpdatePlot() {

  QMutexLocker Locker(&Mutex);

  if (!YLogAction->isChecked()) {
    setAxisScaleEngine(QwtPlot::yLeft, new QwtLinearScaleEngine);
  }
  else setAxisScaleEngine(QwtPlot::yLeft, new QwtLog10ScaleEngine);

  for (int ItemNo=0; ItemNo<Items.size(); ItemNo++) {
  
    double *x = new double [Items[ItemNo].Count];
    double *y = new double [Items[ItemNo].Count];

    // Adapt time scale and normalize y scale if requested
    for (int i=0; i<Items[ItemNo].Count; i++) {
      x[i] = Items[ItemNo].x[i]/86400 + 40587.5 - 55000;
      y[i] = Items[ItemNo].y[i];
      if (NormAction->isChecked()) {
	if (Items[ItemNo].Smallest != Items[ItemNo].Largest) y[i] = (Items[ItemNo].y[i] - Items[ItemNo].Smallest)/(Items[ItemNo].Largest-Items[ItemNo].Smallest);
	else y[i] = 1;
      }  
    }

    // Plot datas
    Items[ItemNo].Signal->setData(x, y, Items[ItemNo].Count);
    Items[ItemNo].Signal->show();
    Zoomer->setZoomBase(Items[ItemNo].Signal->boundingRect());

    delete[] x;
    delete[] y;
  }
  replot();
}

//
// Reset graph axes to autoscale when fully unzoomed
//
void Edd_Plot::HandleZoom(const QwtDoubleRect &) {

  if(Zoomer->zoomRectIndex() == 0) {
    setAxisAutoScale(QwtPlot::xBottom);
    setAxisAutoScale(QwtPlot::yLeft);
  }
}

//
// Drag and drop methods
//

void Edd_Plot::dragEnterEvent(QDragEnterEvent *Event) {
    
  if (Event->mimeData()->hasFormat("text/plain")) Event->acceptProposedAction();
}

void Edd_Plot::dropEvent(QDropEvent *Event) {

  AddService(Event->mimeData()->text().toAscii().data());
}
    
//
// Context menu and its functions
//
void Edd_Plot::contextMenuEvent(QContextMenuEvent *Event) {

  Menu->exec(Event->globalPos());
}

void Edd_Plot::MenuZoomOut() {

  Zoomer->zoom(0);
  emit YEP();
}

void Edd_Plot::MenuSingleTrace() {

  QMutexLocker Locker(&Mutex);

  while (Items.size() > 1) {  
    delete Items.last().Data;
    delete Items.last().Signal;
    delete[] Items.last().x;
    delete[] Items.last().y;
    Items.takeLast();
  }
  emit(YEP());
}

void Edd_Plot::MenuPrint() {

  QPrinter *Printer = new QPrinter;
  QPrintDialog *PrintDialog = new QPrintDialog(Printer, this);
  if (PrintDialog->exec() == QDialog::Accepted) {
    QPainter Painter(Printer);
    QPixmap Pixmap = QPixmap::grabWidget(this);
    Painter.drawPixmap(0, 0, Pixmap);
  }
  delete Printer;	delete PrintDialog;
}

void Edd_Plot::MenuSave() {

  QString Filename = QFileDialog::getSaveFileName(this,
     "Filename of image", "/home/ogrimm/ddd", "Image files (*.bmp *.jpg *.png *.ppm *.tiff *.xbm *.xpm);;All files (*)");
  if (Filename.length()>0) {
    QPixmap Pixmap = QPixmap::grabWidget(this);
    if(!Pixmap.save(Filename)) {
      QMessageBox::warning(this, "ddd Message","Could not write image file.",QMessageBox::Ok);
      remove(Filename.toAscii().data());
    }
  }
}

//
// Main GUI
//
GUI::GUI() {
  
  // Set features of main window
  Central = new QWidget(this);
  setCentralWidget(Central);
  setStatusBar (new QStatusBar(this));
  
   // TextBox for value
  Value = new Edd_Indicator((char *) "SQM/NSB", Central);
  Graph = new Edd_Plot((char *) "SQM/NSB", Central);

  Value1 = new Edd_Indicator((char *) "BIAS/VOLT/ID00/00-000", Central);
  //Graph1 = new Edd_Plot((char *) "BIAS/VOLT/ID00/00-000", Central);

  // Clock (updated every second)
  //Clock = new QwtAnalogClock(Central);
  //Clock->scaleDraw()->setPenWidth(3);
  //Clock->setLineWidth(6);
  //Clock->setFrameShadow(QwtDial::Sunken);
  //Clock->setGeometry(0,0,10,10);
  //Clock->setTime();
  
  //QTimer *Timer = new QTimer(Clock);
  //Timer->connect(Timer, SIGNAL(timeout()), Clock, SLOT(setCurrentTime()));
  //Timer->start(1000);

  MainWidget = new QWidget();

  // Layout of all widgets
  MainLayout = new QGridLayout(MainWidget);
  MainLayout->addWidget(Value, 0, 0, 1, 1);
  //MainLayout->addWidget(Clock, 0, 1, 1, 1);
  MainLayout->addWidget(Graph, 1, 0, 1, 2);
  //MainLayout->setColumnStretch(1, 10);
  
  MainLayout->addWidget(Value1, 0, 1, 1, 1);
  //MainLayout->addWidget(Clock, 0, 1, 1, 1);
  //MainLayout->addWidget(Graph1, 2, 0, 1, 2);

  BiasWidget = new QWidget();
  BiasLayout = new QGridLayout(BiasWidget);
  for (int i=0; i<19; i++) {
    char *Item;
    if (asprintf(&Item, "BIAS/VOLT/ID00/00-%.3d", i) != -1) {
      Value = new Edd_Indicator(Item, NULL);
      BiasLayout->addWidget(Value, i%10, 0+i/10, 1, 1);      
      free(Item);
    }
    if (asprintf(&Item, "BIAS/VOLT/ID00/01-%.3d", i) != -1) {
      Value = new Edd_Indicator(Item, NULL);
      BiasLayout->addWidget(Value, i%10, 2+i/10, 1, 1);      
      free(Item);
    }
  }

  EnvironmentWidget = new QWidget();
  EnvironmentLayout = new QGridLayout(EnvironmentWidget);
  for (int i=0; i<10; i++) {
    char *Item;
    if (asprintf(&Item, "ARDUINO/VAL%.2d", i) != -1) {
      Value = new Edd_Indicator(Item, NULL);
      EnvironmentLayout->addWidget(Value, i%5, i/5, 1, 1);      
      free(Item);
    }
  }
    
  // Tab widget
  TabWidget = new QTabWidget(Central);
  TabWidget->addTab(MainWidget, "&Main");
  TabWidget->addTab(BiasWidget, "&Bias");
  TabWidget->addTab(EnvironmentWidget, "&Environment");

  // Menu bar
  QMenu* Menu = menuBar()->addMenu("&Menu");
  Menu->addAction("About", this, SLOT(MenuAbout()));
  Menu->addSeparator();
  QAction* QuitAction = Menu->addAction("Quit", qApp, SLOT(quit()));
  QuitAction->setShortcut(Qt::CTRL + Qt::Key_Q);
}  
    
GUI::~GUI() {
  delete Central;
}

void GUI::MenuAbout() {
  QMessageBox::about(this, "About Edd","Evidence Data Display\n\n"
    "Written by Oliver Grimm, IPP, ETH Zurich\n"
    "EMail: oliver.grimm@phys.ethz.ch\n"
    "This version compiled "__DATE__".\n"
    "subversion revision "SVN_REVISION".\n\n"
    "Graphical user interface implemented with Qt.\n"
    "Evidence control system based on DIM (http://dim.web.cern.ch).");
     printf("Revision: $Revision$\n");
}

//---------------------------------------------------------------------
//************************ All functions ****************************
//-------------------------------------------------------------------

// Quit application when clicking close button on window
void GUI::closeEvent(QCloseEvent *) {
  qApp->quit();
}


//---------------------------------------------------------------------
//**************************** Main program ***************************
//---------------------------------------------------------------------

int main(int argc, char *argv[]) {
  QApplication app(argc, argv);
  
  GUI MainWindow;
  MainWindow.setGeometry(100, 100, 800, 500);
  MainWindow.setWindowTitle("Edd - Evidence Data Display");
  MainWindow.show();
  
  return app.exec();
}
