/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  05/2002 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Harald Kornmayer 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MCamDisplay
//
// Camera Display
//
////////////////////////////////////////////////////////////////////////////
#include "MCamDisplay.h"

#include <math.h>
#include <fstream.h>
#include <iostream.h>

#include <TClonesArray.h>
#include <TCanvas.h>
#include <TStyle.h>
#include <TBox.h>
#include <TText.h>
#include <TButton.h>

#include "MHexagon.h"

#include "MGeomPix.h"
#include "MGeomCam.h"

#include "MCerPhotPix.h"
#include "MCerPhotEvt.h"

#define kItemsLegend 50 // see SetPalette(1,0)

ClassImp(MCamDisplay);

// ------------------------------------------------------------------------
//
//  default constructor
//
MCamDisplay::MCamDisplay(MGeomCam *geom)
    : fAutoScale(kTRUE), fMinPhe(-2), fMaxPhe(50), fW(0), fH(0), fDrawingPad(NULL), fIsAllocated(kFALSE)
{
    fGeomCam = (MGeomCam*)geom; // FIXME: Clone doesn't work! (MGeomCam*)geom->Clone();

    //
    //  create the hexagons of the display
    //
    fNumPixels = geom->GetNumPixels();
    fRange     = geom->GetMaxRadius();

    //
    // Construct all hexagons. Use new-operator with placement
    //

    // root 3.02
    //  * base/inc/TObject.h:
    //    register BIT(8) as kNoContextMenu. If an object has this bit set it will
    //    not get an automatic context menu when clicked with the right mouse button.

    fPixels = new TClonesArray("MHexagon", fNumPixels);
    for (UInt_t i=0; i<fNumPixels; i++)
        new ((*fPixels)[i]) MHexagon((*geom)[i]);

    //
    // set the color palette for the TBox elements
    //
#if ROOT_VERSION_CODE < ROOT_VERSION(3,01,06)
    SetPalette(1, 0);
#else
    SetPalette(51, 0);
#endif

    //
    // set up the Legend
    //
    fLegend  = new TClonesArray("TBox",  kItemsLegend);
    fLegText = new TClonesArray("TText", kItemsLegend);

    for (Int_t i = 0; i<kItemsLegend; i++)
    {
        TBox  *newbox = new ((*fLegend)[i])  TBox;
        TText *newtxt = new ((*fLegText)[i]) TText;

        newbox->SetFillColor(fColors[i]);

        newtxt->SetTextSize(0.025);
        newtxt->SetTextAlign(12);
    }
}

// ------------------------------------------------------------------------
//
// Destructor. Deletes TClonesArrays for hexagons and legend elements.
//
MCamDisplay::~MCamDisplay()
{
    fPixels->Delete();
    fLegend->Delete();
    fLegText->Delete();

    delete fPixels;
    delete fLegend;
    delete fLegText;

    // delete fGeomCam;

    //    if (fIsAllocated)
    //        delete fDrawingPad;
}

inline void MCamDisplay::SetPixColor(const MCerPhotPix &pix, const Int_t i)
{
    //
    // Fixme: Divide pnum by the (real) area of the pixel
    //
    const Float_t ratio = (*fGeomCam)[0].GetA()/(*fGeomCam)[i].GetA();
    const Float_t pnum  = ratio*pix.GetNumPhotons();

    (*this)[pix.GetPixId()].SetFillColor(GetColor(pnum));
}

// ------------------------------------------------------------------------
//
// This is called at any time the canvas should get repainted.
// Here we maintain an aspect ratio of 5/4=1.15. This makes sure,
// that the camera image doesn't get distorted by resizing the canvas.
//
void MCamDisplay::Paint(Option_t *opt)
{
    const UInt_t w = (UInt_t)(gPad->GetWw()*gPad->GetAbsWNDC());
    const UInt_t h = (UInt_t)(gPad->GetWh()*gPad->GetAbsHNDC());

    //
    // Check for a change in width or height, and make sure, that the
    // first call also sets the range
    //
    if (w*fH == h*fW && fW && fH)
        return;

    //
    // Calculate aspect ratio (5/4=1.25 recommended)
    //
    const Double_t ratio = (Double_t)w/h;

    Float_t x;
    Float_t y;

    if (ratio>1.25)
    {
        x = (ratio*2-1)*fRange; 
        y = fRange;
    }
    else
    {
        x = fRange*1.5;
        y = fRange*1.25/ratio;
    }

    fH = h;
    fW = w;

    //
    // Set new range
    //
    gPad->Range(-fRange, -y, x, y);
}

// ------------------------------------------------------------------------
//
//  With this function you can change the color palette. For more
// information see TStyle::SetPalette. Only palettes with 50 colors
// are allowed.
// In addition you can use SetPalette(52, 0) to create an inverse
// deep blue sea palette
//
void MCamDisplay::SetPalette(Int_t ncolors, Int_t *colors)
{
    //
    // If not enough colors are specified skip this.
    //
    if (ncolors>1 && ncolors<50)
    {
        cout << "MCamDisplay::SetPalette: Only default palettes with 50 colors are allowed... ignored." << endl;
        return;
    }

    //
    // If ncolors==52 create a reversed deep blue sea palette
    //
    if (ncolors==52)
    {
        gStyle->SetPalette(51, NULL);
        Int_t c[50];
        for (int i=0; i<50; i++)
            c[49-i] = gStyle->GetColorPalette(i);
        gStyle->SetPalette(50, c);
    }
    else
        gStyle->SetPalette(ncolors, colors);

    if (fDrawingPad)
    {
        //
        // Set the colors of the legend
        //
        for (int i=0; i<kItemsLegend; i++)
        {
            Int_t col = GetBox(i)->GetFillColor();

            //
            // Make sure, that the legend is already colored
            //
            if (col==10 || col==22)
                continue;
            GetBox(i)->SetFillColor(gStyle->GetColorPalette(i));
        }

        //
        // Change the colors of the pixels
        //
        for (unsigned int i=0; i<fNumPixels; i++)
        {
            //
            // Get the old color index and check whether it is
            // background or transparent
            //
            Int_t col = (*this)[i].GetFillColor();
            if (col==10 || col==22)
                continue;

            //
            // Search for the color index (level) in the color table
            //
            int idx;
            for (idx=0; idx<kItemsLegend; idx++)
                if (col==fColors[idx])
                    break;
            //
            // Should not happen
            //
            if (idx==kItemsLegend)
            {
                cout << "MCamDisplay::SetPalette: Strange... FIXME!" << endl;
                continue;
            }

            //
            // Convert the old color index (level) into the new one
            //
            (*this)[i].SetFillColor(gStyle->GetColorPalette(idx));
        }

        //
        // Update the pad on the screen
        //
        fDrawingPad->Modified();
        fDrawingPad->Update();
    }

    //
    // Store the color palette used for a leter reverse lookup
    //
    for (int i=0; i<kItemsLegend; i++)
        fColors[i] = gStyle->GetColorPalette(i);
}

// ------------------------------------------------------------------------
//
// Call this function to draw the camera layout into your canvas.
// Setup a drawing canvas. Add this object and all child objects
// (hexagons, etc) to the current pad. If no pad exists a new one is
// created.
//
void MCamDisplay::Draw(Option_t *option)
{
    // root 3.02:
    // gPad->SetFixedAspectRatio()

    if (fDrawingPad)
        return;

    //
    // if no canvas is yet existing to draw into, create a new one
    //
    if (!gPad)
    {
        fDrawingPad = new TCanvas("CamDisplay", "Magic Camera Display", 0, 0, 750, 600);
        fIsAllocated = kTRUE;
    }
    else
    {
        fDrawingPad = gPad;
        fIsAllocated = kFALSE;
    }

    //
    // Setup the correct environment
    //
    fDrawingPad->SetBorderMode(0);
    fDrawingPad->SetFillColor(22);

    //
    // Set the initial coordinate range
    //
    Paint();

    //
    // Create and draw the buttons which allows changing the
    // color palette of the display
    //
    TButton *but;
    char txt[100];
    sprintf(txt, "((MCamDisplay*)%p)->SetPalette(1,0);", this);
    but = new TButton("Pretty", txt, 0.01, 0.95, 0.15, 0.99);
    but->Draw();
    sprintf(txt, "((MCamDisplay*)%p)->SetPalette(51,0);", this);
    but = new TButton("Deap Sea", txt, 0.16, 0.95, 0.30, 0.99);
    but->Draw();
    sprintf(txt, "((MCamDisplay*)%p)->SetPalette(52,0);", this);
    but = new TButton("Blue Inv", txt, 0.31, 0.95, 0.45, 0.99);
    but->Draw();

    //
    // Draw all pixels of the camera
    //  (means apend all pixelobjects to the current pad)
    //
    for (UInt_t i=0; i<fNumPixels; i++)
    {
#if ROOT_VERSION_CODE > ROOT_VERSION(3,01,06)
        (*this)[i].SetBit(kNoContextMenu|kCannotPick);
#endif
        (*this)[i].SetFillColor(22);
        (*this)[i].Draw();
    }

    //
    // initialize and draw legend
    //
    const Float_t H = 0.9*fRange;
    const Float_t h = 2./kItemsLegend;

    const Float_t w = fRange/sqrt(fNumPixels);

    for (Int_t i=0; i<kItemsLegend; i++)
    {
        TBox *box = GetBox(i);
        box->SetX1(fRange);
        box->SetX2(fRange+w);
        box->SetY1(H*( i   *h - 1.));
        box->SetY2(H*((i+1)*h - 1.));
        box->SetFillColor(22);
#if ROOT_VERSION_CODE > ROOT_VERSION(3,01,06)
        box->SetBit(kNoContextMenu|kCannotPick);
#endif
        box->Draw();

        TText *txt = GetText(i);
        txt->SetX(fRange+1.5*w);
        txt->SetY(H*((i+0.5)*h - 1.));
#if ROOT_VERSION_CODE > ROOT_VERSION(3,01,06)
        txt->SetBit(kNoContextMenu|kCannotPick);
#endif
        txt->Draw();
    }

    //
    // Append this object, so that the aspect ratio is maintained
    // (Paint-function is called)
    // Add it here so that root will have it first in the internal list:
    // This means, that root 'sees' the whole camera instead of all the
    // single hexagons.
    //
    AppendPad(option);

    //fDrawingPad->SetEditable(kFALSE);
}

void MCamDisplay::DrawPixelNumbers()
{
    if (!fDrawingPad)
        Draw();

    fDrawingPad->cd();

    TText txt;
    txt.SetTextFont(122);
    txt.SetTextAlign(22);   // centered/centered

    for (UInt_t i=0; i<fNumPixels; i++)
    {
        TString num;
        num += i;

        const MHexagon &h = (*this)[i];
        TText *nt = txt.DrawText(h.GetX(), h.GetY(), num);
        nt->SetTextSize(0.3*h.GetD()/fGeomCam->GetMaxRadius());
    }
}

// ------------------------------------------------------------------------
//
// Call this function to draw the number of photo electron into the
// camera.
//
void MCamDisplay::DrawPhotNum(const MCerPhotEvt *event)
{
    if (!event)
        return;

    if (!fDrawingPad)
        Draw();

    fDrawingPad->cd();

    for (int i=0; i<kItemsLegend; i++)
        GetBox(i)->SetFillColor(fColors[i]);

    //
    // Reset pixel colors to default value
    //
    Reset();

    //
    //  if the autoscale is true, set the values for the range for
    //  each event
    //
    if (fAutoScale)
    {
        fMinPhe = event->GetNumPhotonsMin(fGeomCam);
        fMaxPhe = event->GetNumPhotonsMax(fGeomCam);

        if (fMaxPhe < 20.)
            fMaxPhe = 20.;

        UpdateLegend();
    }

    //
    //   update the colors in the picture
    //
    const Int_t entries = event->GetNumPixels();

    for (Int_t i=0; i<entries; i++)
    {
        const MCerPhotPix &pix = (*event)[i];

        if (!pix.IsPixelUsed())
            continue;

        SetPixColor(pix, i);
    }

    //
    // Update display physically
    //
    fDrawingPad->Modified();
    fDrawingPad->Update();
}

// ------------------------------------------------------------------------
//
// reset the all pixel colors to a default value
//
void MCamDisplay::Reset()
{
    for (UInt_t i=0; i<fNumPixels; i++)
        (*this)[i].SetFillColor(10);
} 

// ------------------------------------------------------------------------
//
//   Here we calculate the color index for the current value.
//   The color index is defined with the class TStyle and the
//   Color palette inside. We use the command gStyle->SetPalette(1,0)
//   for the display. So we have to convert the value "wert" into
//   a color index that fits the color palette.
//   The range of the color palette is defined by the values fMinPhe
//   and fMaxRange. Between this values we have 50 color index, starting
//   with 0 up to 49.
//
Int_t MCamDisplay::GetColor(Float_t val)
{
    //
    //   first treat the over- and under-flows
    //
    const Int_t maxcolidx = kItemsLegend-1;

    if (val >= fMaxPhe)
        return fColors[maxcolidx];

    if (val <= fMinPhe)
        return fColors[0];

    //
    // calculate the color index
    //
    const Float_t ratio  = (val-fMinPhe) / (fMaxPhe-fMinPhe);
    const Int_t   colidx = (Int_t)(ratio*maxcolidx + .5);

    return fColors[colidx];
}

// ------------------------------------------------------------------------
//
//    change the text on the legend according to the range of the
//    Display
//
void MCamDisplay::UpdateLegend()
{
    char text[10];

    for (Int_t i=0; i<kItemsLegend; i+=3)
    {
        const Float_t val = fMinPhe + (Float_t)i/kItemsLegend * (fMaxPhe-fMinPhe) ;

        sprintf(text, "%5.1f", val);

        TText &txt = *GetText(i);

        txt.SetText(txt.GetX(), txt.GetY(), text);
    }
}

// ------------------------------------------------------------------------
//
// Save primitive as a C++ statement(s) on output stream out
//
void MCamDisplay::SavePrimitive(ofstream &out, Option_t *opt)
{
    if (!gROOT->ClassSaved(TCanvas::Class()))
        fDrawingPad->SavePrimitive(out, opt);

    out << "   " << fDrawingPad->GetName() << "->SetWindowSize(";
    out << fDrawingPad->GetWw() << "," << fDrawingPad->GetWh() << ");" << endl;
}

// ------------------------------------------------------------------------
//
// compute the distance of a point (px,py) to the Camera
// this functions needed for graphical primitives, that
// means without this function you are not able to interact
// with the graphical primitive with the mouse!!!
//
// All calcutations are running in pixel coordinates
//
Int_t MCamDisplay::DistancetoPrimitive(Int_t px, Int_t py)
{
    Int_t dist = 999999;

    for (UInt_t i=0; i<fNumPixels; i++)
    {
        Int_t d = (*fPixels)[i]->DistancetoPrimitive(px, py);

        if (d<dist)
            dist=d;
    }
    return dist==0?0:999999;
}

// ------------------------------------------------------------------------
//
// Execute a mouse event on the camera
//
void MCamDisplay::ExecuteEvent(Int_t event, Int_t px, Int_t py)
{
    cout << "Execute Event Camera " << event << " @ " << px << " " << py << endl;
}
