#include <iostream>
#include <TMath.h>
#include <TF1.h>

#include "rootfilehandler.h"
#include "pixel.h" // class implemented
using namespace std;

/////////////////////////////// PUBLIC ///////////////////////////////////////

//============================= LIFECYCLE ====================================


Pixel::Pixel(
        int         pixelID,
        int         maxPulsorder,
        int         verbosityLevel,
        bool        stats,
        TString     options,
        int         pixelOverlayXaxisLeft,
        int         pixelOverlayXaxisRight,
        float       bSLMean,
        float       gainMean,
        TFile*      filename,
        TFile*      outfilename
        )
{
    mConstructorType        = 1;        //important for deletion 0 delete distribution 1 delete TemplateHistos

    mChid                   = pixelID;
    mMaxPulseOrder          = maxPulsorder;
    mVerbosityLevel         = verbosityLevel;
    mStats                  = stats;
    mOptions                = options;
    mPixelOverlayXaxisLeft  = pixelOverlayXaxisLeft;
    mPixelOverlayXaxisRight = pixelOverlayXaxisRight;
    mBSLMean                = bSLMean;
    mGainMean               = gainMean;
    mRootFile               = filename;
    mOutRootFile            = outfilename;

    hMaxOverlay  = new TH2F*[mMaxPulseOrder];
    hEdgeOverlay = new TH2F*[mMaxPulseOrder];
    hMaxProfile  = new TProfile*[mMaxPulseOrder];
    hEdgeProfile = new TProfile*[mMaxPulseOrder];

    hPixelMax  = new TH1F*[mMaxPulseOrder];
    hPixelMedian = new TH1F*[mMaxPulseOrder];
    hPixelMean  = new TH1F*[mMaxPulseOrder];

    hPixelEdgeMax  = new TH1F*[mMaxPulseOrder];
    hPixelEdgeMedian = new TH1F*[mMaxPulseOrder];
    hPixelEdgeMean  = new TH1F*[mMaxPulseOrder];

    hList           = new TList();

    if (mOptions.Contains("C") )
    {
        hChi2EdgetoMax = new TH1F*[mMaxPulseOrder];
    }

    if (options.Contains("L"))
    {
        LoadPulseHistos( );
    }
    else
    {
       BookPixelHistos();
    }

    if (mOptions.Contains("C"))
    {
        BookDistributionHistos();
    }

    BookTemplateHistos();
    BookEdgeTemplateHistos();
}

Pixel::Pixel(
        int         pixelID,
        int         maxPulsorder,
        int         verbosityLevel,
        bool        stats,
        TString     options,
        int         pixelOverlayXaxisLeft,
        int         pixelOverlayXaxisRight,
        float       bSLMean,
        float       gainMean
        )
{
    mConstructorType        = 0;

    mChid                   = pixelID;
    mMaxPulseOrder          = maxPulsorder;
    mVerbosityLevel         = verbosityLevel;
    mStats                  = stats;        ///TODO: HANDOVER THE VALUE
    mOptions                = options;
    mPixelOverlayXaxisLeft  = pixelOverlayXaxisLeft;
    mPixelOverlayXaxisRight = pixelOverlayXaxisRight;
    mBSLMean                = bSLMean;
    mGainMean               = gainMean;

    hMaxOverlay  = new TH2F*[mMaxPulseOrder];
    hEdgeOverlay = new TH2F*[mMaxPulseOrder];
    hMaxProfile  = new TProfile*[mMaxPulseOrder];
    hEdgeProfile = new TProfile*[mMaxPulseOrder];

    if (mOptions.Contains("S") )
    {
        hSlopeRisingEdge = new TH1F*[mMaxPulseOrder];
    }

    if (mOptions.Contains("R") )
    {
        hRisingEdgeToMax = new TH1I*[mMaxPulseOrder];
    }

    if (mOptions.Contains("M") )
    {
        hPosOfMax = new TH1I*[mMaxPulseOrder];
    }

    hList           = new TList();

    BookPixelHistos();
    BookDistributionHistos();
}

Pixel::~Pixel()
{
    if (mVerbosityLevel > 1)
    {
        cout << endl << "...delete histograms of pixel " << mChid ;
        cout << endl << "...mConstructorType = " << mConstructorType ;
    }

    if (mConstructorType == 0)
    {
        DeleteDistributionHistos();
    }

    if (mConstructorType == 1)
    {
        DeleteTemplateHistos();
        DeleteEdgeTemplateHistos();
    }

    DeletePixelHistos();
    delete hList;
    hList = NULL;
    if (mVerbosityLevel > 1)
    {
        cout << endl << "...histograms of pixel " << mChid ;
        cout << "...deleted " << endl ;
    }
}// ~Pixel


//============================= OPERATORS ====================================

//XX&
//XX::operator=(const XX&);
//{
//    return *this;

//}// =

//============================= OPERATIONS ===================================
void
Pixel::DrawOverlayHistograms(
        TCanvas**    pixelCanvas,
        int*        histoFrameNr
        )
{
    if (mVerbosityLevel > 2) cout << endl << "...drawing pulse histograms" ;
    for (int pulse_order = 0; pulse_order < mMaxPulseOrder; pulse_order++)
    {
        pixelCanvas[pulse_order]->cd( histoFrameNr[0] );
        hMaxOverlay[pulse_order]->Draw("COLZ");
        pixelCanvas[pulse_order]->cd( histoFrameNr[2] );
        hMaxProfile[pulse_order]->Draw();

        pixelCanvas[pulse_order]->cd( histoFrameNr[1] );
        hEdgeOverlay[pulse_order]->Draw("COLZ");
        pixelCanvas[pulse_order]->cd( histoFrameNr[3] );
        hEdgeProfile[pulse_order]->Draw();
    }
}
// end of DrawOverlayHistograms
//----------------------------------------------------------------------------

void
Pixel::DrawDistributionHistograms(
        TCanvas**    pixelCanvas,
        int*        histoFrameNr
        )
{
    if (mVerbosityLevel > 2) cout << endl << "...drawing distribution histograms" ;
    bool nothing_to_fill = true;
    for (int pulse_order = 0; pulse_order < mMaxPulseOrder; pulse_order++)
    {
        if (mOptions.Contains("S") )
        {
        pixelCanvas[pulse_order]->cd( histoFrameNr[0] );
        hSlopeRisingEdge[pulse_order]->Draw();
        nothing_to_fill = false;
        }

        if (mOptions.Contains("R") )
        {
        pixelCanvas[pulse_order]->cd( histoFrameNr[1] );
        hRisingEdgeToMax[pulse_order]->Draw();
        nothing_to_fill = false;
        }

        if (mOptions.Contains("M") )
        {
        pixelCanvas[pulse_order]->cd( histoFrameNr[2] );
        hPosOfMax[pulse_order]->Draw();
        nothing_to_fill = false;
        }

    }
    if (nothing_to_fill)
    {
        cout << endl << "there were NO DISTRIBUTION HISTOGRAMS to fill" << endl;
    }
}
// end of DrawDistributionHistograms
//----------------------------------------------------------------------------

void
Pixel::DrawTemplateHistograms(
        TCanvas**    pixelCanvas,
        int*        histoFrameNr
        )
{
    if (mVerbosityLevel > 2) cout << "...drawing Template histograms" << endl ;
    for (int pulse_order = 0; pulse_order < mMaxPulseOrder; pulse_order++)
    {
        pixelCanvas[pulse_order]->cd( histoFrameNr[0] );
        hMaxOverlay[pulse_order]->Draw("COLZ");

        pixelCanvas[pulse_order]->cd( histoFrameNr[1] );
        hPixelMax[pulse_order]->Draw();

        pixelCanvas[pulse_order]->cd( histoFrameNr[2] );
        hPixelMedian[pulse_order]->Draw();

        pixelCanvas[pulse_order]->cd( histoFrameNr[3] );
        hPixelMean[pulse_order]->Draw();
//        hPixelMax[pulse_order]->Draw("SAME");
//        hPixelMedian[pulse_order]->Draw("SAME");

    }
}
// end of DrawTemplateHistograms
//----------------------------------------------------------------------------

void
Pixel::DrawEdgeTemplateHistograms(
        TCanvas**    pixelCanvas,
        int*        histoFrameNr
        )
{
    if (mVerbosityLevel > 2) cout << endl << "...drawing Template histograms" ;
    for (int pulse_order = 0; pulse_order < mMaxPulseOrder; pulse_order++)
    {
        pixelCanvas[pulse_order]->cd( histoFrameNr[4] );
        hEdgeOverlay[pulse_order]->Draw("COLZ");

        pixelCanvas[pulse_order]->cd( histoFrameNr[5] );
        hPixelEdgeMax[pulse_order]->Draw();

        pixelCanvas[pulse_order]->cd( histoFrameNr[6] );
        hPixelEdgeMedian[pulse_order]->Draw();

        pixelCanvas[pulse_order]->cd( histoFrameNr[7] );
        hPixelEdgeMean[pulse_order]->Draw();
//        hPixelEdgeMax[pulse_order]->Draw("SAME");
//        hPixelEdgeMedian[pulse_order]->Draw("SAME");

    }
}
// end of DrawTemplateHistograms
//----------------------------------------------------------------------------




void
Pixel::SavePixelHistograms(
        TString     outRootFileName,
        bool        saveResults
        )
{
    if (mVerbosityLevel > 2) cout << endl << "Saving histograms of Pixel# " << mChid;
    if (!saveResults) return;
    SaveHistograms(
            outRootFileName,
            CreateSubDirName( mChid ),
            hList,
            saveResults,
            mVerbosityLevel
            );
}
// end of SavePixelHistograms
//----------------------------------------------------------------------------

TString
Pixel::HistoName(
        TString     histoname,
        int         order
        )
{
    histoname +=   mChid;
    histoname +=   "_";
    histoname +=   order;
    return histoname;
}

TString
Pixel::ChooseCycleNumber(
        TString     histoname,
        int         cycleNumber
        )
{
    histoname +=   ";";
    histoname +=   cycleNumber;
    return histoname;
}

TString
Pixel::HistoTitle(
        TString     histo_title,
        int         order
        )
{
    histo_title +=   " [Pixel_Order] ";
    histo_title +=   mChid;
    histo_title +=   "_";
    histo_title +=   order;
    return histo_title;
}

TString
Pixel::CsvFileName(
        TString     path,
        TString     overlayMethod,
        int         order
        )
{
    path += "CSV/";
    path += overlayMethod;
    path += "OverlayTemplate";
    path += "_";
    path +=   mChid;
    path +=   "_";
    path +=   order;
    path +=   ".csv";
    return path;
}


//============================= ACESS      ===================================
//============================= INQUIRY    ===================================
/////////////////////////////// PROTECTED  ///////////////////////////////////

/////////////////////////////// PRIVATE    ///////////////////////////////////

void
Pixel::BookPixelHistos()
{
    if (mVerbosityLevel > 2) cout << endl << "...book pixel histograms" << endl;
    for (int order = 0; order < mMaxPulseOrder; order++)
    {
        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hMaxOverlay", order) << endl;
        hMaxOverlay[order]=new TH2F(
                    HistoName("hMaxOverlay", order),
                    HistoTitle("Overlay at peak maximum of detected pulses of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5 ,
                    512,
                    -55.5,
                    200.5
                    );

        hMaxOverlay[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hMaxOverlay[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hMaxOverlay[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hMaxOverlay[order]->SetStats(mStats);
        hList->Add( hMaxOverlay[order] );

//------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hEdgeOverlay", order) << endl;
        hEdgeOverlay[order] = new TH2F(
                    HistoName("hEdgeOverlay", order),
                    HistoTitle("Overlay at rising edge of detected pulses of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5 ,
                    512,
                    -55.5,
                    200.5
                    );

        hEdgeOverlay[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hEdgeOverlay[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hEdgeOverlay[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        hEdgeOverlay[order]->SetStats(mStats);
        hList->Add( hEdgeOverlay[order] );

    //------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hMaxProfile", order) << endl;
        hMaxProfile[order] = new TProfile(
                    HistoName("hMaxProfile", order),
                    HistoTitle("Mean value of each slice in overlay plot (Tprofile)", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,//nbinsx
                    (-1*mPixelOverlayXaxisLeft)-0.5,                 //xlow
                    mPixelOverlayXaxisRight-0.5 ,                    //xup
    //                512,                                            //nbinsy
                    -55.5,                                          //ylow
                    300.5,                                          //yup
                    "s");                                           //option
        hMaxProfile[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hMaxProfile[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hMaxProfile[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hMaxProfile[order]->SetStats(mStats);
        hList->Add( hMaxProfile[order] );


    //------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hEdgeProfile", order) << endl;
        hEdgeProfile[order] = new TProfile(
                    HistoName("hEdgeProfile", order),
                    HistoTitle("Mean value of each slice in overlay plot (Tprofile)", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,//nbinsx
                    (-1*mPixelOverlayXaxisLeft)-0.5,                 //xlow
                    mPixelOverlayXaxisRight-0.5 ,                    //xup
    //                512,                                            //nbinsy
                    -55.5,                                          //ylow
                    300.5,                                          //yup
                    "s");                                           //option
        hEdgeProfile[order]->SetLineColor(kRed);
        hEdgeProfile[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hEdgeProfile[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hEdgeProfile[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hEdgeProfile[order]->SetStats(mStats);
        hList->Add( hEdgeProfile[order] );

    }
    if (mVerbosityLevel > 2) cout << "...done" << endl;
}
//end of BookPixelHistos
//----------------------------------------------------------------------------

void
Pixel::BookDistributionHistos( )
{
    if (!mOptions.IsNull() )
    {
        int x_min = 0;
        int x_max = 0;

        for (int order =0; order < mMaxPulseOrder; order++)
        {
            if (mVerbosityLevel > 2) cout << endl
                                          << "...book distribution histograms"
                                          << endl;

            if (mOptions.Contains("S"))
            {
                if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hSlopeRisingEdge", order) << endl;
                hSlopeRisingEdge[order] = new TH1F (
                            HistoName("hSlopeRisingEdge", order),
                            HistoTitle("Distribution of rising edge's slope", order),
                            600,
                            -10.1,
                            10.1
                            );
                hSlopeRisingEdge[order]->SetAxisRange(
                                    -1,
                                    7,
                                    "X");
                hSlopeRisingEdge[order]->GetXaxis()->SetTitle( "Slope Amplitude/time [mV/timeslices]" );
                hSlopeRisingEdge[order]->GetYaxis()->SetTitle( "counts" );
                hSlopeRisingEdge[order]->SetStats(mStats);
                hList->Add( hSlopeRisingEdge[order] );
            }

            if (mOptions.Contains("R"))
            {
                x_min = -15;
                x_max = 35;

                if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hRisingEdgeToMax", order) << endl;
                hRisingEdgeToMax[order] = new TH1I (
                            HistoName("hRisingEdgeToMax", order),
                            HistoTitle("Distribution of distance between rising edge and pulse's maximum", order),
                            x_max -x_min,
                            x_min,
                            x_max
                            );
                hSlopeRisingEdge[order]->SetAxisRange(
                                    -5,
                                    25,
                                    "X");
                hRisingEdgeToMax[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
                hRisingEdgeToMax[order]->GetYaxis()->SetTitle( "counts" );
                hRisingEdgeToMax[order]->SetStats(mStats);
                hList->Add( hRisingEdgeToMax[order] );
            }

            if (mOptions.Contains("M"))
            {
                x_min = 10;
                x_max = 290;
                if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPosOfMax", order) << endl;
                hPosOfMax[order] = new TH1I (
                            HistoName("hPosOfMax", order),
                            HistoTitle("Distribution of pulse's maximum's positon", order),
                            x_max - x_min,
                            x_min,
                            x_max
                            );
                hPosOfMax[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
                hPosOfMax[order]->GetYaxis()->SetTitle( "counts" );
                hRisingEdgeToMax[order]->SetStats(mStats);
                hList->Add( hPosOfMax[order] );
            }

            if (mOptions.Contains("C"))
            {
                if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hSlopeRisingEdge", order) << endl;
                hChi2EdgetoMax[order] = new TH1F (
                            HistoName("hChi2EdgetoMax", order),
                            HistoTitle("Distribution of CHI2 comparison of Edge and Max", order),
                            600,
                            -299,
                            300
                            );
                hChi2EdgetoMax[order]->SetAxisRange(
                                    -1,
                                    7,
                                    "X");
                hChi2EdgetoMax[order]->GetXaxis()->SetTitle( "p-Value" );
                hChi2EdgetoMax[order]->GetYaxis()->SetTitle( "counts" );
                hChi2EdgetoMax[order]->SetStats(mStats);
//                hList->Add( hChi2EdgetoMax[order] );
            }

            if (mVerbosityLevel > 2) cout << "...done" << endl;
        }
    }
}
//end of BookDistributionHistos
//----------------------------------------------------------------------------

void
Pixel::BookTemplateHistos()
{
    if (mVerbosityLevel > 2) cout << endl << "...book pixel histograms" << endl;
    for (int order = 0; order < mMaxPulseOrder; order++)
    {
        if (mVerbosityLevel > 3)
        {
            cout << "\t...booking " << HistoName("hPixelMax", order) << endl;
        }

        hPixelMax[order]=new TH1F(
                    HistoName("hPixelMax", order),
                    HistoTitle("Maximum value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5
                    );

        hPixelMax[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelMax[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelMax[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hPixelMax[order]->SetStats(mStats);
        hList->Add( hPixelMax[order] );

//------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPixelMean", order) << endl;
        hPixelMean[order] = new TH1F(
                    HistoName("hPixelMean", order),
                    HistoTitle("Mean value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5
                    );
        hPixelMean[order]->SetLineColor(kBlue);
        hPixelMean[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelMean[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelMean[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        hPixelMean[order]->SetStats(mStats);
        hList->Add( hPixelMean[order] );

 //------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPixelMedian", order) << endl;
        hPixelMedian[order] = new TH1F(
                    HistoName("hPixelMedian", order),
                    HistoTitle("Median value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,//nbinsx
                    (-1*mPixelOverlayXaxisLeft)-0.5,                 //xlow
                    mPixelOverlayXaxisRight-0.5                     //xup
                    );
        hPixelMedian[order]->SetLineColor(kRed);
        hPixelMedian[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelMedian[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelMedian[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hPixelMedian[order]->SetStats(mStats);
        hList->Add( hPixelMedian[order] );

    }
    if (mVerbosityLevel > 2) cout << "...done" << endl;
}
//end of BookTemplateHistos
//----------------------------------------------------------------------------

void
Pixel::BookEdgeTemplateHistos()
{
    if (mVerbosityLevel > 2) cout << endl << "...book pixel histograms" << endl;
    for (int order = 0; order < mMaxPulseOrder; order++)
    {
        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPixelEdgeMax", order) << endl;
        hPixelEdgeMax[order]=new TH1F(
                    HistoName("hPixelEdgeMax", order),
                    HistoTitle("Maximum value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5
                    );

        hPixelEdgeMax[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelEdgeMax[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelEdgeMax[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hPixelEdgeMax[order]->SetStats(mStats);
        hList->Add( hPixelEdgeMax[order] );

//------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPixelEdgeMean", order) << endl;
        hPixelEdgeMean[order] = new TH1F(
                    HistoName("hPixelEdgeMean", order),
                    HistoTitle("Mean value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                    (-1*mPixelOverlayXaxisLeft)-0.5,
                    mPixelOverlayXaxisRight-0.5
                    );
        hPixelEdgeMean[order]->SetLineColor(kBlue);
        hPixelEdgeMean[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelEdgeMean[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelEdgeMean[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        hPixelEdgeMean[order]->SetStats(mStats);
        hList->Add( hPixelEdgeMean[order] );

 //------------------------------------------------------------------------

        if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName("hPixelEdgeMedian", order) << endl;
        hPixelEdgeMedian[order] = new TH1F(
                    HistoName("hPixelEdgeMedian", order),
                    HistoTitle("Median value of each slice in overlay plot of", order),
                    mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,//nbinsx
                    (-1*mPixelOverlayXaxisLeft)-0.5,                 //xlow
                    mPixelOverlayXaxisRight-0.5                     //xup
                    );
        hPixelEdgeMedian[order]->SetLineColor(kRed);
        hPixelEdgeMedian[order]->SetAxisRange(
                    mBSLMean - 5,
                    (mGainMean*(order+1)) + 10,
                    "Y");
        hPixelEdgeMedian[order]->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
        hPixelEdgeMedian[order]->GetYaxis()->SetTitle( "Amplitude [mV]" );
        //hMaxProfile->SetBit(TH2F::kCanRebin);
        hPixelEdgeMedian[order]->SetStats(mStats);
        hList->Add( hPixelEdgeMedian[order] );

    }
    if (mVerbosityLevel > 2) cout << "...done" << endl;
}
//end of BookEdgeTemplateHistos
//----------------------------------------------------------------------------

void
Pixel::LoadPulseHistos()
{
    mRootFile->cd();

    if (mVerbosityLevel > 2)
    {
        if (mVerbosityLevel > 3)
        {
            mRootFile->ls();
        }
        cout << "loading current File: " ;
        gFile->pwd();
    }

    mRootFile->cd( CreateSubDirName( mChid ) ); //Go to pixel's subdirectory
    mDirectory = gDirectory;

    if (mVerbosityLevel > 2)
    {
        if (mVerbosityLevel > 3)
        {
            gDirectory->ls();
        }
        cout << "Current Directory: " ;
        gDirectory->pwd();
    }

//    mPixelOverlayXaxisRight   =70;   ///TODO: get it from the root file
//    mPixelOverlayXaxisLeft    =230;  ///TODO: get it from the root file
//    mBSLMean                  =-1;   ///TODO: get it from the root file
//    mGainMean                 =9;    ///TODO: get it from the root file
//    mOptions                  ="SRM"; ///TODO: get it from the root file

    if (mVerbosityLevel > 2) cout << endl << "...load pixel histograms" << endl;
    for (int order = 0; order < mMaxPulseOrder; order++)
    {

        if (mVerbosityLevel > 3) cout << "\t...loading " << ChooseCycleNumber( HistoName("hMaxOverlay", order), 1) << endl;
        hMaxOverlay[order]  = (TH2F*)mDirectory->Get( ChooseCycleNumber( HistoName("hMaxOverlay", order), 1) );

        if (mVerbosityLevel > 3) cout << "\t...loading " << HistoName("hEdgeOverlay", order) << endl;
        hEdgeOverlay[order] = (TH2F*)mDirectory->Get( ChooseCycleNumber( HistoName("hEdgeOverlay", order), 1) );

        if (mVerbosityLevel > 3) cout << "\t...loading " << HistoName("hMaxProfile", order) << endl;
        hMaxProfile[order]  = (TProfile*)mDirectory->Get( ChooseCycleNumber( HistoName("hMaxProfile", order), 1) );

        if (mVerbosityLevel > 3) cout << "\t...loading " << HistoName("hEdgeProfile", order) << endl;
        hEdgeProfile[order] = (TProfile*)mDirectory->Get( ChooseCycleNumber( HistoName("hEdgeProfile", order), 1) );
    }

}
// end of LoadPulseHistos
//----------------------------------------------------------------------------

void
Pixel::DeletePixelHistos()
{
    if (mVerbosityLevel > 2)
    {
        cout << endl
             << endl
             << "\t...delete current overlay histograms of Pixel# " << mChid;
    }
    for (int order = 0;
         order < mMaxPulseOrder;
         order ++)
    {
        if (mVerbosityLevel > 3)
        cout << endl << "\t\t...deleting hMaxOverlay"
                                      << mChid << "_" << order;

        delete hMaxOverlay[order];
        hMaxOverlay[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hEdgeOverlay"
                                      << mChid << "_" << order ;
        delete hEdgeOverlay[order];
        hEdgeOverlay[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hMaxProfile"
                                      << mChid << "_" << order ;
        delete hMaxProfile[order];
        hMaxProfile[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hMaxProfile2"
                                      << mChid << "_" << order ;
        delete hEdgeProfile[order];
        hEdgeProfile[order] = NULL;
    }
    if (mVerbosityLevel > 3) cout << endl << "\t...deleting histogram Arrays";

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hMaxOverlay";
    delete[] hMaxOverlay;
    hMaxOverlay = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hEdgeOverlay";
    delete[] hEdgeOverlay;
    hEdgeOverlay = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hMaxProfile";
    delete[] hMaxProfile;
    hMaxProfile = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hEdgeProfile";
    delete[] hEdgeProfile;
    hEdgeProfile = NULL;
}
// end of DeletePixelHistos
//----------------------------------------------------------------------------

void
Pixel::DeleteDistributionHistos()
{
    if (mVerbosityLevel > 2)
    {
        cout << endl
             << "\t...delete current distribution histograms" ;
    }

    for (int order = 0;
         order < mMaxPulseOrder;
         order ++)
    {
        if (mOptions.Contains("S"))
        {
            if (mVerbosityLevel > 3) cout << endl
                                          << "\t\t...deleting hSlopeRisingEdge"
                                          << mChid << "_" << order ;
            delete hSlopeRisingEdge[order];
        }

        if (mOptions.Contains("R"))
        {
            if (mVerbosityLevel > 3) cout << endl
                                          << "\t\t...deleting hRisingEdgeToMax"
                                          << mChid << "_" << order ;
            delete hRisingEdgeToMax[order];
        }

        if (mOptions.Contains("M"))
        {
            if (mVerbosityLevel > 3) cout << endl
                                          << "\t\t...deleting hPosOfMax"
                                          << mChid << "_" << order ;

            delete hPosOfMax[order];
        }

    }
    if (mVerbosityLevel > 3) cout << endl << "\t...deleting histogram Arrays";

    if (mOptions.Contains("S"))
    {
        if (mVerbosityLevel > 3)
            cout << endl << "\t\t...deleting hSlopeRisingEdge";
        delete[] hSlopeRisingEdge;
        hSlopeRisingEdge = NULL;
    }

    if (mOptions.Contains("R"))
    {
        if (mVerbosityLevel > 3)
            cout << endl << "\t\t...deleting hRisingEdgeToMax";
        delete[] hRisingEdgeToMax;
        hRisingEdgeToMax = NULL;
    }

    if (mOptions.Contains("M"))
    {
        if (mVerbosityLevel > 3)
            cout << endl << "\t\t...deleting hPosOfMax";
        delete[] hPosOfMax;
        hPosOfMax = NULL;
    }
}
// end of DeletePixelHistos
//----------------------------------------------------------------------------

void
Pixel::DeleteTemplateHistos()
{
    if (mVerbosityLevel > 2)
    {
        cout << endl
             << "\t...delete current template histograms of Pixel# " << mChid;
    }
    for (int order = 0;
         order < mMaxPulseOrder;
         order ++)
    {
        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelMean"
                                      << mChid << "_" << order ;
        delete hPixelMean[order];
        hPixelMean[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelMedian"
                                      << mChid << "_" << order ;
        delete hPixelMedian[order];
        hPixelMedian[order] = NULL;

        if (mVerbosityLevel > 3)
                                 cout << endl << "\t\t...deleting hPixelMax"
                                      << mChid << "_" << order;
        delete hPixelMax[order];
        hPixelMax[order] = NULL;

    }
    if (mVerbosityLevel > 3) cout << endl << "\t...deleting histogram Arrays";

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelMax";
    delete[] hPixelMax;
    hPixelMax = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelMedian";
    delete[] hPixelMedian;
    hPixelMedian = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelMean";
    delete[] hPixelMean;
    hPixelMean = NULL;
}
// end of DeleteTemplateHistos
//----------------------------------------------------------------------------

void
Pixel::DeleteEdgeTemplateHistos()
{
    if (mVerbosityLevel > 2)
    {
        cout << endl
             << "\t...delete current Edge template histograms of Pixel# " << mChid;
    }
    for (int order = 0;
         order < mMaxPulseOrder;
         order ++)
    {
        if (mVerbosityLevel > 3)
        cout << endl << "\t\t...deleting hPixelEdgeMax"
                                      << mChid << "_" << order;

        delete hPixelEdgeMax[order];
        hPixelEdgeMax[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelEdgeMedian"
                                      << mChid << "_" << order ;
        delete hPixelEdgeMedian[order];
        hPixelEdgeMedian[order] = NULL;

        if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelEdgeMean"
                                      << mChid << "_" << order ;
        delete hPixelEdgeMean[order];
        hPixelEdgeMean[order] = NULL;

    }
    if (mVerbosityLevel > 3) cout << endl << "\t...deleting histogram Arrays";

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelEdgeMax";
    delete[] hPixelEdgeMax;
    hPixelEdgeMax = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelEdgeMedian";
    delete[] hPixelEdgeMedian;
    hPixelEdgeMedian = NULL;

    if (mVerbosityLevel > 3) cout << endl << "\t\t...deleting hPixelEdgeMean";
    delete[] hPixelEdgeMean;
    hPixelEdgeMean = NULL;
}
// end of DeleteTemplateHistos
//----------------------------------------------------------------------------

void
Pixel::MakeTH1Pretty(
        TH1*                histo,
        TString             histName,
        TString             histTitle,
        int                 order
        )
{
    if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName(histName, order) << endl;

    histo->SetNameTitle(
                HistoName(histName, order),
                HistoTitle(histTitle, order)
                );

    histo->SetBins(
                mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                (-1*mPixelOverlayXaxisLeft)-0.5,
                mPixelOverlayXaxisRight-0.5
                );

    histo->SetAxisRange(
                        mBSLMean - 5,
                        (mGainMean*(order+1)) + 10,
                        "Y");

    histo->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
    histo->GetYaxis()->SetTitle( "Amplitude [mV]" );
    //histo->SetBit(TH2F::kCanRebin);
    histo->SetStats(mStats);
}
// end of MakeTH2Pretty
//----------------------------------------------------------------------------


void
Pixel::MakeTH2Pretty(
        TH2*                histo,
        TString             histName,
        TString             histTitle,
        int                 order
        )
{
    if (mVerbosityLevel > 3) cout << "\t...designing " << HistoName(histName, order) << endl;

    histo->SetNameTitle(
                HistoName(histName, order),
                HistoTitle(histTitle, order)
                );

    histo->SetBins(
                mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                (-1*mPixelOverlayXaxisLeft)-0.5,
                mPixelOverlayXaxisRight-0.5 ,
                512,
                -55.5,
                200.5
                );

    histo->SetAxisRange(
                        mBSLMean - 5,
                        (mGainMean*(order+1)) + 10,
                        "Y");

    histo->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
    histo->GetYaxis()->SetTitle( "Amplitude [mV]" );
    //histo->SetBit(TH2F::kCanRebin);
    histo->SetStats(mStats);
}
// end of MakeTH2Pretty
//----------------------------------------------------------------------------

void
Pixel::MakeTProfilePretty(
        TProfile*           histo,
        TString             histName,
        TString             histTitle,
        int                 order
        )
{
    if (mVerbosityLevel > 3) cout << "\t...booking " << HistoName(histName, order) << endl;

    histo->SetNameTitle(
                HistoName(histName, order),
                HistoTitle(histTitle, order)
                );

    histo->SetBins(
                mPixelOverlayXaxisLeft + mPixelOverlayXaxisRight ,
                (-1*mPixelOverlayXaxisLeft)-0.5,
                mPixelOverlayXaxisRight-0.5
                );
    histo->SetOption( "s" );

    histo->SetAxisRange(
                        mBSLMean - 5,
                        (mGainMean*(order+1)) + 10,
                        "Y");

    histo->GetXaxis()->SetTitle( "Timeslices [a.u.]" );
    histo->GetYaxis()->SetTitle( "Amplitude [mV]" );
    //histo->SetBit(TH2F::kCanRebin);
    histo->SetStats(mStats);
}
// end of MakeTProfilePretty
//----------------------------------------------------------------------------


void
Pixel::ShiftHistoInY(
        TH1*                histo,
        float               shift
        )
{
    int min_bin = histo->GetXaxis()->GetFirst();
    int max_bin = histo->GetXaxis()->GetLast();

    for (int bin = min_bin; bin <= max_bin; bin++)
    {
        histo->AddBinContent( bin, shift);
    }

}


