
/*
  Arduino Ethernet for FACT Interlock
  
  Written by Dominik Neise
  June 2012
  TU Dortmund
  
  The FACT Camera as well as the bias voltage supply for its G-APDs
  may only be switched on, if the dedicated cooling systems are running.
  This arduino is connected via optocouplers to some status LEDs, 
  and other coolings system output signals, which show 
  the status of both cooling systems:
  * water cooling pump
    - flow meter 
    - level meter
  * bias crate fans
    - rotation frequency
  A simple WebServer is implemented for showing the 
  status and access the actual switches.

  In order to switch on the camera one needs to switch on 
  the water pump first. In case this pump is running, the 
  Camera power suplly (51VDC Agilent) gets an enable signal
  and will power the camera.
  In order to enable the bias voltage supply, one needs to swtich on 
  the Bias Crate Fans.  
  
  Circuit:
  IL_ON : Interlock on : digital input D1
  FL_OK : level ok     : digital input D2
  FC_OK : current ok   : digital input D3
  FP_EN : pump enabled : digital input D4
  
  FC_ON : enable pump  : digital output D5
  FC_OFF: disable pump : digital output D6
 */

#include <SPI.h>
#include <Ethernet.h>
#define BUF_LEN 256

// inputs
int IL_ON = 4;
int FL_OK = 5;
int FC_OK = 6;
int FP_EN = 7;
  // outputs
int FC_ON = 2;
int FC_OFF = 3;

unsigned long last_time = millis();
unsigned long current_time = millis();
unsigned int fc_on_time = 0; // time in ms, for FC_ON to be HIGH
unsigned int fc_off_time = 0;// time in ms, for FC_OFF to be HIGH

byte mac[] = { 0xFA, 0xC7, 0xFC, 0xB1, 0x00, 0x01 };
IPAddress ip(129,217,160,65);

// Initialize the Ethernet server library
// with the IP address and port you want to use 
// (port 80 is default for HTTP):
EthernetServer server(80);

// ethernet receive buffer

char eth_rx_buf[BUF_LEN];
unsigned int eth_rd = 0;
unsigned int eth_wr = 0;
byte eth_buff_full = false;




void setup() {
 // Open serial communications and wait for port to open:
  Serial.begin(9600);
   while (!Serial) {
    ; // wait for serial port to connect. Needed for Leonardo only
  }


  // start the Ethernet connection and the server:
  Ethernet.begin(mac, ip);
  server.begin();
  Serial.print("server is at ");
  Serial.println(Ethernet.localIP());
  
  pinMode( IL_ON, INPUT );
  pinMode( FL_OK, INPUT );
  pinMode( FC_OK, INPUT );
  pinMode( FP_EN, INPUT );

  pinMode( FC_ON, OUTPUT );
  pinMode( FC_OFF,OUTPUT );

}


void loop() {
   Check_n_Set_FC_ON_OFF();
   unsigned long before = millis();
   if (Check_for_clients() )
   {
     unsigned long duration = millis() - before;
    
     Serial.println( duration, DEC);
   }

   delay(300);
}
/////////////////////////////////////////////////////////////////////////////////////////////
int Check_for_clients()
{

  int got_client = 0;
  // listen for incoming clients
  EthernetClient client = server.available();
  if (client) {
   got_client = 1;
    Serial.println("new client");
    // an http request ends with a blank line
    boolean currentLineIsBlank = true;
    boolean get_http_found = false;
    String eth = "";
    
    while (client.connected()) {
      if (client.available()) {
        char c = client.read();
        if (!get_http_found)
        {
         eth += c;
//        Serial.write(c);
         //Serial.println(eth.length(), DEC);
         if ( eth.indexOf('\n') != -1 )
         {
           if (eth.indexOf(String("GET /")) !=-1 &&  eth.indexOf(String("HTTP")) !=-1 )
           {
             get_http_found = true;
             if (eth.indexOf(String("D2on=")) !=-1){
               Serial.println("found D2on");
               digitalWrite( 2, HIGH);
               fc_on_time = 3000;
             }
             
             if (eth.indexOf(String("D2off=")) !=-1)
             {
               Serial.println("found D2off");
               digitalWrite( 3, HIGH);
               fc_off_time = 500;
             }
             
             Serial.println("found it");
           }
         }
        }

        // if you've gotten to the end of the line (received a newline
        // character) and the line is blank, the http request has ended,
        // so you can send a reply
        if (c == '\n' && currentLineIsBlank) {
          // send a standard http response header
          html_header( &client );
          html_page( &client );
          html_footer( &client );       
          break; // this breaks out of the while loop!!!
        }
        if (c == '\n') {
          // you're starting a new line
          currentLineIsBlank = true;
        } 
        else if (c != '\r') {
          // you've gotten a character on the current line
          currentLineIsBlank = false;
        }
      }
    }
    // give the web browser time to receive the data
    delay(1);
    // close the connection:
    client.stop();
    Serial.println("client disonnected");
  }
  return got_client;
}
/////////////////////////////////////////////////////////////////////////////////////////////

//////////// HTML Generators ////////////////////////////////////////////////////////////////
void html_header(EthernetClient *client)
{
  // send a standard http response header
  client->println("HTTP/1.1 200 OK");
  client->println("Content-Type: text/html");
  client->println("Connnection: close");
  client->println();
  client->println("<!DOCTYPE HTML>");
  client->println("<html><head>");
  
  // add a meta refresh tag, so the browser pulls again every 5 seconds:
  client->print("<meta http-equiv=\"refresh\" content=\"5\">");  
  
  client->println("<title>FACT camera power interlock control</title></head><body>");
  client->println("<FORM NAME=\"Form\">");
}

void send_ana_outs_example( EthernetClient *client) {
   // output the value of each analog input pin
  for (int analogChannel = 0; analogChannel < 6; analogChannel++) 
  {
    client->print("analog input ");
    client->print(analogChannel);
    client->print(" is ");
    client->print(analogRead(analogChannel));
    client->println("<br />");
  }
}

void send_dig_ins( EthernetClient *client) 
{
  for (int ch = 4; ch < 9; ch++) 
  {
    client->print("digital input ");
    client->print(ch);
    client->print(" is ");
    if (digitalRead(ch))
    {
      client->print("<font color=\"green\" size=20>HIGH</font>"); 
    }
    else
    {
      client->print("<font color=\"red\" size=20>LOW</font>"); 
    }
    
    
    
    client->println("<br />");
  }
 
}

void html_page( EthernetClient *client)
{
  client->println("<P><INPUT TYPE=SUBMIT NAME=\"D2on\" VALUE=\"Enable Pump\"></P> Switches D2 HIGH for 3 seconds <br />");
  client->println("<P><INPUT TYPE=SUBMIT NAME=\"D2off\" VALUE=\"Disable Pump\"></P> Switches D3 HIGH for 0.5 seconds <br />");
  
  send_dig_ins( client );
  
  //send_ana_outs_example( client );
}

void html_footer(EthernetClient *client)
{
  client->print("</FORM></body></html>\n");
}


//////////// HTML Generators ---- END /////////////////////////////////////////////////////////

void 
Check_n_Set_FC_ON_OFF()
{
  Serial.print("FC ON TIME");
  Serial.println(fc_on_time);
  Serial.print("FC OFF TIME");
  Serial.println(fc_off_time);
  
  unsigned long since_last = millis() - last_time;
  
  if (since_last == 0)
  {
    // not even a milisecond passed by
    // strange, should not happen
  }
  else if (since_last >= fc_on_time)
  {
    // FC_ON was long enough HIGH
    fc_on_time = 0;
  }
  else if (since_last < fc_on_time)
  {
    // fc_on_time is decreased
    fc_on_time -= since_last;
  }
  
  if (fc_on_time == 0)
  {
     digitalWrite( FC_ON, LOW);
  }
  else
  {
    //digitalWrite( FC_ON, HIGH);
  }
 
  if (since_last == 0)
  {
    // not even a milisecond passed by
    // strange, should not happen
  }
  else if (since_last >= fc_off_time)
  {
    // FC_OFF was long enough HIGH
    fc_off_time = 0;
  }
  else if (since_last < fc_off_time)
  {
    // fc_on_time is decreased
    fc_off_time -= since_last;
  }
  
  if (fc_off_time == 0)
  {
     digitalWrite( FC_OFF, LOW);
  }
  else
  {
    //digitalWrite( FC_OFF, HIGH);
  }
 
  last_time = millis(); 
}
/////////////////////////////////////////////////////////////////////////////////////////////


///////////////Ethernet input buffer////////////////////////////////////////////////////////
int eth_buf_add(char c)
{
 if (!eth_buff_full)
 {
  eth_rx_buf[eth_wr] = c;
  eth_wr = (eth_wr+1)%BUF_LEN;
  if (eth_wr == eth_rd) // buffer full
  {
   eth_buff_full = true;
  }
  return true;
 }
 else
 {
  return false;
 }
}

inline int eth_len()
{
 return (eth_wr-eth_rd) % BUF_LEN;
}

char * buf[16];





/////////////////////////////////////////////////////////////////////////////////////////////

