#include "csv.h"

Csv::Csv(TString path, TString fileName)
{
    Csv(path, fileName, "", 0);
}

Csv::Csv(TString path, TString fileName, int verbLevel)
{
    Csv(path, fileName, "", verbLevel);
}

Csv::Csv(TString path, TString fileName, TString suffix, int verbLevel)
{
    mDirectory  = path;
    mFilename   = fileName;
    mSuffix     = suffix;
    mVerbLevel  = verbLevel;

    BuildPath();
    OpenCsv();
}

Csv::~Csv()
{
    CloseCsv();
}

void
Csv::BuildPath()
{
    mPath = mDirectory;
    mPath.Append(mFilename);

    if (!mSuffix.IsNull()){

        if (mSuffix = -1)
            mSuffix = "AllPixel";

        mSuffix.Append("_");
        mPath.Append(mSuffix);
    }

    mPath.Append(".csv");
    return;
}

bool
Csv::WritePixelToCsv(
        Pixel*          pixel,
        TString         overlayMethod,
        int             order
        )
{
    if ( !overlayMethod.Contains("Edge") && !overlayMethod.Contains("Maximum") )
    {
        cout << endl << "Unknown Overlay Method-->aborting" << endl;
        return 0;
    }

    WritePointsetExplain();

    WritePointSetToCsv(
            pixel,
            overlayMethod,
            order
            );

    return 1;
}

void
Csv::WritePointSet(
        Pixel*          pixel,
        TString         overlayMethod,
        int             order
        )
{
    TH1F*  Max_histo        = NULL;
    TH1F*  Median_histo     = NULL;
    TH1F*  Mean_histo       = NULL;

    if (overlayMethod.Contains("Maximum"))
    {
        Max_histo       = pixel->hPixelMax[order];
        Median_histo    = pixel->hPixelMedian[order];
        Mean_histo      = pixel->hPixelMean[order];
    }
    else if (overlayMethod.Contains("Edge"))
    {
        Max_histo       = pixel->hPixelMax[order];
        Median_histo    = pixel->hPixelMedian[order];
        Mean_histo      = pixel->hPixelMean[order];
    }
    else
    {
        cout << endl << "Unknown Overlay Method-->aborting" << endl;
        return 1;
    }


    Int_t nbins = Max_histo->GetXaxis()->GetNbins();

    if (mVerbLevel > 2)
    {
        cout << "writing point-set to csv file: " ;
        cout << mPath << endl;
    }
    if (mVerbLevel > 2) cout << "...number of bins " << nbins << endl;

    //name coulums header
    mCsvFile << "pixel [CHid]"          << ",";
    mCsvFile << "OverlayPosition"       << ",";
    mCsvFile << "AmplitudeMax [mV]"     << ",";
    mCsvFile << "AmplitudeMean [mV]"    << ",";
    mCsvFile << "AmplitudeMedian [mV]";
    mCsvFile << endl;

    //fill coulums
    for (int TimeSlice=1;TimeSlice<=nbins;TimeSlice++)
    {
        mCsvFile << pixel->mChid << "," ;
        mCsvFile << overlayMethod << "," ;
        mCsvFile << TimeSlice << "," ;
        mCsvFile << Max_histo->GetBinContent(TimeSlice) << ",";
        mCsvFile << Mean_histo->GetBinContent(TimeSlice) << ",";
        mCsvFile << Median_histo->GetBinContent(TimeSlice) << endl;
    }
    return 0;
}

void
Csv::WritePointSetExplain()
{
    mCsvFile << "### point-set of a single photon pulse template"
             << endl;
             << "### Slice's Amplitude determined by calculating the "
             << endl
             << "### value of maximum propability of slice -> AmplitudeMax "
             << endl
             << "### mean of slice -> AmplitudeMean "
             << endl
             << "### median of slice -> AmplitudeMedian "
             << endl
             << "### for each slice"
             << endl
             << "### " << endl;
}

void
Csv::WritePulseAttributes(
        Pixel*          pixel,
        Pulse*          pulse,
        TString         overlayMethod
        )
{
    //name coulums header
    mCsvFile << "pixel [CHid]"      << ",";
    mCsvFile << "OverlayPosition"   << ",";
    mCsvFile << "ModelName"         << "," ;
    mCsvFile << "Bsl"               << ",";
    mCsvFile << "Height"            << "," ;
    mCsvFile << "T0"                << "," ;
    mCsvFile << "T1"                << "," ;
    mCsvFile << "Tau1"              << "," ;
    mCsvFile << "Tau2"              << "," ;
    mCsvFile << "Integral"          << "," ;
    mCsvFile << "Amplitude"         << "," ;
    mCsvFile << "PhE"               << "," ;
    mCsvFile << "Type"              << "," ;
    mCsvFile << "FitMin"            << "," ;
    mCsvFile << "FitMax"            << "," ;
    mCsvFile << "FitProb"           << "," ;
    mCsvFile << "FitNCalls"         << "," ;
    mCsvFile << "FitNdf"            << "," ;
    mCsvFile << "Chi2"                     ;
    mCsvFile << endl;

    //fill coulums
    for (int TimeSlice=1;TimeSlice<=nbins;TimeSlice++)
    {
        mCsvFile << pixel->mChid << "," ;
        mCsvFile << overlayMethod << "," ;
        mCsvFile << pulse->mName << "," ;
        mCsvFile << pulse->mBsl << "," ;
        mCsvFile << pulse->mHeight << "," ;
        mCsvFile << pulse->mT0 << "," ;
        mCsvFile << pulse->mT1 << "," ;
        mCsvFile << pulse->mTau1 << "," ;
        mCsvFile << pulse->mTau2 << "," ;
        mCsvFile << pulse->mIntegral << "," ;
        mCsvFile << pulse->mAmplitude << "," ;
        mCsvFile << pulse->mPhE << "," ;
        mCsvFile << pulse->mType << "," ;
        mCsvFile << pulse->mFitMin << "," ;
        mCsvFile << pulse->mFitMax << "," ;
        mCsvFile << pulse->mFitProb << "," ;
        mCsvFile << pulse->mFitNCalls << "," ;
        mCsvFile << pulse->mFitNdf << "," ;
        mCsvFile << pulse->mChi2 << "," ;
        mCsvFile << endl;
    }
}

void
Csv::OpenCsv(){
    if (mVerbLevel > 2) cout << "...opening csv file" << endl;
    mCsvFile.open( mPath );
}

void
Csv::CloseCsv(){
    mCsvFile.close();
    if (mVerbLevel > 2) cout << "...csv file closed" << endl;
}

void
Csv::GetFilename(){
    return mFilename;
}

void
Csv::GetPath(){
    return mPath;
}
