/*
  Arduino Ethernet for FACT Interlock
  
  Written by Dominik Neise
  June 2012
  TU Dortmund
  
  The FACT Camera as well as the bias voltage supply for its G-APDs
  may only be switched on, if the dedicated cooling systems are running.
  This arduino is connected via optocouplers to some status LEDs, 
  and other coolings system output signals, which show 
  the status of both cooling systems:
  * water cooling pump
    - flow meter 
    - level meter
  * bias crate fans
    - rotation frequency
  A simple WebServer is implemented for showing the 
  status and access the actual switches.

  In order to switch on the camera one needs to switch on 
  the water pump first. In case this pump is running, the 
  Camera power suplly (51VDC Agilent) gets an enable signal
  and will power the camera.
  In order to enable the bias voltage supply, one needs to swtich on 
  the Bias Crate Fans.  
  
  Circuit:
  IL_ON : Interlock on : digital input D1
  FL_OK : level ok     : digital input D2
  FC_OK : current ok   : digital input D3
  FP_EN : pump enabled : digital input D4
  
  FC_ON : enable pump  : digital output D5
  FC_OFF: disable pump : digital output D6
 */
#include <SPI.h>
#include <Ethernet.h>

// inputs
int FC_OK = 2;
int BP_ON = 3;

int IL_ON = 14;
int FL_OK = 15;
int FP_EN = 16;
int X_ON = 17;
int Y_ON = 18;
int Z_OK = 19;

int my_inputs[] =          {FC_OK,   BP_ON,   IL_ON,   FL_OK,   FP_EN,   X_ON,   Y_ON,   Z_OK};
//char * my_input_names[] = {"FC_OK", "BP_ON", "IL_ON", "FL_OK", "FP_EN", "X_ON", "Y_ON", "Z_OK"};

char * my_input_names[] = { "Is the cooling liquid flow meter measuring enough throuput? ", 
                            "Is Bias Power enabled? ", 
                            "Is 24VDC Interlock power supply enabled? ", 
                            "Is the Level of the cooling liquid ok? ", 
                            "Is the Pump powered? ", 
                            "<br />Is the Drive Cabinet Enable Signal on? ", 
                            "Is the Drive Cabinet Power Signal on? ", 
                            "Is Drive Container Enabled? "};

char * my_input_span_id[] = {"flow_meter", "bias_power", "power_24v", "level", "pump", "drive_enable", "drive_power", "drive_on"};

int Number_of_Buttons = 3;
// This is in HTML called the VALUE of the button,
// but it is in fact what is written on it.
char * button_caption[] = {
    "Camera ON",
    "Camera OFF",
    "Drive ON/OFF"};

// This is in HTML called the NAME of the button, and it is, what is transmit
// within the request. So this is a kind of reference string .. which i searh for 
// in the request ... so the strings schould be nice ... no capital letters no spaces
char * button_name[] = {
    "cam_on",
    "cam_off",
    "dt"}; // read drive_toggle

  // outputs
int FC_ON = 4;
int FC_OFF = 5;
int X = 6;
int Y = 7;


unsigned long last_time = millis();
unsigned long current_time = millis();
unsigned int fc_on_time = 0; // time in ms, for FC_ON to be HIGH
unsigned int fc_off_time = 0;// time in ms, for FC_OFF to be HIGH


byte mac[] = { 0xFA, 0xC7, 0xFC, 0xB1, 0x00, 0x01 }; // FACT internal network



// Initialize the Ethernet server library
// with the IP address and port you want to use 
// (port 80 is default for HTTP):
EthernetServer server(80);


boolean x_signal = false;
boolean y_signal = false;


void setup() {
 // Open serial communications and wait for port to open:
//  Serial.begin(9600);
  Serial.begin(19200);


  // start the Ethernet connection and the server:
  Ethernet.begin(mac);
  server.begin();
  Serial.print("server is at ");
  Serial.println(Ethernet.localIP());
  
  pinMode( FC_ON, OUTPUT );
  pinMode( FC_OFF,OUTPUT );
  pinMode( X, OUTPUT );
  pinMode( Y,OUTPUT );
  // the others are all INPUTs
}




void loop() {
   Check_n_Set_FC_ON_OFF();
    
   unsigned long before = millis();
    
   if (Check_for_clients() )
   {
     unsigned long duration = millis() - before;
    
     Serial.println( duration, DEC);
   }

   delay(300);
}




/////////////////////////////////////////////////////////////////////////////////////////////
int Check_for_clients()
{

  int got_client = 0;
  // listen for incoming clients
  EthernetClient client = server.available();
  if (client) {
   got_client = 1;
    Serial.println("new client");
    // an http request ends with a blank line
    boolean currentLineIsBlank = true;
    boolean get_http_found = false;
    String eth = "";
    
    while (client.connected()) {
      if (client.available()) {
        char c = client.read();
        if (!get_http_found)
        {
         eth += c;
         Serial.write(c);
         Serial.print(' ');
         Serial.print(c, HEX);
         Serial.print(' ');
         Serial.println(eth.length(), DEC);
         if ( eth.indexOf('\n') != -1 )
         {
           if (eth.indexOf(String("GET /")) !=-1 &&  eth.indexOf(String("HTTP")) !=-1 )
           {
             get_http_found = true;
             // -- Cam on --
             if (eth.indexOf(String(button_name[0])) !=-1){
               Serial.println("User request: Enable Pump");
               digitalWrite( FC_ON, HIGH);
               if ((fc_on_time+3000) > fc_on_time)
                  fc_on_time += 3000;
             }
             // -- Cam Off --
             else if (eth.indexOf(String(button_name[1])) !=-1)
             {
               Serial.println("User request: Disable Pump");
               digitalWrite( FC_OFF, HIGH);
               fc_off_time = 500;
             }
             // -- Drive Toggle --
             else if (eth.indexOf(String(button_name[2])) !=-1)
             {
                Serial.println("User request: Toggle Drive");
                x_signal = !x_signal;
                digitalWrite( X, x_signal );
                y_signal = !y_signal;
                digitalWrite( Y, y_signal );
             }
             //else if (eth.indexOf(String("toggle_y=")) !=-1)
             //{
             //  Serial.println("User request: Toggle Y");
             //  y_signal = !y_signal;
             //  digitalWrite( Y, y_signal );
             //}
              else
              {
              Serial.println("No User request - just sending webpage");
              }
             
             Serial.println("found it");
           }
         }
        }

        // if you've gotten to the end of the line (received a newline
        // character) and the line is blank, the http request has ended,
        // so you can send a reply
        if (c == '\n' && currentLineIsBlank) {
          // send a standard http response header
          html_header( &client );
          html_page( &client );
          html_footer( &client );       
          break; // this breaks out of the while loop!!!
        }
        if (c == '\n') {
          // you're starting a new line
          currentLineIsBlank = true;
        } 
        else if (c != '\r') {
          // you've gotten a character on the current line
          currentLineIsBlank = false;
        }
      }
    }
    // give the web browser time to receive the data
    delay(1);
    // close the connection:
    client.stop();
    Serial.println("client disonnected");
  }
  return got_client;
}
/////////////////////////////////////////////////////////////////////////////////////////////

//////////// HTML Generators ////////////////////////////////////////////////////////////////
void html_header(EthernetClient *client)
{
  // send a standard http response header
  client->print("HTTP/1.1 200 OK\n");
  client->print("Content-Type: text/html\n");
  client->print("Connnection: close\n");
  client->print("\n");
  client->print("<!DOCTYPE HTML>\n");
  client->print("<html><head>\n");
  
  // add a meta refresh tag, so the browser pulls again every 5 seconds:
  client->print("<meta http-equiv=\"refresh\" content=\"5; URL=index.html\" />\n");
  client->print("<meta http-equiv=\"Content-type\" content=\"text/html;charset=UTF-8\" />\n");  
  
  client->print("<title>FACT camera power interlock control</title></head><body>\n");
  client->print("<FORM NAME=\"Form\">\n");
}

void print_input_status( EthernetClient *client) 
{
  for (int ch = 0; ch < 8; ch++) 
  {
    client->print(my_input_names[ch]);
    client->print(" :");
    client->print("    ");
    client->print("<span id=\"");
    client->print(my_input_span_id[ch]);
    client->print("\" title=\"");
    if ( digitalRead(my_inputs[ch]) )
    {
      client->print("1\"> Yes");
      //client->print(" Yes "); 
    }
    else
    {
      client->print("0\"> No "); 
      //client->print(" No "); 
    }
    client->print("</span> <br />\n");
  }
}

void make_buttons( EthernetClient *client )
{
    for ( int button_id = 0; button_id < Number_of_Buttons; ++button_id )
    {
        client->print("<P><INPUT TYPE=SUBMIT NAME=\"");
        client->print(button_name[button_id]);
        client->print("\" VALUE=\"");
        client->print(button_caption[button_id]);
        client->print("\" ");
        
        // the 'OFF' button should be disabled 
        // in case somebody just pressed the 'ON' button.
        if (fc_on_time != 0 && button_id == 1){
            client->print("disabled");
        }
        
        client->print("><br />\n");
    }
}


void html_page( EthernetClient *client)
{
    make_buttons( client );
    print_input_status( client );
}

void html_footer(EthernetClient *client)
{
  client->print("</FORM></body></html>\n");
}


//////////// HTML Generators ---- END /////////////////////////////////////////////////////////

void 
Check_n_Set_FC_ON_OFF()
{
  
  unsigned long since_last = millis() - last_time;
  
  if (since_last == 0)
  {
    // not even a milisecond passed by
    // strange, should not happen
  }
  else if (since_last >= fc_on_time)
  {
    // FC_ON was long enough HIGH
    fc_on_time = 0;
  }
  else if (since_last < fc_on_time)
  {
    // fc_on_time is decreased
    fc_on_time -= since_last;
  }
  
  if (fc_on_time == 0)
  {
     digitalWrite( FC_ON, LOW);
  }
  else
  {
    //digitalWrite( FC_ON, HIGH);
  }
 
  if (since_last == 0)
  {
    // not even a milisecond passed by
    // strange, should not happen
  }
  else if (since_last >= fc_off_time)
  {
    // FC_OFF was long enough HIGH
    fc_off_time = 0;
  }
  else if (since_last < fc_off_time)
  {
    // fc_on_time is decreased
    fc_off_time -= since_last;
  }
  
  if (fc_off_time == 0)
  {
     digitalWrite( FC_OFF, LOW);
  }
  else
  {
    //digitalWrite( FC_OFF, HIGH);
  }
 
  last_time = millis(); 
}
/////////////////////////////////////////////////////////////////////////////////////////////

