#include "csv.h"

Csv::Csv(TString path, TString fileName)
{
    Csv(path, fileName, "", 0);
}

Csv::Csv(TString path, TString fileName, int verbLevel)
{
    Csv(path, fileName, "", verbLevel);
}

Csv::Csv(TString path, TString fileName, TString suffix, int verbLevel)
{
    mDirectory  = path;
    mFilename   = fileName;
    mSuffix     = suffix;
    mVerbLevel  = verbLevel;
    mSeparator  = ";";

    BuildPath();
    OpenCsv();
}

Csv::~Csv()
{
    CloseCsv();
}

// ===========================================================================
// PRIVATE OPERATIONS
// ===========================================================================

void
Csv::BuildPath()
{
    mPath = mDirectory;
    mPath.Append(mFilename);

    if (!mSuffix.IsNull()){

        if (mSuffix.Contains("-1"))
            mSuffix = "AllPixel";

        mSuffix.Prepend("_");
        mPath.Append(mSuffix);
    }

    mPath.Append(".csv");
    return;
}

// ===========================================================================
// PUBLIC OPERATIONS
// ===========================================================================

bool
Csv::WritePointSetToCsv(
        Pixel*          pixel,
        TString         overlayMethod,
        int             order
        )
{
    if ( !overlayMethod.Contains("Edge") && !overlayMethod.Contains("Maximum") )
    {
        cout << endl << "Unknown Overlay Method-->aborting" << endl;
        return 0;
    }

    WritePointSetHeader();

    WritePointSet(
            pixel,
            overlayMethod,
            order
            );

    return 1;
}

void
Csv::WritePointSet(
        Pixel*          pixel,
        TString         overlayMethod,
        int             order
        )
{
    TH1F*  Max_histo        = NULL;
    TH1F*  Median_histo     = NULL;
    TH1F*  Mean_histo       = NULL;
    int overlayPos          = 0;

    if (overlayMethod.Contains("Maximum"))
    {
        Max_histo       = pixel->hPixelMax[order];
        Median_histo    = pixel->hPixelMedian[order];
        Mean_histo      = pixel->hPixelMean[order];
        overlayPos      = 1;
    }
    else if (overlayMethod.Contains("Edge"))
    {
        Max_histo       = pixel->hPixelMax[order];
        Median_histo    = pixel->hPixelMedian[order];
        Mean_histo      = pixel->hPixelMean[order];
        overlayPos      = 2;
    }
    else
    {
        cout << endl << "Unknown Overlay Method-->aborting" << endl;
        return ;
    }


    Int_t nbins = Max_histo->GetXaxis()->GetNbins();

    if (mVerbLevel > 2)
    {
        cout << "writing point-set to csv file: " ;
        cout << mPath << endl;
    }
    if (mVerbLevel > 2) cout << "...number of bins " << nbins << endl;

    //fill coulums
    for (int TimeSlice=1;TimeSlice<=nbins;TimeSlice++)
    {
        mCsvFile << TimeSlice << mSeparator ;
        mCsvFile << pixel->mChid << mSeparator ;
        mCsvFile << order << mSeparator ;
        mCsvFile << overlayPos << mSeparator ;
        mCsvFile.precision(8);
        mCsvFile << Max_histo->GetBinContent(TimeSlice) << mSeparator;
        mCsvFile << Mean_histo->GetBinContent(TimeSlice) << mSeparator;
        mCsvFile << Median_histo->GetBinContent(TimeSlice) << endl;
        mCsvFile.precision(mPrecision);
    }

    return ;
}

void
Csv::WritePointSetHeader()
{
    mCsvFile << "### point-set of a single photon pulse template"
             << endl
             << "### Slice's Amplitude determined by calculating the "
             << endl
             << "### value of maximum propability of slice -> AmplitudeMax "
             << endl
             << "### mean of slice -> AmplitudeMean "
             << endl
             << "### median of slice -> AmplitudeMedian "
             << endl
             << "### for each slice"
             << endl
             << "### "
             << endl
             << "### OverlayMethods (1) at Maximum, (2) at rising Edge" << endl;

    //name coulums header
    mCsvFile << "TimeSlice [a.u.]"          << mSeparator;
    mCsvFile << "pixel [CHid]"          << mSeparator;
    mCsvFile << "Pulseorder [PhE]"          << mSeparator;
    mCsvFile << "OverlayPosition"       << mSeparator;
    mCsvFile << "AmplitudeMax [mV]"     << mSeparator;
    mCsvFile << "AmplitudeMean [mV]"    << mSeparator;
    mCsvFile << "AmplitudeMedian [mV]";
    mCsvFile << endl;
}

void
Csv::WritePulseAttributesHeader()
{
    //name coulums header
    mCsvFile << "pixel [CHid]"      << mSeparator;
    mCsvFile << "OverlayPosition"   << mSeparator;
    mCsvFile << "ModelName"         << mSeparator ;
    mCsvFile << "Bsl"               << mSeparator;
    mCsvFile << "BslErr"            << mSeparator;
    mCsvFile << "Height"            << mSeparator ;
    mCsvFile << "HeightErr"         << mSeparator ;
    mCsvFile << "T0"                << mSeparator ;
    mCsvFile << "T0Err"             << mSeparator ;
    mCsvFile << "T1"                << mSeparator ;
    mCsvFile << "T1Err"             << mSeparator ;
    mCsvFile << "Tau1"              << mSeparator ;
    mCsvFile << "Tau1Err"           << mSeparator ;
    mCsvFile << "Tau2"              << mSeparator ;
    mCsvFile << "Tau2Err"           << mSeparator ;
    mCsvFile << "Integral"          << mSeparator ;
    mCsvFile << "IntegralErr"       << mSeparator ;
    mCsvFile << "Amplitude"         << mSeparator ;
    mCsvFile << "AmplitudeErr"      << mSeparator ;
    mCsvFile << "Phe"               << mSeparator ;
    mCsvFile << "PheErr"            << mSeparator ;
    mCsvFile << "Order"             << mSeparator ;
    mCsvFile << "Type"              << mSeparator ;
    mCsvFile << "FitProb"           << mSeparator ;
    mCsvFile << "FitNCalls"         << mSeparator ;
    mCsvFile << "FitNdf"            << mSeparator ;
    mCsvFile << "Chi2"                     ;
    mCsvFile << endl;
}
void
Csv::WritePulseAttributes(
        Pixel*          pixel,
        Pulse*          pulse,
        TString         overlayMethod,
        int             order
        )
{
    //fill coulums
    mCsvFile << pixel->mChid << mSeparator ;
    mCsvFile << overlayMethod << mSeparator ;
    mCsvFile << pulse->GetName() << mSeparator ;
    mCsvFile.precision(12);
    mCsvFile << pulse->GetBsl() << mSeparator ;
    mCsvFile << pulse->GetBslErr() << mSeparator ;
    mCsvFile << pulse->GetHeight() << mSeparator ;
    mCsvFile << pulse->GetHeightErr() << mSeparator ;
    mCsvFile.precision(12);
    mCsvFile << pulse->GetT0() << mSeparator ;
    mCsvFile << pulse->GetT0Err() << mSeparator ;
    mCsvFile << pulse->GetT1() << mSeparator ;
    mCsvFile << pulse->GetT1Err() << mSeparator ;
    mCsvFile << pulse->GetTau1() << mSeparator ;
    mCsvFile << pulse->GetTau1Err() << mSeparator ;
    mCsvFile << pulse->GetTau2() << mSeparator ;
    mCsvFile << pulse->GetTau2Err() << mSeparator ;
    mCsvFile.precision(8);
    mCsvFile << pulse->GetIntegral() << mSeparator ;
    mCsvFile << pulse->GetAmplitude() << mSeparator ;
    mCsvFile.precision(2);
    mCsvFile << pulse->GetPhE() << mSeparator ;
    mCsvFile << pulse->GetPhEErr() << mSeparator ;
    mCsvFile << order << mSeparator ;
    mCsvFile << pulse->GetType() << mSeparator ;
    mCsvFile << pulse->GetFitProb() << mSeparator ;
    mCsvFile.precision(12);
    mCsvFile << pulse->GetFitNCalls() << mSeparator ;
    mCsvFile << pulse->GetFitNdf() << mSeparator ;
    mCsvFile << pulse->GetChi2() << mSeparator ;
    mCsvFile << endl;
    mCsvFile.precision(mPrecision);
}

void
Csv::OpenCsv(){
    if (mVerbLevel > 2) cout << "...opening csv file" << endl;
    mCsvFile.open( mPath );
    mPrecision = mCsvFile.precision();
}

void
Csv::CloseCsv(){
    mCsvFile.close();
    if (mVerbLevel > 2) cout << "...csv file closed" << endl;
}

// ===========================================================================
// ACCESS
// ===========================================================================

TString
Csv::GetFilename(){
    return mFilename;
}

TString
Csv::GetPath(){
    return mPath;
}
