#ifndef FACT_InterpreterV8
#define FACT_InterpreterV8

#include <map>
#include <list>
#include <string>

#ifdef HAVE_V8
#include <v8.h>
#endif

#include "State.h"
#include "Description.h"
#include "EventImp.h"

class Database;

class InterpreterV8
{
    static InterpreterV8 *This;

    // The main thread id, needed to be able to terminate
    // the thread forcefully from 'the outside'
    int fThreadId;

    // A loookup table which allows to indentify the
    // the JavaScript object corrsponding to the
    // service name (for checking of an .onchange
    // subscription exists for that object)
    std::map<std::string, v8::Persistent<v8::Object>> fReverseMap;

    // Lookup table for the callbacks in cases of state changes
    std::map<std::string, v8::Persistent<v8::Value>>  fStateCallbacks;

#ifdef HAVE_SQL
    std::list<Database*> fDatabases;
#endif

#ifdef HAVE_V8
    bool ReportException(v8::TryCatch* try_catch);
    bool ExecuteStringNT(const v8::Handle<v8::String> &code, const v8::Handle<v8::Value>  &file);
    bool ExecuteCode(const v8::Handle<v8::String> &code, const v8::Handle<v8::Value>  &file);
    bool ExecuteCode(const std::string &code, const std::string &file="");
    bool ExecuteFile(const std::string &name);

    v8::Handle<v8::Value> FuncWait(const v8::Arguments& args);
    v8::Handle<v8::Value> FuncSend(const v8::Arguments& args);
    v8::Handle<v8::Value> FuncSleep(const v8::Arguments& args);
    v8::Handle<v8::Value> FuncPrint(const v8::Arguments& args);
    v8::Handle<v8::Value> FuncAlarm(const v8::Arguments& args);
    v8::Handle<v8::Value> FuncOut(const v8::Arguments& args);
    v8::Handle<v8::Value> FuncInclude(const v8::Arguments& args);
    v8::Handle<v8::Value> FuncExit(const v8::Arguments& args);
    v8::Handle<v8::Value> FuncState(const v8::Arguments& args);
    v8::Handle<v8::Value> FuncOpen(const v8::Arguments& args);
    v8::Handle<v8::Value> FuncGetData(const v8::Arguments &args);
    v8::Handle<v8::Value> FuncClose(const v8::Arguments &args);
    v8::Handle<v8::Value> FuncQuery(const v8::Arguments &args);
    v8::Handle<v8::Value> FuncDatabase(const v8::Arguments &args);
    v8::Handle<v8::Value> FuncDbQuery(const v8::Arguments &args);
    v8::Handle<v8::Value> FuncDbClose(const v8::Arguments &args);
    v8::Handle<v8::Value> OnChangeSet(v8::Local<v8::String>, v8::Local<v8::Value>, const v8::AccessorInfo &);

    static v8::Handle<v8::Value> FuncVersion(const v8::Arguments&);
    static v8::Handle<v8::Value> WrapInclude(const v8::Arguments &args)  { if (This) return This->FuncInclude(args);  else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapPrint(const v8::Arguments &args)    { if (This) return This->FuncPrint(args);    else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapAlarm(const v8::Arguments &args)    { if (This) return This->FuncAlarm(args);    else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapOut(const v8::Arguments &args)      { if (This) return This->FuncOut(args);      else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapWait(const v8::Arguments &args)     { if (This) return This->FuncWait(args);     else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapSend(const v8::Arguments &args)     { if (This) return This->FuncSend(args);     else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapSleep(const v8::Arguments &args)    { if (This) return This->FuncSleep(args);    else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapExit(const v8::Arguments &args)     { if (This) return This->FuncExit(args);     else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapState(const v8::Arguments &args)    { if (This) return This->FuncState(args);    else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapOpen(const v8::Arguments &args)     { if (This) return This->FuncOpen(args);     else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapGetData(const v8::Arguments &args)  { if (This) return This->FuncGetData(args);  else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapClose(const v8::Arguments &args)    { if (This) return This->FuncClose(args);    else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapQuery(const v8::Arguments &args)    { if (This) return This->FuncQuery(args);    else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapDatabase(const v8::Arguments &args) { if (This) return This->FuncDatabase(args); else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapDbQuery(const v8::Arguments &args)  { if (This) return This->FuncDbQuery(args);  else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapDbClose(const v8::Arguments &args)  { if (This) return This->FuncDbClose(args);  else return v8::Undefined(); }
    static v8::Handle<v8::Value> WrapOnChangeSet(v8::Local<v8::String> prop, v8::Local<v8::Value> value, const v8::AccessorInfo &info)
    {
        if (This) return This->OnChangeSet(prop, value, info);  else return v8::Undefined();
    }

    static v8::Handle<v8::Value> OnChangeGet(v8::Local<v8::String>, const v8::AccessorInfo &)
    {
        return v8::Handle<v8::Value>();
    }

    static v8::Handle<v8::Value> Convert(char type, const char* &ptr);
    v8::Handle<v8::Value> ConvertEvent(const EventImp *evt, uint64_t, const char *str);
#endif

public:
    InterpreterV8() : fThreadId(-1)
    {
        This = this;
    }
    virtual ~InterpreterV8()
    {
        This = 0;

#ifdef HAVE_V8
        v8::Locker locker;
        v8::V8::Dispose();
#endif
    }

    virtual void  JsLoad(const std::string &) { }
    virtual void  JsStart(const std::string &) { }
    virtual void  JsEnd(const std::string &) { }
    virtual void  JsPrint(const std::string &) { }
    virtual void  JsAlarm(const std::string & = "") { }
    virtual void  JsOut(const std::string &) { }
    virtual void  JsResult(const std::string &) { }
    virtual void  JsException(const std::string &) { }
    virtual bool  JsSend(const std::string &) { return true; }
    //virtual void  JsSleep(uint32_t) { }
    //virtual int   JsWait(const std::string &, int32_t, uint32_t) { return -1; };
    virtual State JsState(const std::string &) { return State(); };
    virtual void *JsSubscribe(const std::string &) { return 0; };
    virtual bool  JsUnsubscribe(const std::string &) { return false; };
    virtual std::vector<Description> JsDescription(const std::string &) { return std::vector<Description>(); };
    virtual std::pair<uint64_t, EventImp *> JsGetEvent(const std::string &) { return std::make_pair(0, (EventImp*)0); };

    void JsHandleEvent(const EventImp &, uint64_t, const std::string &);
    void JsHandleState(const std::string &, const State &);

    bool JsRun(const std::string &, const std::map<std::string,std::string> & = std::map<std::string,std::string>());
    static void JsStop();
};

#ifndef HAVE_V8
inline bool InterpreterV8::JsRun(const std::string &, const std::map<std::string,std::string> &) { return false; }
inline void InterpreterV8::JsStop() { }
#endif

#endif
