#include "InterpreterV8.h"

#ifdef HAVE_SQL
#include "Database.h"
#endif

#include <boost/tokenizer.hpp>

InterpreterV8 *InterpreterV8::This = 0;

#ifdef HAVE_V8

#include <v8.h>
#include <fstream>
#include <sstream>
#include <iomanip>

using namespace std;
using namespace v8;

bool InterpreterV8::ReportException(TryCatch* try_catch)
{
    if (!try_catch->CanContinue())
        return false;

    const HandleScope handle_scope;

    const String::Utf8Value exception(try_catch->Exception());

    if (*exception && string(*exception)=="exit")
        return true;
    if (*exception && string(*exception)=="null")
        return false;

    const Handle<Message> message = try_catch->Message();
    if (message.IsEmpty())
        return false;

    // Print (filename):(line number): (message).
    const String::Utf8Value filename(message->GetScriptResourceName());

    ostringstream out;

    if (*filename)
        out << *filename << ": ";
    out << "l." << message->GetLineNumber();
    if (*exception)
        out << ": " << *exception;

    JsException(out.str());

    // Print line of source code.
    const String::Utf8Value sourceline(message->GetSourceLine());
    if (*sourceline)
        JsException(*sourceline);

    // Print wavy underline (GetUnderline is deprecated).
    const int start = message->GetStartColumn();
    const int end   = message->GetEndColumn();

    out.str("");
    if (start>0)
        out << setfill(' ') << setw(start) << ' ';
    out << setfill('^') << setw(end-start) << '^';

    JsException(out.str());

    String::Utf8Value stack_trace(try_catch->StackTrace());
    if (stack_trace.length()<=0)
        return false;

    //if (*stack_trace)
    //    JsException(string("\n")+*stack_trace);

    return false;
}

// Executes a string within the current v8 context.
bool InterpreterV8::ExecuteStringNT(const Handle<String> &code, const Handle<Value> &file)
{
    if (code.IsEmpty())
        return true;

    const HandleScope handle_scope;

    const Handle<Script> script = Script::Compile(code, file);
    if (script.IsEmpty())
        return false;

    JsSetState(3);

    const Handle<Value> result = script->Run();
    if (result.IsEmpty())
        return false;

    // If all went well and the result wasn't undefined then print
    // the returned value.
    if (!result->IsUndefined())
        JsResult(*String::Utf8Value(result));

    return true;
}

bool InterpreterV8::ExecuteCode(const Handle<String> &code, const Handle<Value> &file)
{
    TryCatch exception;

    const bool rc = ExecuteStringNT(code, file);

    // Check if this is a termination exception
    //if (!exception.CanContinue())
    //    return false;

    if (exception.HasCaught())
        return ReportException(&exception);

    return rc;
}

bool InterpreterV8::ExecuteCode(const string &code, const string &file)
{
    return ExecuteCode(String::New(code.c_str(), code.size()),
                       String::New(file.c_str()));
}

bool InterpreterV8::ExecuteFile(const string &name)
{
    ifstream fin(name.c_str());
    if (!fin)
    {
        JsException("Error - Could not open file '"+name+"'");
        return false;
    }

    string buffer;
    if (!getline(fin, buffer, '\0'))
        return true;

    if (fin.fail())
    {
        JsException("Error - reading file.");
        return false;
    }

    return ExecuteCode(buffer, name);
}

Handle<Value> InterpreterV8::FuncWait(const Arguments& args)
{
    if (args.Length()!=2 && args.Length()!=3)
        return ThrowException(String::New("Number of arguments must be 2 or 3."));

    if (!args[0]->IsString())
        return ThrowException(String::New("Argument 1 not a string."));

    if (!args[1]->IsInt32() && !args[1]->IsString())
        return ThrowException(String::New("Argument 2 not an int32 and not a string."));

    if (args.Length()==3 && !args[2]->IsUint32())
        return ThrowException(String::New("Argument 3 not an uint32."));

    // Using a Javascript function has the advantage that it is fully
    // interruptable without the need of C++ code

    const string index   = args[1]->IsInt32() ? "s.index" : "s.name";
    const bool   timeout = args.Length()==3;
    const string arg0    = *String::Utf8Value(args[0]);
    const string state   = args[1]->IsString() ? *String::Utf8Value(args[1]) : "";
    const string arg1    = args[1]->IsString() ? ("\""+state+"\"") : to_string(args[1]->Int32Value());

    if (arg0.find_first_of("\"'")!=string::npos)
        return ThrowException(String::New("Server name must not contain quotation marks."));

    if (args[1]->IsString())
        if (state.find_first_of("\"'")!=string::npos)
            return ThrowException(String::New("State name must not contain quotation marks."));

    string code =  "(function(name,state,ms)"
                   "{";
    if (timeout)
        code +=       "var t = new Date();";
    code +=           "while (1)"
                      "{"
                         "var s = dim.state(name);"
                         "if(!"+index+")throw 'Waitig for state "+arg1+" of server "+arg0+" failed.';"
                         "if(state=="+index+")return true;";
    if (timeout)
        code +=          "if((new Date()-t)>ms)return false;";

    code +=              "dim.sleep();"
                      "}"
                   "})('"+arg0+"',"+arg1;
    if (timeout)
        code +=    "," + to_string(args[2]->Int32Value());
    code +=        ");";

    const HandleScope handle_scope;

    // It is not strictly necessary to catch the exception, instead
    // script->Run() could just be returned, but catching the
    // exception allow to print the position in the file in
    // the exception handler instead of just the posiiton in the script.
    TryCatch exception;

    const Handle<Script> script = Script::Compile(String::New(code.c_str()));
    const Handle<Value>  result = script->Run();

    return exception.HasCaught() ? exception.ReThrow() : result;

    /*
    const string   server   = *String::Utf8Value(args[0]);
    const  int32_t state    = args[1]->Int32Value();
    const uint32_t millisec = args.Length()==3 ? args[2]->Int32Value() : 0;

    const int rc = JsWait(server, state, millisec);

    if (rc==0 || rc==1)
        return Boolean::New(rc);

    return ThrowException(String::New(("Waitig for state "+to_string(state)+" of server '"+server+"' failed.").c_str()));
        */
}

Handle<Value> InterpreterV8::FuncSend(const Arguments& args)
{
    if (args.Length()==0)
        return ThrowException(String::New("Number of arguments must be at least 1."));

    if (!args[0]->IsString())
        return ThrowException(String::New("Argument 1 must be a string."));

    const HandleScope handle_scope;

    const String::Utf8Value str(args[0]);

    string command = *str;

    // Escape all string arguments. All others can be kept as they are.
    for (int i=1; i<args.Length(); i++)
    {
        const String::Utf8Value arg(args[i]);
        if (args[i]->IsString())
            command += " \""+string(*arg)+"\"";
        else
            command += " "+string(*arg);
    }

    return Boolean::New(JsSend(command));
}

Handle<Value> InterpreterV8::FuncSleep(const Arguments& args)
{
    if (args.Length()==0)
    {
        // Theoretically, the CPU usage can be reduced by maybe a factor
        // of four using a larger value, but this also means that the
        // JavaScript is locked for a longer time.
        usleep(1000);
        return Undefined();
    }

    if (args.Length()!=1)
        return ThrowException(String::New("Number of arguments must be exactly 1."));

    if (!args[0]->IsUint32())
        return ThrowException(String::New("Argument 1 must be an uint32."));

    // Using a Javascript function has the advantage that it is fully
    // interruptable without the need of C++ code

    const string code =
        "(function(){"
        "var t=new Date();"
        "while ((new Date()-t)<"+to_string(args[0]->Int32Value())+") dim.sleep();"
        "})();";

    const HandleScope handle_scope;

    const Handle<Script> script = Script::Compile(String::New(code.c_str()));
    return script->Run();

    //JsSleep(args[0]->Int32Value());
    //return Undefined();
}

Handle<Value> InterpreterV8::FuncState(const Arguments& args)
{
    if (args.Length()!=1)
        return ThrowException(String::New("Number of arguments must be exactly 1."));

    if (!args[0]->IsString())
        return ThrowException(String::New("Argument 1 must be a string."));

    // Return state.name/state.index

    const String::Utf8Value str(args[0]);

    const State rc = JsState(*str);

    //if (rc.first<=-256)
    //    return Undefined();

    HandleScope handle_scope;

    // It is important to catch the exception thrown
    // by Date::New in case of thread termination!
    Local<Value> date;
    {
        TryCatch exception;
        date = Date::New(rc.time.JavaDate());
        if (exception.HasCaught())
            return exception.ReThrow();
    }

    Handle<ObjectTemplate> obj = ObjectTemplate::New();
    obj->Set(String::New("index"), rc.index<=-256?Undefined():Integer::New(rc.index),       ReadOnly);
    obj->Set(String::New("name"),  rc.index<=-256?Undefined():String::New(rc.name.c_str()), ReadOnly);
    if (rc.index>-256)
        obj->Set(String::New("time"),  date);
    //obj->Set(String::New("toString"),  String::New(("[Object state "+string(*str)+":"+to_string(rc.index)+"]").c_str()));

    return handle_scope.Close(obj->NewInstance());
}

Handle<Value> InterpreterV8::FuncNewState(const Arguments& args)
{
    if (args.Length()<1 || args.Length()>3)
        return ThrowException(String::New("Number of arguments must be 1, 2 or 3."));

    if (!args[0]->IsUint32())
        return ThrowException(String::New("Argument 1 must be an uint32."));
    if (args.Length()>1 && !args[1]->IsString())
        return ThrowException(String::New("Argument 2 must be a string."));
    if (args.Length()>2 && !args[2]->IsString())
        return ThrowException(String::New("Argument 3 must be a string."));

    const HandleScope handle_scope;

    const uint32_t index   = args[0]->Int32Value();
    const string   name    = *String::Utf8Value(args[1]);
    const string   comment = *String::Utf8Value(args[2]);

    if (index<10 || index>255)
        return ThrowException(String::New("State must be in the range [10, 255]."));

    if (name.empty())
        return ThrowException(String::New("State name must not be empty."));

    if (name.find_first_of(':')!=string::npos || name.find_first_of('=')!=string::npos)
        return ThrowException(String::New("State name must not contain : or =."));

    if (JsHasState(index) || JsHasState(name))
    {
        const string what =
            "State index ["+to_string(index)+"] or name ["+name+"] already defined.";

        return ThrowException(String::New(what.c_str()));
    }

    return Boolean::New(JsNewState(index, name, comment));
}

Handle<Value> InterpreterV8::FuncSetState(const Arguments& args)
{
    if (args.Length()!=1)
        return ThrowException(String::New("Number of arguments must be exactly 1."));

    if (!args[0]->IsUint32() && !args[0]->IsString())
        return ThrowException(String::New("Argument must be an unint32 or a  string."));

    const HandleScope handle_scope;

    int index = -2;
    if (args[0]->IsUint32())
    {
        index = args[0]->Int32Value();
    }
    else
    {
        const string name = *String::Utf8Value(args[0]);
        index = JsGetState(name);
        if (index==-2)
            return ThrowException(String::New(("State '"+name+"' not found.").c_str()));
    }

    if (index<10 || index>255)
        return ThrowException(String::New("State must be in the range [10, 255]."));

    return Boolean::New(JsSetState(index));
}

Handle<Value> InterpreterV8::FuncExit(const Arguments &)
{
    V8::TerminateExecution(fThreadId);
    return ThrowException(String::New("exit"));
/*
    if (args.Length()!=1)
        return ThrowException(String::New("Number of arguments must be exactly 1."));

    if (!args[0]->IsUint32())
        return ThrowException(String::New("Argument 1 must be an uint32."));

    const HandleScope handle_scope;

    JsSleep(args[0]->Int32Value());
*/
    return Undefined();
}

// The callback that is invoked by v8 whenever the JavaScript 'print'
// function is called.  Prints its arguments on stdout separated by
// spaces and ending with a newline.
Handle<Value> InterpreterV8::FuncPrint(const Arguments& args)
{
    for (int i=0; i<args.Length(); i++)
    {
        const HandleScope handle_scope;

        const String::Utf8Value str(args[i]);
        if (*str)
            JsPrint(*str);
    }
    return Undefined();
}

Handle<Value> InterpreterV8::FuncAlarm(const Arguments& args)
{
    for (int i=0; i<args.Length(); i++)
    {
        const HandleScope handle_scope;

        const String::Utf8Value str(args[i]);
        if (*str)
            JsAlarm(*str);
    }

    if (args.Length()==0)
        JsAlarm();

    return Undefined();
}

Handle<Value> InterpreterV8::FuncOut(const Arguments& args)
{
    for (int i=0; i<args.Length(); i++)
    {
        const HandleScope handle_scope;

        const String::Utf8Value str(args[i]);
        if (*str)
            JsOut(*str);
    }
    return Undefined();
}

// The callback that is invoked by v8 whenever the JavaScript 'load'
// function is called.  Loads, compiles and executes its argument
// JavaScript file.
Handle<Value> InterpreterV8::FuncInclude(const Arguments& args)
{
    for (int i=0; i<args.Length(); i++)
    {
        const HandleScope handle_scope;

        const String::Utf8Value file(args[i]);
        if (*file == NULL)
            return ThrowException(String::New(("Error loading file '"+string(*file)+"'").c_str()));

        if (!ExecuteFile(*file))
            return ThrowException(String::New(("Execution of '"+string(*file)+"' failed").c_str()));
    }
    return Undefined();
}

Handle<Value> InterpreterV8::FuncVersion(const Arguments&)
{
    return String::New(V8::GetVersion());
}

Handle<Value> InterpreterV8::FuncDbClose(const Arguments &args)
{
    HandleScope handle_scope;

    void *ptr = Handle<External>::Cast(args.This()->GetInternalField(0))->Value();
    if (!ptr)
        return Boolean::New(false);

#ifdef HAVE_SQL
    Database *db = reinterpret_cast<Database*>(ptr);
    auto it = find(fDatabases.begin(), fDatabases.end(), db);
    fDatabases.erase(it);
    delete db;
#endif

    args.This()->SetInternalField(0, External::New(0));

    return Boolean::New(true);
}
Handle<Value> InterpreterV8::FuncDbQuery(const Arguments &args)
{
    if (args.Length()!=1)
        return ThrowException(String::New("Number of arguments must be exactly 1."));

    if (!args[0]->IsString())
        return ThrowException(String::New("Both arguments must be a string."));

    HandleScope handle_scope;

    void *ptr = Handle<External>::Cast(args.This()->GetInternalField(0))->Value();
    if (!ptr)
        return Undefined();

    const String::Utf8Value query(args[0]);

#ifdef HAVE_SQL
    try
    {
        Database *db = reinterpret_cast<Database*>(ptr);

        const mysqlpp::StoreQueryResult res = db->query(*query).store();

        Handle<Array> ret = Array::New();
        ret->Set(String::New("table"), String::New(res.table()), ReadOnly);

        Handle<Array> cols = Array::New();

        int irow=0;
        for (vector<mysqlpp::Row>::const_iterator it=res.begin(); it<res.end(); it++)
        {
            Handle<Array> row = Array::New();

            const mysqlpp::FieldNames *list = it->field_list().list;

            for (size_t i=0; i<it->size(); i++)
            {
                const Handle<Value> name = String::New((*list)[i].c_str());
                if (irow==0)
                    cols->Set(Integer::NewFromUnsigned(i), name, ReadOnly);

                if ((*it)[i].is_null())
                {
                    row->Set(name, Undefined(), ReadOnly);
                    continue;
                }

                const string sql_type = (*it)[i].type().sql_name();

                const bool uns = sql_type.find("UNSIGNED")==string::npos;

                if (sql_type.find("BIGINT")!=string::npos)
                {
                    if (uns)
                    {
                        const uint64_t val = (uint64_t)(*it)[i];
                        if (val>UINT32_MAX)
                            row->Set(name, Number::New(val), ReadOnly);
                        else
                            row->Set(name, Integer::NewFromUnsigned(val), ReadOnly);
                    }
                    else
                    {
                        const int64_t val = (int64_t)(*it)[i];
                        if (val<INT32_MIN || val>INT32_MAX)
                            row->Set(name, Number::New(val), ReadOnly);
                        else
                            row->Set(name, Integer::NewFromUnsigned(val), ReadOnly);
                    }
                    continue;
                }

                // 32 bit
                if (sql_type.find("INT")!=string::npos)
                {
                    if (uns)
                        row->Set(name, Integer::NewFromUnsigned((uint32_t)(*it)[i]), ReadOnly);
                    else
                        row->Set(name, Integer::New((int32_t)(*it)[i]), ReadOnly);
                }

                if (sql_type.find("BOOL")!=string::npos )
                {
                    row->Set(name, Boolean::New((bool)(*it)[i]), ReadOnly);
                    continue;
                }

                if (sql_type.find("FLOAT")!=string::npos)
                {
                    ostringstream val;
                    val << setprecision(7) << (float)(*it)[i];
                    row->Set(name, Number::New(stod(val.str())), ReadOnly);
                    continue;

                }
                if (sql_type.find("DOUBLE")!=string::npos)
                {
                    row->Set(name, Number::New((double)(*it)[i]), ReadOnly);
                    continue;
                }

                if (sql_type.find("CHAR")!=string::npos ||
                    sql_type.find("TEXT")!=string::npos)
                {
                    row->Set(name, String::New((const char*)(*it)[i]), ReadOnly);
                    continue;
                }

                time_t date = 0;
                if (sql_type.find("TIMESTAMP")!=string::npos)
                    date = mysqlpp::Time((*it)[i]);

                if (sql_type.find("DATETIME")!=string::npos)
                    date = mysqlpp::DateTime((*it)[i]);

                if (sql_type.find(" DATE ")!=string::npos)
                    date = mysqlpp::Date((*it)[i]);

                if (date>0)
                {
                    // It is important to catch the exception thrown
                    // by Date::New in case of thread termination!
                    TryCatch exception;
                    Local<Value> val = Date::New(date*1000);
                    if (exception.HasCaught())
                        return exception.ReThrow();
                    //if (V8::IsExecutionTerminating())
                    //    return Undefined();

                    row->Set(name, val, ReadOnly);
                }
            }

            ret->Set(Integer::NewFromUnsigned(irow++), row, ReadOnly);
        }

        if (irow>0)
            ret->Set(String::New("cols"), cols, ReadOnly);

        return handle_scope.Close(ret);
    }
    catch (const exception &e)
    {
        return ThrowException(String::New(e.what()));
    }
#endif
}

Handle<Value> InterpreterV8::FuncDatabase(const Arguments &args)
{
    if (args.Length()!=1)
        return ThrowException(String::New("Number of arguments must be exactly 1."));

    if (!args[0]->IsString())
        return ThrowException(String::New("Argument 1 must be a string."));

    HandleScope handle_scope;

    const String::Utf8Value database(args[0]);
    const String::Utf8Value query   (args[1]);

#ifdef HAVE_SQL
    try
    {
        Database *db = new Database(*database);
        fDatabases.push_back(db);

        Handle<ObjectTemplate> tem = ObjectTemplate::New();
        tem->Set(String::New("user"),     String::New(db->user.c_str()), ReadOnly);
        tem->Set(String::New("server"),   String::New(db->server.c_str()), ReadOnly);
        tem->Set(String::New("database"), String::New(db->db.c_str()), ReadOnly);
        tem->Set(String::New("port"),     db->port==0?Undefined():Integer::NewFromUnsigned(db->port), ReadOnly);
        tem->Set(String::New("query"),    FunctionTemplate::New(WrapDbQuery), ReadOnly);
        tem->Set(String::New("close"),    FunctionTemplate::New(WrapDbClose),   ReadOnly);
        tem->SetInternalFieldCount(1);

        Handle<Object> obj = tem->NewInstance();
        obj->SetInternalField(0, External::New(db));

        return handle_scope.Close(obj);
    }
    catch (const exception &e)
    {
        return ThrowException(String::New(e.what()));
    }
#endif
}

Handle<Value> InterpreterV8::Convert(char type, const char* &ptr)
{
    // Dim values are always unsigned per (FACT++) definition
    switch (type)
    {
    case 'F':
        {
            // Remove the "imprecision" effect coming from casting a float to
            // a double and then showing it with double precision
            ostringstream val;
            val << setprecision(7) << *reinterpret_cast<const float*>(ptr);
            Handle<Value> v=Number::New(stod(val.str()));
            ptr+=4;
            return v;
        }
    case 'D':  { Handle<Value> v=Number::New(*reinterpret_cast<const double*>(ptr)); ptr+=8; return v; }
    case 'I':
    case 'L':  { Handle<Value> v=Integer::NewFromUnsigned(*reinterpret_cast<const uint32_t*>(ptr)); ptr += 4; return v; }
    case 'X':
        {
            const uint64_t val = *reinterpret_cast<const uint64_t*>(ptr);
            ptr += 8;
            if (val>UINT32_MAX)
                return Number::New(val);
            return Integer::NewFromUnsigned(val);
        }
    case 'S':  { Handle<Value> v=Integer::NewFromUnsigned(*reinterpret_cast<const uint16_t*>(ptr)); ptr += 2; return v; }
    case 'C':  { Handle<Value> v=Integer::NewFromUnsigned((uint16_t)*reinterpret_cast<const uint8_t*>(ptr));  ptr += 1; return v; }
    case ':':  { Handle<Value> v=String::New(ptr); return v; }
    }
    return Undefined();
}

Handle<Value> InterpreterV8::FuncClose(const Arguments &args)
{
    const HandleScope handle_scope;

    //const void *ptr = Local<External>::Cast(args.Holder()->GetInternalField(0))->Value();

    const String::Utf8Value str(args.Holder()->Get(String::New("name")));

    const auto it = fReverseMap.find(*str);
    if (it!=fReverseMap.end())
    {
        it->second.Dispose();
        fReverseMap.erase(it);
    }

    args.Holder()->Set(String::New("isOpen"), Boolean::New(false), ReadOnly);

    return Boolean::New(JsUnsubscribe(*str));
}

Handle<Value> InterpreterV8::ConvertEvent(const EventImp *evt, uint64_t counter, const char *str)
{
    Local<Value> date;

    // It is important to catch the exception thrown
    // by Date::New in case of thread termination!
    {
        TryCatch exception;
        date = Date::New(evt->GetJavaDate());
        if (exception.HasCaught())
            return exception.ReThrow();
    }

    const vector<Description> vec = JsDescription(str);

    Handle<Array> ret = Array::New();
    ret->Set(String::New("name"),    String::New(str),             ReadOnly);
    ret->Set(String::New("format"),  String::New(evt->GetFormat().c_str()), ReadOnly);
    ret->Set(String::New("named"),   Boolean::New(vec.size()>0),    ReadOnly);
    ret->Set(String::New("qos"),     Integer::New(evt->GetQoS()),   ReadOnly);
    ret->Set(String::New("size"),    Integer::New(evt->GetSize()),  ReadOnly);
    ret->Set(String::New("counter"), Integer::New(counter),         ReadOnly);
    ret->Set(String::New("time"),    date,                          ReadOnly);

    typedef boost::char_separator<char> separator;
    const boost::tokenizer<separator> tokenizer(evt->GetFormat(), separator(";:"));

    const vector<string> tok(tokenizer.begin(), tokenizer.end());

    Handle<Array> obj = tok.size()==1 ? ret : Array::New();

    const char *ptr = evt->GetText();
    try
    {
        size_t pos = 1;
        for (auto it=tok.begin(); it!=tok.end(); it++, pos++)
        {
            if (ptr>=evt->GetText())
                return ret;

            char type = (*it)[0];
            it++;

            if (it==tok.end() && type=='C')
                type = ':';

            if (it==tok.end() && type!=':')
                return Exception::Error(String::New(("Format string invalid '"+evt->GetFormat()+"'").c_str()));

            string name = pos<vec.size() ? vec[pos].name : "";
            if (tok.size()==1)
                name = "data";

            const uint32_t cnt = it==tok.end() ? 1 : stoi(it->c_str());

            if (cnt==1)
            {
                const Handle<Value> v = Convert(type, ptr);
                if (tok.size()>1)
                    obj->Set(pos-1, v);
                if (!name.empty())
                    obj->Set(String::New(name.c_str()), v);
            }
            else
            {
                Handle<Array> a = Array::New(cnt);
                for (uint32_t i=0; i<cnt; i++)
                    a->Set(i, Convert(type, ptr));
                if (tok.size()>1)
                    obj->Set(pos-1, a);
                if (!name.empty())
                    obj->Set(String::New(name.c_str()), a);
            }

            if (it==tok.end())
                break;
        }

        if (tok.size()>1)
            ret->Set(String::New("data"), obj, ReadOnly);

        return ret;
    }
    catch (...)
    {
        return Exception::Error(String::New(("Format string conversion '"+evt->GetFormat()+"' failed.").c_str()));
    }
}

Handle<Value> InterpreterV8::FuncGetData(const Arguments &args)
{
    HandleScope handle_scope;

    const String::Utf8Value str(args.Holder()->Get(String::New("name")));

    const pair<uint64_t, EventImp *> p = JsGetEvent(*str);

    const EventImp *evt = p.second;
    if (!evt)
        return Undefined();

    //if (counter==cnt)
    //    return info.Holder();//Holder()->Get(String::New("data"));

    Handle<Value> ret = ConvertEvent(evt, p.first, *str);
    return ret->IsNativeError() ? ThrowException(ret) : handle_scope.Close(ret);
}

// This is a callback from the RemoteControl piping event handling
// to the java script ---> in test phase!
void InterpreterV8::JsHandleEvent(const EventImp &evt, uint64_t cnt, const string &service)
{
    if (fThreadId<0)
        return;

    Locker locker;

    const auto it = fReverseMap.find(service);
    if (it==fReverseMap.end())
        return;

    const HandleScope handle_scope;

    Handle<Object> obj = it->second;
    if (obj.IsEmpty())
        return;

    const Handle<String> onchange = String::New("onchange");
    if (!obj->Has(onchange))
        return;

    const Handle<Value> val = obj->Get(onchange);
    if (!val->IsFunction())
        return;

    // -------------------------------------------------------------------
    // We are not in a context... we need to get into one for Array::New

    Persistent<Context> context = Context::New();
    if (context.IsEmpty())
        return;

    const Context::Scope scope(context);

    // -------------------------------------------------------------------

    TryCatch exception;

    Handle<Value> ret = ConvertEvent(&evt, cnt, service.c_str());
    if (ret->IsArray())
    {
        Handle<Array> data = Handle<Array>::Cast(ret);
        Handle<Value> args[] = { data };

        Handle<Function>::Cast(val)->Call(obj, 1, args);
    }

    if (exception.HasCaught())
        ReportException(&exception);

    if (ret->IsNativeError())
        JsException(service+".onchange callback - "+*String::Utf8Value(ret));

    context.Dispose();

    if (ret->IsUndefined() || ret->IsNativeError() || exception.HasCaught())
        V8::TerminateExecution(fThreadId);
}

Handle<Value> InterpreterV8::OnChangeSet(Local<String> prop, Local< Value > value, const AccessorInfo &)
{
    const HandleScope handle_scope;

    // Returns the value if the setter intercepts the request. Otherwise, returns an empty handle.
    const string server = *String::Utf8Value(prop);
    auto it = fStateCallbacks.find(server);

    if (it!=fStateCallbacks.end())
    {
        it->second.Dispose();
        fStateCallbacks.erase(it);
    }

    if (value->IsFunction())
        fStateCallbacks[server] = Persistent<Value>::New(value);

    return Handle<Value>();
}


void InterpreterV8::JsHandleState(const std::string &server, const State &state)
{
    if (fThreadId<0)
        return;

    Locker locker;

    auto it = fStateCallbacks.find(server);
    if (it==fStateCallbacks.end())
    {
        it = fStateCallbacks.find("*");
        if (it==fStateCallbacks.end())
            return;
    }

    const HandleScope handle_scope;

    if (it->second.IsEmpty() || !it->second->IsFunction())
        return;

    // -------------------------------------------------------------------
    // We are not in a context... we need to get into one for Array::New

    Persistent<Context> context = Context::New();
    if (context.IsEmpty())
        return;

    const Context::Scope scope(context);

    // -------------------------------------------------------------------

    TryCatch exception;

    // It is important to catch the exception thrown
    // by Date::New in case of thread termination!
    Local<Value> date = Date::New(state.time.JavaDate());

    if (!exception.HasCaught())
    {
        Handle<ObjectTemplate> obj = ObjectTemplate::New();
        obj->Set(String::New("index"),   state.index<=-256?Undefined():Integer::New(state.index),          ReadOnly);
        obj->Set(String::New("name"),    state.index<=-256?Undefined():String::New(state.name.c_str()),    ReadOnly);
        obj->Set(String::New("comment"), state.index<=-256?Undefined():String::New(state.comment.c_str()), ReadOnly);
        obj->Set(String::New("server"),  String::New(server.c_str()), ReadOnly);
        if (state.index>-256)
            obj->Set(String::New("time"), date);

        Handle<Value> args[] = { obj->NewInstance() };

        Handle<Function> fun = Handle<Function>::Cast(it->second);
        fun->Call(fun, 1, args);
    }

    if (exception.HasCaught())
        ReportException(&exception);

    context.Dispose();

    if (exception.HasCaught())
        V8::TerminateExecution(fThreadId);
}

/*
void Cleanup( Persistent<Value> object, void *parameter )
{
    cout << "======================> RemoveMyObj()" << endl;
}*/

Handle<Value> InterpreterV8::FuncSubscribe(const Arguments &args)
{
    if (args.Length()!=1)
        return ThrowException(String::New("Number of arguments must be exactly 1."));

    if (!args[0]->IsString())
        return ThrowException(String::New("Argument 1 must be a string."));

    //if (!args.IsConstructCall())
    //    return ThrowException(String::New("Must be used as constructor."));

    HandleScope handle_scope;

    const String::Utf8Value str(args[0]);

    void *ptr = JsSubscribe(*str);
    if (ptr==0)
        return ThrowException(String::New(("Subscription to '"+string(*str)+"' already exists.").c_str()));

    Handle<ObjectTemplate> tem = ObjectTemplate::New();
    tem->Set(String::New("get"),    FunctionTemplate::New(WrapGetData), ReadOnly);
    tem->Set(String::New("close"),  FunctionTemplate::New(WrapClose),   ReadOnly);
    tem->Set(String::New("name"),   String::New(*str), ReadOnly);
    tem->Set(String::New("isOpen"), Boolean::New(true));
    tem->SetInternalFieldCount(1);
    //tem->Set(String::New("toString"), String::New(("[object Dim "+string(*str)+"]").c_str()), ReadOnly);

    Handle<Object> obj = tem->NewInstance();
    obj->SetInternalField(0, External::New(ptr));

    fReverseMap[*str] = Persistent<Object>::New(obj);

    return handle_scope.Close(obj);

    // Persistent<Object> p = Persistent<Object>::New(obj->NewInstance());
    // obj.MakeWeak((void*)1, Cleanup);
    // return obj;
}

bool InterpreterV8::JsRun(const string &filename, const map<string, string> &map)
{
    Locker locker;
    fThreadId = V8::GetCurrentThreadId();

    JsLoad(filename);

    HandleScope handle_scope;

    // Create a template for the global object.
    Handle<ObjectTemplate> dim = ObjectTemplate::New();
    dim->Set(String::New("print"),     FunctionTemplate::New(WrapPrint),     ReadOnly);
    dim->Set(String::New("alarm"),     FunctionTemplate::New(WrapAlarm),     ReadOnly);
    dim->Set(String::New("out"),       FunctionTemplate::New(WrapOut),       ReadOnly);
    dim->Set(String::New("wait"),      FunctionTemplate::New(WrapWait),      ReadOnly);
    dim->Set(String::New("send"),      FunctionTemplate::New(WrapSend),      ReadOnly);
    dim->Set(String::New("state"),     FunctionTemplate::New(WrapState),     ReadOnly);
    dim->Set(String::New("newState"),  FunctionTemplate::New(WrapNewState),  ReadOnly);
    dim->Set(String::New("setState"),  FunctionTemplate::New(WrapSetState),  ReadOnly);
    dim->Set(String::New("sleep"),     FunctionTemplate::New(WrapSleep),     ReadOnly);
    dim->Set(String::New("subscribe"), FunctionTemplate::New(WrapSubscribe), ReadOnly);
    dim->Set(String::New("database"),  FunctionTemplate::New(WrapDatabase),  ReadOnly);

    Handle<ObjectTemplate> onchange = ObjectTemplate::New();
    onchange->SetNamedPropertyHandler(OnChangeGet, WrapOnChangeSet);
    dim->Set(v8::String::New("onchange"), onchange);

    Handle<ObjectTemplate> global = ObjectTemplate::New();
    global->Set(String::New("dim"), dim, ReadOnly);
    global->Set(String::New("include"), FunctionTemplate::New(WrapInclude),                ReadOnly);
    global->Set(String::New("exit"),    FunctionTemplate::New(WrapExit),                   ReadOnly);
    global->Set(String::New("version"), FunctionTemplate::New(InterpreterV8::FuncVersion), ReadOnly);

    // Persistent
    Persistent<Context> context = Context::New(NULL, global);
    if (context.IsEmpty())
    {
        //printf("Error creating context\n");
        return false;
    }

    Context::Scope scope(context);

    Handle<Array> args = Array::New(map.size());
    for (auto it=map.begin(); it!=map.end(); it++)
        args->Set(String::New(it->first.c_str()), String::New(it->second.c_str()));
    context->Global()->Set(String::New("$"),   args, ReadOnly);
    context->Global()->Set(String::New("arg"), args, ReadOnly);

    JsStart(filename);

    //context->Enter();
    Locker::StartPreemption(10);
    const bool rc = ExecuteFile(filename);

    // -----
    // This is how an exit handler could look like, but there is no way to interrupt it
    // -----
    // Handle<Object> obj = Handle<Object>::Cast(context->Global()->Get(String::New("dim")));
    // if (!obj.IsEmpty())
    // {
    //     Handle<Value> onexit = obj->Get(String::New("onexit"));
    //     if (!onexit->IsUndefined())
    //         Handle<Function>::Cast(onexit)->NewInstance(0, NULL); // argc, argv
    //     // Handle<Object> result = Handle<Function>::Cast(onexit)->NewInstance(0, NULL); // argc, argv
    // }

    Locker::StopPreemption();
    //context->Exit();

    for (auto it=fStateCallbacks.begin(); it!=fStateCallbacks.end(); it++)
        it->second.Dispose();
    fStateCallbacks.clear();

    for (auto it=fReverseMap.begin(); it!=fReverseMap.end(); it++)
        it->second.Dispose();
    fReverseMap.clear();

    context.Dispose();

#ifdef HAVE_SQL
    for (auto it=fDatabases.begin(); it!=fDatabases.end(); it++)
        delete *it;
    fDatabases.clear();
#endif

    JsEnd(filename);

    return rc;
}

void InterpreterV8::JsStop()
{
    Locker locker;

    //cout << "Terminate " << fThreadId << endl;
    if (This->fThreadId>=0)
        V8::TerminateExecution(This->fThreadId);
    //cout << "Terminate " << fThreadId << endl;

    //Unlocker unlocker;
}

#endif
