#include "MonteCarlo.h"

MonteCarlo::MonteCarlo()
{
    InitVariables();

    return;
}

MonteCarlo::MonteCarlo(
        TString filename
        )
{
    InitVariables();

    mFileName   = filename;

    OpenRootFile();
    if ( !mRootFileOpend ) return;

    LoadEventTree(  "Events");
    LoadHeaderTree( "RunHeaders");

    ReadRunHeader();

    return;
}

MonteCarlo::MonteCarlo(
        TString filename,
        TString evtsTreeName
        )
{
    InitVariables();

    mFileName   = filename;

    OpenRootFile();
    if ( !mRootFileOpend ) return;

    LoadEventTree(  evtsTreeName);
    LoadHeaderTree( "RunHeaders");

    ReadRunHeader();

    return;
}

MonteCarlo::MonteCarlo(
        TString filename,
        TString evtsTreeName,
        TString headerTreeName
        )
{
    InitVariables();

    mFileName   = filename;

    OpenRootFile();
    if ( !mRootFileOpend ) return;

    LoadEventTree(  evtsTreeName);
    LoadHeaderTree( headerTreeName);

    ReadRunHeader();

    return;
}

MonteCarlo::~MonteCarlo(void)
{
    delete[] mpPixel;

    CloseRootFile();

    delete mpRootFile;

    return;
}

void
MonteCarlo::InitVariables()
{
    mFileName       = "";
    mSeparator      = " ";

    // source file
    mpRootFile      = NULL;
    mRootFileOpend  = false;

    // Trees
    mpEventTree     = NULL;
    mpHeaderTree    = NULL;


    //header data types
    mpIntendedPulsePos      = NULL;
    mpMcRunHeader           = NULL;
    mpGeomCam               = NULL;
    mpRawRunHeader          = NULL;
    mpCorsikaRunHeader      = NULL;

    //Evt data types
    mpElectronicNoise       = NULL;
    mpRawEventData          = NULL;
    mpIncidentAngle         = NULL;
    mpMcEventMetaData       = NULL;
    mpRawEventHeader        = NULL;
    mpCorsikaEvtHeader      = NULL;
    //

    // containers
    mpPixel         = NULL;
    mpSamples       = NULL;
    mEventNumber    = 0;
    mNumberOfEvents = 2;

    mVerbosityLvl   = 3;

    return;
}

void
MonteCarlo::SetVerbosityLevel(int verbLvl)
{
    mVerbosityLvl = verbLvl;

    return;
}

int
MonteCarlo::GetVerbosityLevel()
{
    return mVerbosityLvl;
}

void
MonteCarlo::OpenRootFile()
{
    if (mVerbosityLvl > 0) cout << "...opening root file: " << mFileName << endl;

    mpRootFile = new TFile(mFileName, "READ");

    if (!mpRootFile->IsOpen())
    {
        cout << "ERROR - Could not find file " << mFileName << endl;
        mRootFileOpend =false;
        return ;
    }
    mRootFileOpend = true;

    return;
}

void
MonteCarlo::CloseRootFile()
{
    if (mVerbosityLvl > 0) cout << "...closing root file: " << mFileName << endl;
    mpRootFile->Close("R");
    mpRootFile=NULL;

    return;
}

void
MonteCarlo::OpenCsvFile(TString fileName)
{
    mCsvFileName    = fileName;

    if (mVerbosityLvl > 0) cout << "...opening csv file: " << mCsvFileName << endl;

    mCsv.open( mCsvFileName );

    return;
}

void
MonteCarlo::CloseCsvFile()
{
    if (mVerbosityLvl > 0) cout << "...closing csv file: " << mCsvFileName << endl;

    mCsv.close();

    return;
}

void
MonteCarlo::LoadHeaderTree(TString treeName)
{
    if (mVerbosityLvl > 0) cout << "...loading run header tree" << endl;

    mpHeaderTree       = (TTree*)mpRootFile->Get(treeName);

    if (mpHeaderTree->IsZombie())
    {
        cout << "...could not load tree" << endl;
        return;
    }

    // =======================================================================
    //Set Adresses to Branches in RunHeader-Tree

    // -----------------------------------------------------------------------

    // MGeomCam
    if ( mpHeaderTree->GetBranchStatus("MGeomCam.") )
    {
        if (mVerbosityLvl > 1) cout << "   ...MGeomCam" << endl;
        mpHeaderTree->SetBranchAddress("MGeomCam.",         &mpGeomCam);
    }
    else cout << "...could not load branch: MGeomCam" << endl;

    // -----------------------------------------------------------------------

    // IntendedPulsePos
    if ( mpHeaderTree->GetBranchStatus("IntendedPulsePos.") )
    {
        if (mVerbosityLvl > 1) cout << "   ...IntendedPulsePos" << endl;
        mpHeaderTree->SetBranchAddress("IntendedPulsePos.", &mpIntendedPulsePos);
    }
    else cout << "...could not load branch: IntendedPulsePos" << endl;

    // -----------------------------------------------------------------------

    // MMcRunHeader
    if ( mpHeaderTree->GetBranchStatus("MMcRunHeader.") )
    {
        if (mVerbosityLvl > 1) cout << "   ...MMcRunHeader" << endl;
        mpHeaderTree->SetBranchAddress("MMcRunHeader.",     &mpMcRunHeader);
    }
    else cout << "...could not load branch: MMcRunHeader" << endl;

    // -----------------------------------------------------------------------

    // ElectronicNoise
    if ( mpHeaderTree->GetBranchStatus("ElectronicNoise.") )
    {
        if (mVerbosityLvl > 1) cout << "   ...ElectronicNoise" << endl;
        mpHeaderTree->SetBranchAddress("ElectronicNoise.",  &mpElectronicNoise);
    }
    else cout << "...could not load branch: ElectronicNoise" << endl;

    // -----------------------------------------------------------------------

    // MRawRunHeader
    if ( mpHeaderTree->GetBranchStatus("MRawRunHeader.") )
    {
        if (mVerbosityLvl > 1) cout << "   ...MRawRunHeader" << endl;
        mpHeaderTree->SetBranchAddress("MRawRunHeader.",    &mpRawRunHeader);
    }
    else cout << "...could not load branch: MRawRunHeader" << endl;

    // -----------------------------------------------------------------------

    // MCorsikaRunHeader
    if ( mpHeaderTree->GetBranchStatus("MCorsikaRunHeader.") )
    {
        if (mVerbosityLvl > 1) cout << "   ...MCorsikaRunHeader" << endl;
        mpHeaderTree->SetBranchAddress("MCorsikaRunHeader.",&mpCorsikaRunHeader);
    }
    else cout << "...could not load branch: MCorsikaRunHeader" << endl;

    // =======================================================================

    return;
}

void
MonteCarlo::ReadRunHeader()
{
    if (mVerbosityLvl > 0)
        cout << "...reading run header " << mpHeaderTree << endl;

    //Read Values from RunHeader-Tree

    mpHeaderTree->GetEntry();    //This causes problems

    // -----------------------------------------------------------------------

    //Get values from "MGeomCam" Branch
        //Getter functions from "MGeomCamFACT.h"
    if ( mpGeomCam != NULL)
    {
        mCamDist                = mpGeomCam->GetCameraDist();
        mNumberOfPixels         = mpGeomCam->GetNumPixels();
        mNumberOfSectors        = mpGeomCam->GetNumSectors();
        mNumberOfAreas          = mpGeomCam->GetNumAreas();
    }
    else if (mVerbosityLvl > 2)
        cout << "...could not read data from: MGeomCam" << endl;

    // -----------------------------------------------------------------------

    //Get values from "IntendedPulsePos" Branch
    if ( mpIntendedPulsePos != NULL)
    {
        mIntendedPulsePos       = mpIntendedPulsePos->GetVal();
    }
    else if (mVerbosityLvl > 2)
        cout << "...could not read data from: IntendedPulsePos" << endl;

    // -----------------------------------------------------------------------

    //Get values from "MMcRunHeader" Branch from event Tree
    if ( mpMcRunHeader != NULL)
    {
        mNumSimulatedShowers    = mpMcRunHeader->GetNumSimulatedShowers();
    }
    else if (mVerbosityLvl > 2)
        cout << "...could not read data from: MMcRunHeader" << endl;

    // -----------------------------------------------------------------------

    //Get number of Events from event Tree
    if ( mpEventTree != NULL)
    {
        mNumberOfEntries        = mpEventTree->GetEntries();
    }
    else if (mVerbosityLvl > 2)
        cout << "...could not read number of Events from event Tree" << endl;

    if (mVerbosityLvl > 0)
        cout << "Event Tree has " << mNumberOfEntries << " entries" << endl;

    // -----------------------------------------------------------------------

    //Get values from "MRawRunHeader" Branch
    if ( mpRawRunHeader != NULL)
    {
        mNumberOfEvents         = mpRawRunHeader->GetNumEvents(); // couses problems
        mNumberOfEventsRead     = mpRawRunHeader->GetNumEventsRead();
        mSamplingFrequency      = mpRawRunHeader->GetFreqSampling();
        mSourceName             = mpRawRunHeader->GetSourceName();
        mFileNumber             = mpRawRunHeader->GetFileNumber();
        mNumberOfSamples        = mpRawRunHeader->GetNumSamplesHiGain();
        mNumberOfBytes          = mpRawRunHeader->GetNumBytesPerSample();
        mRunNumber              = mpRawRunHeader->GetRunNumber();
        mRunType                = mpRawRunHeader->GetRunType();
    }
    else if (mVerbosityLvl > 2)
        cout << "...could not read data from: MRawRunHeader" << endl;

    // -----------------------------------------------------------------------

    //Get values from "MCorsikaRunHeader" Branch
    if ( mpCorsikaRunHeader != NULL)
    {
        mSlopeSpectrum          = mpCorsikaRunHeader->GetSlopeSpectrum();
        mEnergyMin              = mpCorsikaRunHeader->GetEnergyMin();
        mEnergyMax              = mpCorsikaRunHeader->GetEnergyMax();
        mZdMin                  = mpCorsikaRunHeader->GetZdMin();
        mZdMax                  = mpCorsikaRunHeader->GetZdMax();
        mAzMin                  = mpCorsikaRunHeader->GetAzMin();
        mAzMax                  = mpCorsikaRunHeader->GetAzMax();
    }
    else if (mVerbosityLvl > 2)
        cout << "...could not read data from: MCorsikaRunHeader" << endl;

    // -----------------------------------------------------------------------

    // delete Pixel Array before you set refferences for a new one
    if (mpPixel != NULL)
    {
        delete[] mpPixel;
    }
    mpPixel = new pixel_t[mNumberOfPixels];

    // -----------------------------------------------------------------------

    //Get Pedestal from "ElectronicNoise" Branch
    if ( mpElectronicNoise != NULL)
    {
        if (mVerbosityLvl > 1) cout << "   ...reading pedestal offsets" << endl;

        // Read Pedestal Offset
        for ( int i = 0; i < mNumberOfPixels; i++ )
        {
            if ( &(mpElectronicNoise[0][i]) != NULL)
                mpPixel[i].pedestal   = mpElectronicNoise[0][i].GetPedestal();
            else if (mVerbosityLvl > 2)
                cout << "   ...cannot read pedestal offset" << endl;
        }
    }
    else if (mVerbosityLvl > 2)
        cout << "...could not read data from: ElectronicNoise" << endl;

    return;
}

void
MonteCarlo::LoadEventTree(TString treeName)
{
    if (mVerbosityLvl > 0) cout << "...loading event tree" << endl;

    mpEventTree       = (TTree*)mpRootFile->Get(treeName);

    if (mpEventTree->IsZombie())
    {
        cout << "...could not load tree" << endl;
        return;
    }

    // =======================================================================
    //Set Adresses to Branches in Events-Tree

    if (mVerbosityLvl > 1) cout << "...SetBranchAddresses:" << endl;

    // -----------------------------------------------------------------------

    // MRawEvtData
    if ( mpEventTree->GetBranchStatus("MRawEvtData.") != -1 )
    {
        if (mVerbosityLvl > 1) cout << "   ...MRawEvtData" << endl;
        mpEventTree->SetBranchAddress("MRawEvtData.",       &mpRawEventData);
    }
    else cout << "...could not load branch: MRawEvtData" << endl;

    // -----------------------------------------------------------------------

    // IncidentAngle
    if ( mpEventTree->GetBranchStatus("IncidentAngle.") )
    {
        //FIX ME: THIS VALUE IS NOT EXISTANT IN EVERY MC FILE

        if (mVerbosityLvl > 1) cout << "   ...IncidentAngle" << endl;
        mpEventTree->SetBranchAddress("IncidentAngle.",     &mpIncidentAngle);
    }
    else cout << "...could not load branch: IncidentAngle" << endl;

    // -----------------------------------------------------------------------

    // MMcEvt
    if ( mpEventTree->GetBranchStatus("MMcEvt.") )
    {
        if (mVerbosityLvl > 1) cout << "   ...McEvt" << endl;
        mpEventTree->SetBranchAddress("MMcEvt.",            &mpMcEventMetaData);
    }
    else cout << "...could not load branch: McEvt" << endl;

    // -----------------------------------------------------------------------

    // MRawEvtHeader
    if ( mpEventTree->GetBranchStatus("MRawEvtHeader.") )
    {
        if (mVerbosityLvl > 1) cout << "   ...MRawEventHeader" << endl;
        mpEventTree->SetBranchAddress("MRawEvtHeader.",     &mpRawEventHeader);
    }
    else cout << "...could not load branch: MRawEvtHeader" << endl;

    // -----------------------------------------------------------------------

    // MCorsikaEvtHeader
    if ( mpEventTree->GetBranchStatus("MCorsikaEvtHeader.") )
    {
        if (mVerbosityLvl > 1) cout << "   ...MCorsikaEvtHeader" << endl;
        mpEventTree->SetBranchAddress("MCorsikaEvtHeader.", &mpCorsikaEvtHeader);
    }
    else cout << "...could not load branch: MCorsikaEvtHeader" << endl;

    // =======================================================================

    return;
}

void
MonteCarlo::ReadEventMetaData()
{
    if (mVerbosityLvl > 1)
        cout << "...reading event header" << endl;

    //Get values from "MGeomCamFACT" Branch
    if ( mpIncidentAngle != NULL)
    {
        mIncidentAngle          = mpIncidentAngle->GetVal();
    }
    else if (mVerbosityLvl > 2)
        cout << "...could not read data from: MGeomCamFACT" << endl;

    // -----------------------------------------------------------------------

    //Get values from "MMcEvt" Branch
    if ( mpMcEventMetaData != NULL)
    {
        //The following Getter Functions can be found in MMcEvt.h
        mCorsikaEventNumber     = mpMcEventMetaData->GetEvtNumber();
        mPhotElFromShower       = mpMcEventMetaData->GetPhotElfromShower();
        mPhotElinCamera         = mpMcEventMetaData->GetPhotElinCamera();
            //The following Getter Functions can be found in MMcEvtBasic.h
        mPartId                 = mpMcEventMetaData->GetPartId();
        mPartName               = mpMcEventMetaData->GetParticleName(mPartId);
        mPartSymbol             = mpMcEventMetaData->GetParticleSymbol(mPartId);
        mEnergy                 = mpMcEventMetaData->GetEnergy();
        mImpact                 = mpMcEventMetaData->GetImpact();
        mTelescopePhi           = mpMcEventMetaData->GetTelescopePhi();
        mTelescopeTheta         = mpMcEventMetaData->GetTelescopeTheta();
        mPhi                    = mpMcEventMetaData->GetParticlePhi();
        mTheta                  = mpMcEventMetaData->GetParticleTheta();
    }
    else if (mVerbosityLvl > 2)
        cout << "...could not read data from: MMcEvt" << endl;

    // -----------------------------------------------------------------------

    //Get values from "MRawEventHeader" Branch
    if ( mpRawEventHeader != NULL)
    {
        mEventNumber            = mpRawEventHeader->GetDAQEvtNumber();
        mNumTriggerLvl1         = mpRawEventHeader->GetNumTrigLvl1();
        mNumTriggerLvl2         = mpRawEventHeader->GetNumTrigLvl2();
    }
    else if (mVerbosityLvl > 2)
        cout << "...could not read data from: MRawEventHeader" << endl;

    // -----------------------------------------------------------------------

    //Get values from "MCorsikaEvtHeader" Branch
    if ( mpCorsikaEvtHeader != NULL)
    {
        mFirstInteractionHeight = mpCorsikaEvtHeader->GetFirstInteractionHeight();
        mEvtReuse               = mpCorsikaEvtHeader->GetNumReuse();
        mMomentumX              = mpCorsikaEvtHeader->GetMomentum().X();
        mMomentumY              = mpCorsikaEvtHeader->GetMomentum().Y();
        mMomentumZ              = mpCorsikaEvtHeader->GetMomentum().Z();
        mZd                     = mpCorsikaEvtHeader->GetZd();
        mAz                     = mpCorsikaEvtHeader->GetAz();
        mX                      = mpCorsikaEvtHeader->GetX();
        mY                      = mpCorsikaEvtHeader->GetY();
    }
    else if (mVerbosityLvl > 2)
        cout << "...could not read data from: MCorsikaEvtHeader" << endl;

    // -----------------------------------------------------------------------

    // mWeightedNumPhotons     = mpCorsikaEvtHeader->Get;
    // no getter function, no idea how to extract information

    return;
}

void
MonteCarlo::ReadEventRawData()
{
    if (mVerbosityLvl > 1) cout << "...reading event raw data" << endl;

    // -----------------------------------------------------------------------

    // delete Pixel Array before you set refferences for a new one
    if (mpPixel != NULL)
    {
        delete[] mpPixel;
    }
    mpPixel = new pixel_t[mNumberOfPixels];

    // -----------------------------------------------------------------------

    if ( mpRawEventData == NULL)
    {
        cout << "ERROR: cannot read event data" << endl;
        return;
    }

    mpRawEventData->InitRead(mpRawRunHeader);

    int pix_first_sample;
//    int pix_last_sample;

    // -----------------------------------------------------------------------

    unsigned short* all_raw_data    = NULL;

    if ( mpRawEventData != NULL)
    {
        all_raw_data    = (unsigned short*) mpRawEventData->GetSamples();
    }
    else cout << "...cannot read event raw data" << endl;

    // -----------------------------------------------------------------------

    if (mVerbosityLvl > 1)
        cout << "...pixel progress: ";

    for ( int i = 0; i < mNumberOfPixels; i++ )
    {
        if (mVerbosityLvl > 1){
            if ( !(i%(mNumberOfPixels/10) ))
                cout << i/(mNumberOfPixels*1.0)*100 << "%.." ;
            if ( i == mNumberOfPixels-1)
                cout << "100% " ;
        }

        // Read Pedestal Offset
        if ( mpElectronicNoise != NULL)
        {
            mpPixel[i].pedestal   = mpElectronicNoise[0][i].GetPedestal();
        }
        else cout << "...cannot read pedestal offset" << endl;

        // Read Pixel SoftId
        if ( mpRawEventData != NULL)
        {
            mpPixel[i].SoftId     = mpRawEventData->GetPixelIds()[i];
        }
        else cout << "...cannot read pixel id" << endl;

        pix_first_sample        = i*mNumberOfSamples;

        // point beginning of pixel's rawdata array to address
        // of pixel's first sample's adress
        mpPixel[i].rawData      = &(all_raw_data[pix_first_sample]);
    }

    if (mVerbosityLvl > 1)
        cout << endl;

    return;
}

void
MonteCarlo::ReadEvent(int Event)
{
    if (mVerbosityLvl > 0) cout << endl
                                << "====================" << endl
                                << "...reading Event: " << Event << endl
                                << "====================" << endl;


    //load certain event from tree
    mpEventTree->GetEntry(Event);

    ReadEventMetaData();
    ReadEventRawData();

    return;
}

void
MonteCarlo::WriteMc2Csv(TString filename)
{
    if ( !mRootFileOpend ){
        cout << "ERROR - no rootfile loaded, no data loaded, cannot write csv"
             << endl;
        return;
    }

    cout << "...writing mc to csv: " << filename << endl;
    cout << "...processing " << mNumberOfEntries << "Events" << endl;

    mCsvFileName = filename;
    OpenCsvFile(mCsvFileName);

    WriteFileInfo2Csv();
    WriteRunHeaderInfo2Csv();
    WriteRunHeader2Csv();

    WritePedestalInfo2Csv();
    WritePedestal2Csv();

    WriteEventHeaderInfo2Csv();
    WriteEventDataInfo2Csv();

//    for (int evt = 0; evt < mNumberOfEvents; evt++)
    for (int evt = 0; evt < mNumberOfEntries; evt++)
    {
        ReadEvent(evt);
        WriteEventHeader2Csv();
        WriteEventData2Csv();
    }

    cout << endl << "...conversion done " << endl;

    CloseCsvFile();
    return;
}

void
MonteCarlo::WriteFileInfo2Csv()
{
    if (mVerbosityLvl > 0) cout << "...writing file header to csv" << endl;

    mCsv << "### ==========================================================="
            << endl;
    mCsv << "### =             FACT Monte Carlo                            ="
            << endl;
    mCsv << "### ==========================================================="
            << endl;
    mCsv << "### = FileInfo:                                               "
            << endl;
    mCsv << "### =                                                         "
            << endl;
    mCsv << "### = Source Name:       " << mSourceName << endl;
    mCsv << "### = Number of Entries: " << mNumberOfEntries << endl;
    mCsv << "### = Run Number:        " << mRunNumber << endl;
    mCsv << "### = Run Type  :        " << mRunType << endl;
    mCsv << "### = File Number:       " << mFileNumber << endl;
    mCsv << "### ==========================================================="
         << endl ;
    mCsv << "###"
         << endl ;

    return;
}

void
MonteCarlo::WriteRunHeaderInfo2Csv()
{
    if (mVerbosityLvl > 0) cout << "...writing run header names to csv" << endl;

    mCsv << "### [RunHeader]" << endl;

    mCsv << "# mNumberOfEntries" << mSeparator;
    mCsv << "mIntendedPulsePos" << mSeparator;
//    Csv << "mPedestalOffset" << mSeparator;
    mCsv << "mNumSimulatedShowers" << mSeparator;
    mCsv << "mNumberOfPixels" << mSeparator;
    mCsv << "mNumberOfSamples" << mSeparator;
//    mCsv << "mSampleSize" << mSeparator;
    mCsv << "mCamDist" << mSeparator;
    mCsv << "mSourceName" << mSeparator;
    mCsv << "mSlopeSpectrum" << mSeparator;
    mCsv << "mEnergyMin" << mSeparator;
    mCsv << "mEnergyMax" << mSeparator;
    mCsv << "mZdMin" << mSeparator;
    mCsv << "mZdMax" << mSeparator;
    mCsv << "mAzMin" << mSeparator;
    mCsv << "mAzMax" << mSeparator;
    mCsv << "mFileNumber" << mSeparator;
    mCsv << "mRunnumber" << mSeparator;
    mCsv << "mRunType" << endl;

    return;
}

void
MonteCarlo::WriteRunHeader2Csv()
{
    if (mVerbosityLvl > 0) cout << "...writing run header to csv" << endl;

    mCsv << mNumberOfEntries << mSeparator;
    mCsv << mIntendedPulsePos << mSeparator;
//    mCsv << mPedestalOffset << mSeparator;
    mCsv << mNumSimulatedShowers << mSeparator;
    mCsv << mNumberOfPixels << mSeparator;
    mCsv << mNumberOfSamples << mSeparator;
//    mCsv << mSampleSize << mSeparator;
    mCsv << mCamDist << mSeparator;
    mCsv << mSourceName << mSeparator;
    mCsv << mSlopeSpectrum << mSeparator;
    mCsv << mEnergyMin << mSeparator;
    mCsv << mEnergyMax << mSeparator;
    mCsv << mZdMin << mSeparator;
    mCsv << mZdMax << mSeparator;
    mCsv << mAzMin << mSeparator;
    mCsv << mAzMax << mSeparator;
    mCsv << mFileNumber << mSeparator;
    mCsv << mRunNumber << mSeparator;
    mCsv << mRunType << endl;

    return;
}

void
MonteCarlo::WriteEventHeaderInfo2Csv()
{
    if (mVerbosityLvl > 0) cout << "...writing event header names to csv" << endl;

    mCsv << "### [EventHeader]" << endl;

    mCsv << "# mEventNumber" << mSeparator;
    mCsv << "mNumberOfBytes" << mSeparator;
    mCsv << "mIncidentAngle" << mSeparator;
    mCsv << "mPartId" << mSeparator;
    mCsv << "mEnergy" << mSeparator;
    mCsv << "mImpact" << mSeparator;
    mCsv << "mTelescopePhi" << mSeparator;
    mCsv << "mTelescopeTheta" << mSeparator;
    mCsv << "mPhi" << mSeparator;
    mCsv << "mTheta" << mSeparator;
    mCsv << "mCorsikaEventNumber" << mSeparator;
    mCsv << "mPhotElFromShower" << mSeparator;
    mCsv << "mEvtReuse" << mSeparator;
    mCsv << "mNumTriggerLvl1" << mSeparator;
    mCsv << "mFirstInteractionHeight" << mSeparator;
    mCsv << "mMomentumX" << mSeparator;
    mCsv << "mMomentumY" << mSeparator;
    mCsv << "mMomentumZ" << mSeparator;
    mCsv << "mZd" << mSeparator;
    mCsv << "mAz" << mSeparator;
    mCsv << "mX" << mSeparator;
    mCsv << "mY" << mSeparator;
//    mCsv << "mWeightedNumPhotons" ;
    mCsv << endl;

    return;
}

void
MonteCarlo::WriteEventHeader2Csv()
{
    if (mVerbosityLvl > 0) cout << "...writing event header to csv" << endl;

    mCsv << mEventNumber << mSeparator;
    mCsv << mNumberOfBytes << mSeparator;
    mCsv << mIncidentAngle << mSeparator;
    mCsv << mPartId << mSeparator;
    mCsv << mEnergy << mSeparator;
    mCsv << mImpact << mSeparator;
    mCsv << mTelescopePhi << mSeparator;
    mCsv << mTelescopeTheta << mSeparator;
    mCsv << mPhi << mSeparator;
    mCsv << mTheta << mSeparator;
    mCsv << mCorsikaEventNumber << mSeparator;
    mCsv << mPhotElFromShower << mSeparator;
    mCsv << mEvtReuse << mSeparator;
    mCsv << mNumTriggerLvl1 << mSeparator;
    mCsv << mFirstInteractionHeight << mSeparator;
    mCsv << mMomentumX << mSeparator;
    mCsv << mMomentumY << mSeparator;
    mCsv << mMomentumZ << mSeparator;
    mCsv << mZd << mSeparator;
    mCsv << mAz << mSeparator;
    mCsv << mX << mSeparator;
    mCsv << mY << mSeparator;
//    mCsv << mWeightedNumPhotons ;
    mCsv << endl;

    return;
}

void
MonteCarlo::WriteEventDataInfo2Csv()
{
    if (mVerbosityLvl > 0) cout << "...writing event categories to csv" << endl;

    mCsv << "### [RawData]" << endl;
    mCsv << "# mEventNumber" << mSeparator;
    mCsv << "pixelID" << mSeparator;
//     mCsv << "pixelPedestal" << mSeparator;
    for (int i = 0; i < mNumberOfSamples; i++)
    {
        mCsv << "Raw_" << i << mSeparator;
    }
    mCsv << endl;

    return;
}

void
MonteCarlo::WriteEventData2Csv()
{
    if (mVerbosityLvl > 0) cout << "...writing event data to csv" << endl;

    for (int i = 0; i < mNumberOfPixels; i++)
    {
        WritePixelData2Csv(i);
    }

    return;
}

void
MonteCarlo::WritePixelData2Csv(int pixelID)
{
    if (mVerbosityLvl > 3) cout << "...writing pixel data to csv" << endl;
    mCsv << mEventNumber << mSeparator;
    mCsv << mpPixel[pixelID].SoftId << mSeparator;
//    mCsv << mpPixel[pixelID].pedestal << mSeparator;

    for (int i = 0; i < mNumberOfSamples; i++)
    {
         mCsv << mpPixel[pixelID].rawData[i] << mSeparator;
    }
    mCsv << endl;

    return;
}

void
MonteCarlo::WritePedestalInfo2Csv()
{
    mCsv << "### [Pedestal]" << endl;
//    mCsv << "# SoftID" ;
//    mCsv << mSeparator ;
//    mCsv << "Pedestal" << endl;
}

void
MonteCarlo::WritePedestal2Csv()
{
    if (mVerbosityLvl > 3) cout << "...writing pedestal info to csv" << endl;

    for (int pixelID = 0; pixelID < mNumberOfPixels; pixelID++)
    {
//        mCsv << mpPixel[pixelID].SoftId;
        mCsv << mpPixel[pixelID].pedestal;
        if (pixelID < mNumberOfPixels -1)
            mCsv << mSeparator;
    }
    mCsv << endl;

    return;
}


//--------------------------------------------------------------------------------
//
//
//
//  FADC samples (hi gain) of all pixels
//  This is an array of Byte_t variables. The value of a FADC sample has a
//  size of n=fNumBytesPerSample bytes. Therefore, a FADC sample value will
//  occupy n consecutive elements of this array (little endian ordering, i.e,
//  less significant bits (and bytes) go first.
//  If m = GetNumHiGainSamples(), the n bytes corresponding to the value of the
//  i-th FADC sample of the j-th pixel are stored in the n consecutive
//  positions of this array, starting from fHiGainFadcSamples[j*n*m+i*n]




