#ifndef MONTECARLO_H
#define MONTECARLO_H

/*! \class MonteCarlo
 * \brief Monte Carlo class to handle MC-Data from mars-root-file produced with ceres.
 *
 * #include "MonteCarlo.h" <BR>
 * -llib
 *
 * Class can be used in Mars to load Monte Carlo data from a root-file
 * containing Mars classes that where produced by CERES
 *
 * @see something
 */


// SYSTEM INCLUDES
#include <vector>
#include <fstream>
#include <endian.h>
#include <stdio.h>

#include <TSystem.h>
#include <TString.h>
#include <TStyle.h>
#include <TCanvas.h>
#include <TMath.h>
#include <TFile.h>
#include <TH1.h>
#include <TH2F.h>
#include <TF1.h>
#include <TTree.h>
//

// PROJECT INCLUDES
#include "MStatusDisplay.h"
#include "MStatusArray.h"
#include "MParContainer.h"

#include "MParameters.h"
#include "MPedestalCam.h"
#include "MMcRunHeader.hxx"
#include "MGeomCamFACT.h"
#include "MRawRunHeader.h"
#include "MCorsikaRunHeader.h"

//Evt data types
#include "MRawEvtData.h"
#include "MPedestalCam.h"
#include "MMcEvt.hxx"
#include "MMcEvtBasic.h"
#include "MRawEvtHeader.h"
#include "MCorsikaEvtHeader.h"



//

// LOCAL INCLUDES
//

// FORWARD REFERENCES
//

using namespace std;
using namespace TMath;

struct pixel_t          //struct gathering information of a camera pixel
{
    int                 SoftId;         //Mars Software ID of Pixel
//    int                 ChId;         //continous Hardware ID of Pixel
    unsigned short*     rawData;        //array with raw data of slices* amplitudes
    float               pedestal;       //amplitude of baseline
};

class MonteCarlo
{
public:
// LIFECYCLE

    /** Default constructor.
    */
    MonteCarlo();                       // Default constructor
    /*! constructor
     * \param filename name of mc data file that will to load
    */
    MonteCarlo( TString filename);      // name of mc data file will to load
    /*! constructor
     * \param filename name of mc data file that will to load
     * \param evtsTreeName name of tree containg events
    */
    MonteCarlo( TString filename,       // name of mc data file will to load
                TString evtsTreeName);  // name of tree containg events
    /*! constructor
     * \param filename name of mc data file that will to load
     * \param evtsTreeName name of tree containg events
     * \param headerTreeName name of tree containg run meta data
    */
    MonteCarlo( TString filename,       // name of mc data file will to load
                TString evtsTreeName,   // name of tree containg events
                TString headerTreeName);// name of tree containg meta data


//    /** Copy constructor.
//    *
//    * @param from The value to copy to this object.
//    */
//    MonteCarlo(const MonteCarlo& from);


    /** Destructor.
    */
    ~MonteCarlo(void);


// OPERATORS

//    /** Assignment operator.
//    *
//    * @param from THe value to assign to this object.
//    *
//    * @return A reference to this object.
//    */
//    MonteCarlo&                     operator=(const XX& from);

// OPERATIONS
private:
    /*! Initiate all pointer Variables to NULL*/
    void InitVariables();

public:
    /*! Write Monte Carlo Data into a Csv-file WriteMc2Csv(filename)
     * \param filename The Filename of the CSV File to which data will be written
    */
    void WriteMc2Csv(TString filename );

    /*! write pixels' raw data for a given pixel into the given csv file
     * \param pixelID Software ID of pixel
     */
    void WritePixelData2Csv(int pixelID);

    /*! loop over all pixels and call WritePixelData2Csv
     */
    void WriteEventData2Csv();

    /*! write table headings for coulums with pixels' raw data
     */
    void WriteEventDataInfo2Csv();

    /*! write table headings for coulums with event meta information
     */
    void WriteEventHeaderInfo2Csv();

    /*! write meta information of current event to csv
     */
    void WriteEventHeader2Csv();

    /*! write table headings for coulums with run meta information
     */
    void WriteRunHeaderInfo2Csv();

    /*! write meta information of current run to csv
     */
    void WriteRunHeader2Csv();

    /*! write the header of the csv file containing brief info about the converted mc file
     */
    void WriteFileInfo2Csv();

    /*! write table headings for coulums with pixels' pedestal amplitude
     */
    void WritePedestal2Csv();

    /*! write table headings for coulums with pixels' pedestal amplitude
     */
    void WritePedestalInfo2Csv();


// ACCESS
    /*! set the verbosity level for verbosity cout on cmd-line
     * \param verbLvl Votrbosity Level
     */
    void SetVerbosityLevel(int verbLvl);

    /*! Get the verbosity level for verbosity cout on cmd-line
     */
    int  GetVerbosityLevel();

    /*! Open the mars-root-file produced with ceres
     */
    int  OpenRootFile();

    /*!  Close the mars-root-file produced with ceres
     */
    void CloseRootFile();

    /*! Open csv-file to which data will be written
     * \param fileName Filename of the csv-file e.g. "mc20120605.csv"
     */
    void OpenCsvFile(   TString fileName);

    /*! Close csv-file to which data was be written
     */
    void CloseCsvFile();

    /*! set according pointers to tree containing event information and its branches
     * \param treeName Name of TTree containing event information
     */
    void LoadEventTree( TString treeName);

    /*! set according pointers to tree containing run information and its branches
     * \param treeName Name of TTree containing run information
     */
    void LoadHeaderTree(TString treeName);

    /*! set values of members  to according leafs in TTree for run meta data
     */
    void ReadRunHeader();

    /*! set values of members  to according leafs in TTree for event raw data
     */
    void ReadEventRawData();

    /*! set values of members  to according leafs in TTree for event meta data
     */
    void ReadEventMetaData();

    /*! call ReadEventRawData() and ReadEventMetaData() for given event
     * \param Event Event ID to read data
     */
    void ReadEvent(int Event);

// INQUIRY
private:
    /*! Verbostiy Level */
    int                 mVerbosityLvl;
    /*! Whether the rootfile is open */
    bool                mRootFileOpend;

    /*! filename of output csv file */
    TString             mCsvFileName;
    /*! steam into csv output file */
    ofstream            mCsv;
    /*! filename of input root file */
    TString             mFileName;
    /*! pointer to input root file */
    TFile*              mpRootFile;
    /*! Array for pixels' raw data */
    pixel_t*            mpPixel;

    /*! pointer to event tree */
    TTree*              mpEventTree;
    /*! pointer to run header tree */
    TTree*              mpHeaderTree;

    //run header tree branches
    /*! mars class pointer to branch with same name in TTree */
    MParameterD*        mpIntendedPulsePos;
    /*! mars class pointer to branch with same name in TTree */
    MMcRunHeader*       mpMcRunHeader;
    /*! mars class pointer to branch with same name in TTree */
    MGeomCamFACT*       mpGeomCam;
    /*! mars class pointer to branch with same name in TTree */
    MRawRunHeader*      mpRawRunHeader;
    /*! mars class pointer to branch with same name in TTree */
    MCorsikaRunHeader*  mpCorsikaRunHeader;

    //Event tree branches
    /*! mars class pointer to branch with same name in TTree */
    MPedestalCam*       mpElectronicNoise;
    /*! mars class pointer to branch with same name in TTree */
    MRawEvtData*        mpRawEventData;
    /*! mars class pointer to branch with same name in TTree */
    MParameterD*        mpIncidentAngle;
    /*! mars class pointer to branch "MMcEvt" in TTree */
    MMcEvt*             mpMcEventMetaData;
    /*! mars class pointer to branch with same name in TTree */
    MRawEvtHeader*      mpRawEventHeader;
    /*! mars class pointer to branch with same name in TTree */
    MCorsikaEvtHeader*  mpCorsikaEvtHeader;

//    /*! array with MC events raw data */
//    unsigned short *    mpSamples;          // array with MC events raw data
    /*! coulumn seperator in csv file */
    TString             mSeparator;

    //Run Meta Data
    /*! number of entries(events) in event tree  */
    int                 mNumberOfEntries;
    /*! number of events in file NOT WORKING WITH CURRENT CERES VERSION */
    int                 mNumberOfEvents;
    /*! position (slice) of simulated cherenkov pulse in pixel */
    float               mIntendedPulsePos;
    /*! number of showers simulated by corsika*/
    int                 mNumSimulatedShowers;
    /*! total number of simulated camera pixels */
    int                 mNumberOfPixels;
    /*! ??? */
    int                 mNumberOfSectors;
    /*! ??? */
    int                 mNumberOfAreas;
    /*! number of Samples (slices) -> simulated Region of interest */
    float               mNumberOfSamples;
    /*! simulated sampling frequency  */
    float               mSamplingFrequency;
    /*! ??? */
    int                 mNumberOfEventsRead;
    /*! ??? */
    float               mCamDist;
    /*! name of simulated source */
    const char*         mSourceName;
    /*! slope of simulated spectrum */
    float               mSlopeSpectrum;
    /*! minimum of simulated spectrum  */
    float               mEnergyMin;
    /*! maximum of simulated spectrum */
    float               mEnergyMax;
    /*! minimum Zenidth of simulated showers */
    float               mZdMin;
    /*! maximum Zenidth of simulated showers */
    float               mZdMax;
    /*! minimum Azimuth of simulated showers */
    float               mAzMin;
    /*! maximum Azimuth of simulated showers */
    float               mAzMax;
    /*! file number from raw run header */
    unsigned int        mFileNumber;
    /*! corsika run number */
    int                 mRunNumber;
    /*! runtype: e.g. Pedestal, data, calibration run,... */
    int                 mRunType;
    /*! number of bytes per sample */
    int                 mNumberOfBytes;

    //Event Meta Data
    /*! Incident Angle */
    float               mIncidentAngle;
    /*! id of primary particle of shower */
    int                 mPartId;
    /*! energy of primary particle of shower */
    float               mEnergy;
    /*! impact parameter of primary particle of shower  */
    float               mImpact;
    /*! Telescope Phi of shower */
    float               mTelescopePhi;
    /*! Telescope Theta of shower */
    float               mTelescopeTheta;
    /*! Phi of shower */
    float               mPhi;
    /*! Theta of shower */
    float               mTheta;
    /*! name of primary particle of shower */
    TString             mPartName;
    /*! symbol of primary particle of shower */
    TString             mPartSymbol;
    /*! corsika event number */
    unsigned int        mCorsikaEventNumber;
    /*! Passed qe, coming from the shower */
    unsigned int        mPhotElFromShower;
    /*! usPhotElfromShower + mean of phe from NSB */
    unsigned int        mPhotElinCamera;
    /*! Number running from 0 to N-1, being N the number
     * of times a Corsika event has been reused, by
     * orienting the telescope in different ways or by
     * setting it at a different location on the ground. */
    int                 mEvtReuse;
    /*! Number of Events from DAQ */
    int                 mEventNumber;
    /*! Number of 1st level tiggers between 2 events Used in MSimTrigger for the index of the trigger channel */
    float               mNumTriggerLvl1;
    /*! Number of 2nd level tiggers between 2 events */
    float               mNumTriggerLvl2;
    /*! [cm] z coordinate, first intercation height */
    float               mFirstInteractionHeight;
    /*! [GeV/c] "+west"    "-east" */
    float               mMomentumX;
    /*! [GeV/c] "+south"   "-north"
     * (north denotes the magnet north which is defined to be in the geografic north!)
     */
    float               mMomentumY;
    /*! [GeV/c] "+upwards" "-downwards" */
    float               mMomentumZ;
    /*! [rad] Zenith distance */
    float               mZd;
    /*! [rad] Azimuth (north=0; east=90)
     * (north denotes the magnet north which is defined to be in the geografic north!)
    */
    float               mAz;
    /*! [cm] Position of telescope on ground x / - impact parameter x */
    float               mX;
    /*! [cm] Position of telescope on gorund y / - impact parameter y */
    float               mY;
    /*! weighted number of photons arriving at observation level */
//    int                 mWeightedNumPhotons;




protected:

};

// INLINE METHODS
//

// EXTERNAL REFERENCES
//

#endif // MONTECARLO_H

