
/********************************************************************\

  Name:         HVConfig.cc

  Created by:   Sebastian Commichau, November 2008
                commichau@phys.ethz.ch

  Contents:     Class reading the HV utility configuration file

\********************************************************************/


#include "ReadCard.h"
#include "HVConfig.h"


HVConfig::HVConfig(FILE* fptr, char *configfile) {

  fLogPath   = new char[FILENAME_MAX_SIZE];
  fUSBDevice = new char*[MAX_NUM_HVBOARDS];
  fCCClient  = new char[FILENAME_MAX_SIZE];
    
  for (int i=0; i<MAX_NUM_HVBOARDS; i++) {
    fUSBDevice[i]      = new char[FILENAME_MAX_SIZE];
    USBDeviceNumber[i] = 0;

    for (int j=0; j<MAX_NUM_CHAINS; j++)
      for (int k=0; k<2; k++)
	Coef[i][j][k] = 0.;
  }

  NumHVBoards        = 0;
  FileName           = configfile;
  fStatusRefreshRate = 1.;
  fTimeOut           = 1.;
  IsDAC              = true;
  fHVCalibOffset      = -.8;
  fHVCalibSlope       = 0.0064;
  fHVMaxDiff          = 1.0;  

  if (configfile != NULL) {
    if (!ReadHVConfig(fptr, configfile)) {
      fprintf(fptr, "Error (HVConfig): could not configure HV control\n");
      exit(1);
    }
  }
}


HVConfig::~HVConfig() {

  delete [] fLogPath;

  for (int i=0; i<MAX_NUM_HVBOARDS; i++)
    delete [] fUSBDevice[i];
  delete [] fUSBDevice;

}


int HVConfig::ReadHVConfig(FILE* fptr, char *configfile) {

  FILE *f;
  char str[MAX_COM_SIZE], dev[MAX_COM_SIZE];
  int j = 0;

  if ((f = fopen(configfile,"r")) == NULL) {
    fprintf(fptr,"Could not open configuration file: %s\n", configfile);
    return 0;
  }
  else {
    fprintf(fptr,"Opening configuration file: %s\n", configfile);
  }
 
  ReadCard("LogPath",fLogPath,'s',f);

  for (int i=0;i<MAX_NUM_HVBOARDS;i++) {
    sprintf(str,"Board%d",i);

    if (ReadCard(str, dev, 's', f)==0) {
      USBDeviceNumber[j] = i;
      sprintf(fUSBDevice[j++],"%s",dev);
      NumHVBoards++;     
    }
  }
  
  ReadCard("TimeOut",           &fTimeOut,           'f', f);
  ReadCard("StatusRefreshRate", &fStatusRefreshRate, 'f', f);
  ReadCard("CCPort",            &fCCPort,            'I', f);
  ReadCard("CCClient",           fCCClient,          's', f);
  ReadCard("IsDAC",             &str,                's', f);
  ReadCard("HVCalibOffset",     &fHVCalibOffset,     'f', f);
  ReadCard("HVCalibSlope",      &fHVCalibSlope,      'f', f);
  ReadCard("HVMaxDiff",         &fHVMaxDiff,         'f', f);

  if (strcmp(str,"TRUE"))
    IsDAC = false;
  
  fclose(f);
  return 1;
}


int HVConfig::PrintHVConfig(FILE *fptr) {
  
  fprintf(fptr,"\n");                                   
  fprintf(fptr,"********************************************* CONFIG ********************************************\n\n"); 
  
  fprintf(fptr," HV control configuration (%s):\n\n", FileName);
  fprintf(fptr," Log path:          %s\n\n", fLogPath);
  fprintf(fptr," %.2d USB devices:\n\n", NumHVBoards);
  
  for (int i=0;i<NumHVBoards;i++) 
    fprintf(fptr," Board%d: %s\n", USBDeviceNumber[i], fUSBDevice[i]);
  
  fprintf(fptr,"\n");
  fprintf(fptr," TimeOut:           %.2f s\n",   fTimeOut);
  fprintf(fptr," StatusRefreshRate: %.2f Hz\n\n",fStatusRefreshRate);
  fprintf(fptr," CCPort:            %d\n",       fCCPort);
  fprintf(fptr," CCClient:          %s\n\n",     fCCClient);
  fprintf(fptr," Set DAC values:    %s\n\n",     IsDAC ? "yes" : "no");
  fprintf(fptr," HVCalibOffset :    %f\n\n",     fHVCalibOffset);
  fprintf(fptr," HVCalibSlope :     %f\n\n",     fHVCalibSlope);
  fprintf(fptr," HVMaxDiff :        %f\n\n",     fHVMaxDiff);
 
  fprintf(fptr,"*************************************************************************************************\n\n");

  return 1;
}


int HVConfig::WriteHVConfig(FILE* fptr, char *configfile) {

  FILE *f;

  time_t time_now_secs;
  struct tm *time_now;

  time(&time_now_secs);
  time_now = localtime(&time_now_secs);

  if ((f = fopen(configfile,"w")) == NULL) {
    fprintf(fptr,"Could not open file: %s\n", configfile);
    return 0;
  }
  
  fprintf(f,"# Main configuration file for the HV control program %s, %04d %02d %02d, %02d:%02d:%02d\n",
	  HV_CONTROL_VERSION, 
	  1900 + time_now->tm_year, 
	  1 + time_now->tm_mon,
	  time_now->tm_mday,
	  time_now->tm_hour,
	  time_now->tm_min,
	  time_now->tm_sec);

  fprintf(f,"# Note: this file will be updated at program exit\n\n");

  fprintf(f,"LogPath            %s\n\n",   fLogPath);

  fprintf(f,"TimeOut            %.2f s   # Timeout to return from read (%.2f,...%.2f) s\n\n",fTimeOut,MIN_TIMEOUT,MAX_TIMEOUT);

  fprintf(f,"StatusRefreshRate  %.2f Hz   # Status update rate (%.2f,...%.2f) Hz\n\n",fStatusRefreshRate,MIN_RATE,MAX_RATE);

  fprintf(f,"CCPort             %d   # Port used to look for commands from the Central Control\n",fCCPort);
  fprintf(f,"CCClient           %s   # Central Control client name\n\n",fCCClient);
  fprintf(f,"IsDAC              %s   # Define here if user input is interpreted as DAC value or voltage\n\n",((IsDAC) ? "TRUE" : "FALSE"));
  fprintf(f,"HVCalibOffset      %f   # Calibration of DAC to voltage values\n",fHVCalibOffset);
  fprintf(f,"HVCalibSlope       %f   # Calibration of DAC to voltage values\n\n",fHVCalibSlope);
  fprintf(f,"HVMaxDiff          %f   # Speed for HV changes limited to 1V/ms\n",fHVMaxDiff);

  fprintf(f,"# List of HV boards (at most %d HV boards will be recognized):\n",MAX_NUM_HVBOARDS);

  for (int i=0;i<NumHVBoards;i++) 
    fprintf(f,"Board%d             %s\n",USBDeviceNumber[i],fUSBDevice[i]);
  
  fprintf(fptr,"Configuration file successfully updated\n");
  
  fclose(f);
  return 1;

}


