
/********************************************************************\

  Name:         ProcessIO.cc

  Created by:   Sebastian Commichau, November 2008
                commichau@phys.ethz.ch

  Contents:     Main class processing user input

\********************************************************************/


#include "ProcessIO.h"


ProcessIO::ProcessIO(char *ConfigFile) {

  // Get program start time - only needed for uptime command
  time (&StartTime);

  status  = new Status;
  config  = new HVConfig(stdout,ConfigFile);
  log     = new Log(config->fLogPath);
  hv      = new HV(config->fUSBDevice,config->USBDeviceNumber,stdout);

  // Initialize status structure (HVStatus.cc/h)
  InitStatus(status,config);    

  // Print HV control configuration to the log file
  config->PrintHVConfig(log->logptr);

  // The following loop was added to allow for an arbitrary numeration of the HV boards
  for (int i=0;i<hv->GetNumberOfBoards();i++) {
    
    sprintf(status->fUSBDevice[i],"%s",(hv->GetHVBoard(i))->GetSerial());
    status->USBDeviceNumber[i] = (hv->GetHVBoard(i))->GetBoardNumber();

    if ((hv->GetHVBoard(i))->GetBoardNumber() > status->USBMaxDeviceNumber)
      status->USBMaxDeviceNumber = (hv->GetHVBoard(i))->GetBoardNumber();

    if ((hv->GetHVBoard(i))->GetBoardNumber() < status->USBMinDeviceNumber)
      status->USBMinDeviceNumber = (hv->GetHVBoard(i))->GetBoardNumber();

    (hv->GetHVBoard(i))->SetTimeOut(config->fTimeOut);
  }

  // Initialize HV boards (see below)
  InitializeHV();

  status->FirstBoard  = 0;
  status->NumHVBoards = hv->GetNumberOfBoards();
  status->LastBoard   = hv->GetNumberOfBoards()-1;

  // Print status information to the log file
  PrintStatus(status,config,log->logptr);

  // Send reset command to all boards
  ResetAllBoards();

  // Print some information
  sprintf(str,"status monitor: %s\n", GetStateStr(status));
  if (hv->GetNumberOfBoards()>0) DoPrompt(str);
  
  sprintf(str,"type (h)elp to get a list of available commands\n");
  DoPrompt(str);

  // Print empty prompt
  DoPrompt(NULL);
  
  // Initialize mutex variable for thread synchronization
  pthread_mutex_init (&control_mutex, NULL);
  pthread_cond_init  (&control_cond,  NULL);
    
}


ProcessIO::~ProcessIO() {
  
  pthread_mutex_destroy (&control_mutex);
  pthread_cond_destroy  (&control_cond);
  
  config->fStatusRefreshRate = status->fStatusRefreshRate;
  config->fTimeOut           = status->fTimeOut;
  config->WriteHVConfig(stdout,config->FileName);

  if (hv)      delete hv;
  if (status)  delete status;
  if (config)  delete config;
  if (log)     delete log;
  
}


// Print list of all commands 
void ProcessIO::PrintHelp() {

  puts("");
  puts("********************************************** HELP *********************************************\n");
  puts(" board <i>|<i> <j>|<all>           Address board i, boards i-j or all boards");
  puts(" chain <i>|<i> <j>|<all>           Address chain i, chains i-j or all chains");
  puts(" clear                             Clear screen");
  puts(" config                            Print configuration");
  puts(" help                              Print help");
  puts(" hv <ch>|<all> [b][x]<v>           Set chan. <ch>|<all> chan. of active chain(s)/board(s) to <v>");
  puts(" hvdiff <ch>|<all> [b][x]<diff>    Set chan. <ch>|<all> chan. of active chain(s)/board(s) to <diff>");
  puts(" list                              List all HV boards");
  puts(" load <file>                       Load HV settings from <file>");
  puts(" log <on>|<off>                    Enable|disable logging");
  puts(" quit|exit                         Exit program");
  puts(" rate <rate>                       Set status refresh rate to <rate> [Hz]");
  puts(" reset                             Reset active HV board");
  puts(" save [file]                       Save current HV settings to [file]");
  puts(" start                             Start HV status monitor");
  puts(" status                            Show status information");
  puts(" stop                              Stop HV status monitor - not recommended!");
  puts(" time                              Print current date and time");
  puts(" timeout <time>                    Set timeout to return from read to <time> [s]");
  puts(" uptime                            Get program uptime [h:m:s]");
  puts(" verbose <on>|<off>                Enable|disable verbosity");
  puts(" vref [b][x]<v>                    Set reference voltage of active chain(s)/board(s) to <v>\n");
  puts("*************************************************************************************************\n");

}


// Process user input
int ProcessIO::CommandControl() {

  //  bool IsDAC;
  //  float  fHVCalibOffset;
  //  float  fHVCalibSlope;
  //  float  fHVMaxDiff;


  // Adress HV board
  if (Match(status->Param[0], "board")) {
    
    if (!NBoards())
      return 0;
    else {
  
      // Select all boards
      if (status->Param[1][0] == 'a' && !status->Param[2][0]) {
	
	status->FirstBoard = 0;
	status->LastBoard = hv->GetNumberOfBoards()-1;
	
      }
      // Wrong type of argument
      else if (!IsNoDigit(status->Param[1]) || !IsNoDigit(status->Param[2]) || status->Param[3][0] || !status->Param[1][0]) {
	sprintf(str,"usage: <board> <i>|<i> <j>|<all>\n");
	DoPrompt(str);	
	return 0;
      } 
      // Check if board(s) exist(s)
      else if (status->Param[1][0] || status->Param[2][0]) {

	if (!IsBoard(atoi(status->Param[1]))) {
	  sprintf(str,"board #%d does not exist\n", atoi(status->Param[1]));
	  DoPrompt(str);
	  return 0;
	}
	
	if (status->Param[2][0])
	  if (!IsBoard(atoi(status->Param[2]))) {
	    sprintf(str,"board #%d does not exist\n", atoi(status->Param[2]));
	    DoPrompt(str);
	    return 0;
	  }

	if (status->Param[1][0] && status->Param[2][0]) {
	  
	  status->FirstBoard = GetBoardIdx(atoi(status->Param[1]));
	  status->LastBoard  = GetBoardIdx(atoi(status->Param[2]));

	}
	else if (status->Param[1][0]) {

	  status->FirstBoard = GetBoardIdx(atoi(status->Param[1]));
	  status->LastBoard = status->FirstBoard;
	  
	}
      }
    
      //DoPrompt(NULL);
    }
    
    DoPrompt(NULL);
    return 0;
  } 


  // Adress chains
  else if (Match(status->Param[0], "chain")) {
    
    if (!NBoards())
      return 0;
    else {
      
      if (status->Param[1][0] == 'a') {
	
	status->FirstChain = 0;
	status->LastChain  = 3;
	
      } 
      else if (!IsNoDigit(status->Param[1]) || !status->Param[1][0]) {
	sprintf(str,"usage: <chain> <i>|<i> <j>|<all>\n");
	DoPrompt(str);	
	return 0;
      }
      else if (status->Param[1][0] >= 0 && atoi(status->Param[1]) >= 0 && atoi(status->Param[1]) < 4 && !status->Param[2][0]) {
	
	status->FirstChain = atoi(status->Param[1]);
	status->LastChain = status->FirstChain;
	
      }
      else if ((status->Param[1][0] >= 0 && atoi(status->Param[1]) >= 0 && atoi(status->Param[1]) < 4) && 
	       (status->Param[2][0] > 0 && atoi(status->Param[2]) > 0 && atoi(status->Param[2]) < 4)) {
	
	status->FirstChain = atoi(status->Param[1]);
	status->LastChain = atoi(status->Param[2]);
	
      }
      else {
	
	if (atoi(status->Param[1]) < 0 || atoi(status->Param[1]) > 3) {
	  sprintf(str,"chain #%d does not exist\n", atoi(status->Param[1]));
	  DoPrompt(str);
	}
	
	
	if ((atoi(status->Param[2]) < 0 || atoi(status->Param[2]) > 3) && (atoi(status->Param[1]) != atoi(status->Param[2]))) {
	  sprintf(str,"chain #%d does not exist\n", atoi(status->Param[2]));
	  DoPrompt(str);
	}
	
	return 0;
	
      }

      DoPrompt(NULL);
    }
    
    return 0;
  } 


  // Clear screen
  else if (Match(status->Param[0], "clear") || Match(status->Param[0], "cls")) {

  system("clear");
  DoPrompt(NULL);

  return 0;
  
  }


  // Print HV utility configuration
  else if (Match(status->Param[0], "config")) {

    config->PrintHVConfig(stdout);
    
    return 0;
  }


  // Print help
  if (Match(status->Param[0], "help")) {

    PrintHelp();
    DoPrompt(NULL);
   
    return 0;
  } 


  // Write high voltage -----------------------------------------------------------------------------------------
  if (Match(status->Param[0], "hv")) {

    if (!NBoards())
      return 0;

    int errors = 0;
    unsigned int hvoltage = 0;
    float hvoltageV = 0.0;
    int channel = 0;
    bool allchannels = FALSE;

    if (status->Param[1][0]>0 && status->Param[2][0]>0) {

      // Set channel
      if (IsNoDigit(status->Param[1]))
	channel = atoi(status->Param[1]);
      else if(status->Param[1][0] == 'a')
	allchannels = TRUE;
      else {
	DoPrompt("error: wrong input format - usage: hv <channel>|<all> <voltage>\n");
	return 0;
      }
          
      // Binary input
      if (tolower(status->Param[2][0])=='x' && strlen(status->Param[2])>2) {
	if (sPrintHex2Dec(Chop(status->Param[2]+1),&hvoltage)!=0) {
	  DoPrompt("error: wrong input format - usage: hv <channel>|<all> <voltage>\n");
	  return 0;
	}
      }
      // Hexadecimal input
      else if (tolower(status->Param[2][0])=='b' && strlen(status->Param[2])>2) {
	if (sPrintBin2Dec(Chop(status->Param[2]+1),&hvoltage)!=0) {
	  DoPrompt("wrong input format - usage: hv <channel>|<all> <voltage>\n");
	  return 0;
	}
      }
      // Decimal input
      else if (IsNoDigit(status->Param[2])&&config->IsDAC)
	hvoltage = atoi(status->Param[2]);
      else if (IsNoDigit(status->Param[2])&&(!(config->IsDAC)))
	hvoltageV = atof(status->Param[2]);
      // Wrong input format
      else {
	DoPrompt("wrong input format - usage: hv <channel>|<all> <voltage>\n");
	return 0;
      }

      // Check limits
      if (channel>31 || channel <0) {
	DoPrompt("channel out of range (0...31)!\n");
	return 0;
      }
      else if ((hvoltage>0X3FFF || hvoltage <0)&&config->IsDAC) {
	DoPrompt("high voltage out of range (Vmin: 0, Vmax: 16383)!\n");
	return 0;
      }
      else if ((hvoltage>78.0 || hvoltage <0)&&(!(config->IsDAC))) {
	DoPrompt("high voltage out of range (Vmin: 0, Vmax: 78.)!\n");
	return 0;
      }

      if (!(config->IsDAC))
	hvoltage=(unsigned int)((hvoltageV-config->fHVCalibOffset)/config->fHVCalibSlope);

      StopMonitor();
      
      
      for (int i=status->FirstBoard;i<=status->LastBoard;i++) {

	for (int j=status->FirstChain;j<=status->LastChain;j++) {
	  if (!allchannels) {
	    if ((hv->GetHVBoard(i))->SetHV(stdout,j,channel,hvoltage,rbuf,status->Verbose)==1) {
	      
	      status->HV[i][j][channel]=hvoltage;
	      sprintf(str, "Test1   status->HV = %d\n",hvoltage);
	      DoPrompt(str);
	      UpdateStatus(i,rbuf);
	      
	      sprintf(str,"board %d: high voltage of chain %d channel %d set to %d | 0X%.4X\n",hv->GetHVBoard(i)->GetBoardNumber(),j,channel,hvoltage,hvoltage);
	      DoPrompt(str);
	      if (status->Verbose) {
		sPrintStatus(status,str,i);
		DoPrompt(str);
	      }
	      sPrintStatus(status,str,i); // Print status only to socket 
	    }
	    else {
	      sprintf(str,"board %d error: could not set hv - check timeout and try again\n",hv->GetHVBoard(i)->GetBoardNumber());
	      DoPrompt(str);
	      errors++;
	    }

	  }
	  else {

	    sprintf(str,"updating board %d chain %d\n",hv->GetHVBoard(i)->GetBoardNumber(),j);
	    DoPrompt(str);
	    

	    for (int k=0;k<MAX_NUM_CHANNELS;k++) {
	      if ((hv->GetHVBoard(i))->SetHV(stdout,j,k,hvoltage,rbuf,status->Verbose)==1) {
		
		status->HV[i][j][k]=hvoltage;
		UpdateStatus(i,rbuf);
		
		if (status->Verbose) {
		  sprintf(str,"board %d: high voltage of chain %d channel %d set to %d | 0X%.4X\n",hv->GetHVBoard(i)->GetBoardNumber(),j,k,hvoltage,hvoltage);
		  DoPrompt(str);
		  sPrintStatus(status,str,i);
		  DoPrompt(str);
		}
		sPrintStatus(status,str,i); // Print status only to socket 
		
	      }
	      else {
		sprintf(str,"board %d error: could not set HV - check timeout and try again\n",hv->GetHVBoard(i)->GetBoardNumber());
		DoPrompt(str);
		errors++;
	      }
	    }
	  }
	}
      }

      

      StartMonitor();

      if (errors) {
	sprintf(str,"warning %d error(s) => check timeout and try again\n",errors);
	DoPrompt(str);
      }
      else {
	sprintf(str,"no error(s)... success!\n");
	DoPrompt(str);
      }

    }
    else {
      sprintf(str,"usage: hv <channel>|<all> <voltage>\n");
      DoPrompt(str);
    }

    return 0;
  }  // End: Write high voltage  ----------------------------------------------------------------------------------------



  // Write difference of high voltage ------------------------------------------------------------------------------------
  if (Match(status->Param[0], "hvdiff")) {

    if (!NBoards())
      return 0;

    int errors = 0;
    int hvdiff = 0;
    unsigned int hvoltage = 0;
    float hvdiffV = 0.0;
    int channel = 0;
    bool allchannels = FALSE;

    if (status->Param[1][0]>0 && status->Param[2][0]>0) {

      // Set channel
      if (IsNoDigit(status->Param[1]))
	channel = atoi(status->Param[1]);
      else if(status->Param[1][0] == 'a')
	allchannels = TRUE;
      else {
	DoPrompt("error: wrong input format - usage: hvdiff <channel>|<all> <hv difference>\n");
	return 0;
      }

      // Binary input
      if (tolower(status->Param[2][0])=='x' && strlen(status->Param[2])>2) {
	if (sPrintHex2Dec(Chop(status->Param[2]+1),(unsigned int *)hvdiff)!=0) {
	  DoPrompt("error: wrong input format - usage: hvdiff <channel>|<all> <hv difference>\n");
	  return 0;
	}
      }
      // Hexadecimal input
      else if (tolower(status->Param[2][0])=='b' && strlen(status->Param[2])>2) {
	if (sPrintBin2Dec(Chop(status->Param[2]+1),(unsigned int *)hvdiff)!=0) {
	  DoPrompt("wrong input format - usage: hvdiff <channel>|<all> <hv difference>\n");
	  return 0;
	}
      }
      // Decimal input
      else if (IsNoDigit(status->Param[2])&&(config->IsDAC))
	hvdiff = atoi(status->Param[2]);
      else if (IsNoDigit(status->Param[2])&&(!(config->IsDAC)))
	hvdiffV = atof(status->Param[2]);
      // Wrong input format
      else {
	DoPrompt("wrong input format - usage: hvdiff <channel>|<all> <hv difference>\n");
	return 0;
      }
      // Check limits
      if (channel>31 || channel <0) {
	DoPrompt("channel out of range (0...31)!\n");
	return 0;
      }
      else if (hvdiff>0X3FFF) {
	DoPrompt("difference of high voltage out of range (Vmin: -16383, Vmax: 16383)!\n");
	return 0;
      }
      else if (hvdiffV>78.0 && (-78.0<hvdiffV)) {
	DoPrompt("difference of high voltage out of range (Vmin: -78.0, Vmax: 78.0)!\n");
	return 0;
      }

      if (!(config->IsDAC)){
	hvdiff=(int)(hvdiffV/config->fHVCalibSlope);
      }
      StopMonitor();

      
      for (int i=status->FirstBoard;i<=status->LastBoard;i++) {

	for (int j=status->FirstChain;j<=status->LastChain;j++) {
	  if (!allchannels) {
	    hvoltage = status->HV[i][j][channel];
	    for (int k=0;k<=abs((int)(hvdiff/config->fHVMaxDiff));k++){
	      if (k<abs((int)(hvdiff/config->fHVMaxDiff))){
		hvoltage=(unsigned int)(hvoltage + config->fHVMaxDiff*hvdiff/abs(hvdiff));
	      }
	      if (k==(int)(abs((int)(hvdiff/config->fHVMaxDiff)))){
		hvoltage=(unsigned int)(hvoltage + (hvdiff%(int)config->fHVMaxDiff));
	      }


	      status->HV[i][j][channel]=hvoltage;
	      if ((hv->GetHVBoard(i))->SetHV(stdout,j,channel,hvoltage,rbuf,status->Verbose)==1) {
		UpdateStatus(i,rbuf);
		sprintf(str,"board %d: high voltage of chain %d channel %d set to %d | 0X%.4X\n",hv->GetHVBoard(i)->GetBoardNumber(),j,channel,hvoltage,hvoltage);
		DoPrompt(str);
		if (status->Verbose) {
		  sPrintStatus(status,str,i);
		  DoPrompt(str);
		}
		sPrintStatus(status,str,i); // Print status only to socket 
	      }
	      else {
		sprintf(str,"board %d error: could not set hv - check timeout and try again\n",hv->GetHVBoard(i)->GetBoardNumber());
		DoPrompt(str);
		errors++;
	      }
	      //	    Sleep(1);
	    }
	  }
	  else {
	    
	    sprintf(str,"updating board %d chain %d\n",hv->GetHVBoard(i)->GetBoardNumber(),j);
	    DoPrompt(str);
	    

	    for (int k=0;k<MAX_NUM_CHANNELS;k++) {
   	      int hvoltage = status->HV[i][j][k];
	      for (int l=0;l<=abs((int)(hvdiff/config->fHVMaxDiff));l++){
	    	if (l<abs((int)(hvdiff/config->fHVMaxDiff)))
		  hvoltage=(unsigned int)(hvoltage + config->fHVMaxDiff*hvdiff/abs(hvdiff));
		if (l==(int)(abs((int)(hvdiff/config->fHVMaxDiff))))
		  hvoltage=(unsigned int)(hvoltage + (hvdiff%(int)config->fHVMaxDiff));
		status->HV[i][j][k]=hvoltage;
		if ((hv->GetHVBoard(i))->SetHV(stdout,j,k,hvoltage,rbuf,status->Verbose)==1) {
		
		  UpdateStatus(i,rbuf);
		
		  if (status->Verbose) {
		    sprintf(str,"board %d: high voltage of chain %d channel %d set to %d | 0X%.4X\n",hv->GetHVBoard(i)->GetBoardNumber(),j,k,hvoltage,hvoltage);
		    DoPrompt(str);
		    sPrintStatus(status,str,i);
		    DoPrompt(str);
		  }
		  sPrintStatus(status,str,i); // Print status only to socket 
		  
		}
		else {
		  sprintf(str,"board %d error: could not set HV - check timeout and try again\n",hv->GetHVBoard(i)->GetBoardNumber());
		  DoPrompt(str);
		  errors++;
		}
		//		Sleep(0.001);
	      }
	    }
	  }
	}
      }
      StartMonitor();

      if (errors) {
	sprintf(str,"warning %d error(s) => check timeout and try again\n",errors);
	DoPrompt(str);
      }
      else {
	sprintf(str,"no error(s)... success!\n");
	DoPrompt(str);
      }
    }
    else {
      sprintf(str,"usage: hvdiff <channel>|<all> <hv difference>\n");
      DoPrompt(str);
    }
    return 0;

    }

      // End: Write difference of high voltage --------------------------------------------------------------------
  


  // List HV boards
  else if (Match(status->Param[0], "list")) {
    
    sprintf(str,"%d HV board(s) active:\n",hv->GetNumberOfBoards());
    DoPrompt(str);

    for (int i=0;i<hv->GetNumberOfBoards();i++) {
      sprintf(str,"board %d (%s)\n",(hv->GetHVBoard(i))->GetBoardNumber(),(hv->GetHVBoard(i))->GetSerial());
      DoPrompt(str);
    }
    
    return 0;
  }


  // Load HV settings from file
  else if (Match(status->Param[0], "load")) {

    char param[20][MAX_COM_SIZE];
    char buffer[128];

    int nparam  = 0;
    int nrows   = 0;
    int board   = 0;
    int chain   = 0;
    int channel = 0;
    int nboards = 0;
    int errors  = 0;
    unsigned int hvoltage = 0;

    FILE *file;
    
    if (status->Param[1][0] && !status->Param[2][0]) {
      
      if ((file=fopen((char *)Chop(status->Param[1]),"r"))!=NULL) {
	
	if (status->Verbose) {
	  sprintf(str,"file \"%s\" opened\n",Chop(status->Param[1]));
	  DoPrompt(str);
	}
	
	StopMonitor();

	while (fgets(buffer,100,file)) {
	  
	  ParseInput(buffer,&nparam,param);
	  
	  if (nparam==2 && Match(param[0],"Device:")) {
	    	    
	    for (int j=0;j<hv->GetNumberOfBoards();j++)
	      if (Match(Chop(status->fUSBDevice[j]),Chop(param[1]))) {

		board = j;
		
		sprintf(str,"found HV settings for board %d (%s)\n",hv->GetHVBoard(board)->GetBoardNumber(),Chop(param[1]));
		DoPrompt(str);
		
		nboards++;
		nrows = 0;
	      
		while (fgets(buffer,100,file)) {
		  
		  ParseInput(buffer,&nparam,param);
		  
		  if (nparam==8) { 

		    chain = nrows/4;

		    if (!((nrows)%4)) {
		      sprintf(str,"updating board %d chain %d\n",hv->GetHVBoard(board)->GetBoardNumber(),chain);
		      DoPrompt(str);
		    }

		    
		    for (int i=0;i<nparam;i++) {
		      
		      hvoltage = atoi(param[i]);
		      channel  = (i+8*nrows)%32;
		      
		      // Submit HV values
		      if ((hv->GetHVBoard(board))->SetHV(stdout,chain,channel,hvoltage,rbuf,status->Verbose)==1) {
			
			status->HV[board][chain][channel]=hvoltage;
			UpdateStatus(board,rbuf);
			
			if (status->Verbose) {
			  sprintf(str,"board %d: high voltage of chain %d channel %d set to %d | 0X%.4X\n",
				  hv->GetHVBoard(board)->GetBoardNumber(),chain,channel,hvoltage,hvoltage);
			  DoPrompt(str);
			  sPrintStatus(status,str,board);
			  DoPrompt(str);
			}
		      }
		      else {
			sprintf(str,"board %d error: could not set HV - check timeout and try again\n",hv->GetHVBoard(board)->GetBoardNumber());
			DoPrompt(str);
			errors++;
		      }
		      
		    }
		    
		    nrows++;
		    
		  }
		  else if (nparam==2)
		    break;
		}
	      }
	  } 
	}
	
	if (fclose (file)) {
	  sprintf(str,"error: could not close file \"%s\"\n",Chop(status->Param[1]));
	  DoPrompt(str);
	} 
	else {
	  if (status->Verbose) {
	    sprintf(str,"file \"%s\" closed\n",Chop(status->Param[1]));
	    DoPrompt(str);
	  }
	}

	if (nboards!=hv->GetNumberOfBoards()) {
	  sprintf(str,"warning: could not load HV settings for all connected HV boards\n");
	  DoPrompt(str);
	}
	else {
	  sprintf(str,"success: read HV settings for all connected HV boards\n");
	  DoPrompt(str);
	}

	if (errors) {
	  sprintf(str,"warning %d error(s) => check timeout and try again\n",errors);
	  DoPrompt(str);
	}
	else {
	  sprintf(str,"no error(s)... success!\n");
	  DoPrompt(str);
	}
	
      }
      else {
	sprintf(str,"error: could not open file \"%s\"\n",Chop(status->Param[1]));
	DoPrompt(str);
      }
      
    }
    else
      DoPrompt("usage: load <file>\n");

    StartMonitor();

    return 0;
    
  }
	   

  // Enable/disable logging
  else if (Match(status->Param[0], "log")) {

    if (Match(status->Param[1], "on") && status->Param[1][0]) {
      status->Log = TRUE;
      sprintf(str,"logging enabled\n");
      DoPrompt(str);
    }
    
    else if (Match(status->Param[1], "off") && status->Param[1][0]) {
      status->Log = FALSE;
      sprintf(str,"logging disabled\n");
      DoPrompt(str);
    }
    
    else
      DoPrompt("usage: log <on>|<off>\n");
  
    return 0;
  }


  // Set status refresh rate
  if (Match(status->Param[0], "rate")) {

    if (!NBoards())
      return 0;

    if (status->Param[1][0]>0) {

      if (!IsNoDigit(status->Param[1])) {
	DoPrompt("wrong input format - usage: rate <rate>\n");
	return 0;
      }
      // Check limits
      else if (atof(status->Param[1]) < MIN_RATE || atof(status->Param[1]) > MAX_RATE) {
	sprintf(str,"refresh rate out of range (min: %.2f Hz, max: %.2f Hz)!\n",MIN_RATE,MAX_RATE);
	DoPrompt(str);
	return 0;
      }
      else {
	StopMonitor();
	status->fStatusRefreshRate=atof(status->Param[1]);
	sprintf(str,"status refresh rate set to %.2f Hz\n",status->fStatusRefreshRate);
	DoPrompt(str);
      }
      StartMonitor();
      return 0;
      
    }
    else {
      sprintf(str,"usage: rate <rate>\n");
      DoPrompt(str);
    }
    
    return 0;
  }

  /*
  // Read from device - NEVER use this function with a real HV board!!!
  if (Match(status->Param[0], "read")) {

    if (!NBoards())
      return 0;

    if (status->state==active) {
      StopMonitor();
      sprintf(str,"warning: status monitoring deactivated\n");
      DoPrompt(str);
    }
    
    for (int i=status->FirstBoard;i<=status->LastBoard;i++) {
      if (1 == ((hv->GetHVBoard(i))->Read(rbuf,1))) {
	sPrintByteBin(rbuf[0],bdata);
	sprintf(str,"%d byte(s) read (board %d): %d | 0X%.2X | B%s\n",1,i,rbuf[0],rbuf[0],bdata);
      }
      else
	sprintf(str,"error: could not read from board %d\n",i);

      usleep(1000);

      DoPrompt(str);
    }

    return 0;
  } 
  */
  
  // Reset
  if (Match(status->Param[0], "reset")) {

    if (!NBoards())
      return 0;

    StopMonitor();
    ResetActiveBoards();
    ReInitStatus(status);
    StartMonitor();
    return 0;
  }


  // Start ROOT
  else if (Match(status->Param[0], "root")) {
    
    sprintf(str,"starting ROOT... type '.q' to return\n");
    DoPrompt(str);
    system ("root");

  return 0;
  
  }


  // Save HV settings of all boards
  else if (Match(status->Param[0], "save")) {

    if (status->Param[1][0] && !status->Param[2][0]) {
      
      if (SaveHVSettings(Chop(status->Param[1]))) {
	sprintf(str,"HV settings written to \"%s\"\n",Chop(status->Param[1]));
	DoPrompt(str);
      }
      
    }
    else {
      
      char buffer[MAX_COM_SIZE];
      
      time_t time_now_secs;
      struct tm *time_now;
      
      time(&time_now_secs);
      time_now = localtime(&time_now_secs);
      
      sprintf(buffer,"hvsettings/HV_%04d-%02d-%02d_%02d-%02d-%02d.txt",
	      1900 + time_now->tm_year, 
	      1 + time_now->tm_mon,
	      time_now->tm_mday,
	      time_now->tm_hour,
	      time_now->tm_min,
	      time_now->tm_sec);

      //sprintf(str,"warning: HV settings will be written to \"%s\"\n",buffer);
      //DoPrompt(str);
      
      if (SaveHVSettings(buffer)) {
	sprintf(str,"HV settings successfully written to \"%s\"\n",buffer);
	DoPrompt(str);
      }
    }

    return 0;
   
  }


  // Start monitoring
  else if (Match(status->Param[0], "start")) {

    if (!NBoards())
      return 0;
    
    if (status->state==active) {
      sprintf(str,"status monitoring is already active\n");
      DoPrompt(str);
      return 0;
    }

    StartMonitor();

    sprintf(str,"status monitoring activated\n");
    DoPrompt(str);

    return 0;
  
  }

 
  // Print status
  else if (Match(status->Param[0], "status") || Match(status->Param[0], "info")) {
    
    if (!NBoards())
      return 0;

    PrintStatus(status,config,stdout);
    PrintStatus(status,config,log->logptr);

    return 0;
  } 


  // Stop monitoring
  else if (Match(status->Param[0], "stop")) {

    if (!NBoards())
      return 0;
    
    if (status->state!=active) {
      sprintf(str,"status monitoring is already deactivated\n");
      DoPrompt(str);
      return 0;
    }

    StopMonitor();
    
    sprintf(str,"warning: status monitoring deactivated\n");
    DoPrompt(str);

    return 0;
  
  }

  /*
  // Sweep HV
  else if (Match(status->Param[0], "sweep")) {

    if (!NBoards())
      return 0;

    int errors = 0;
    int channel = 0;

    unsigned int hv1 = 0;
    unsigned int hv2 = 0;
    unsigned int dv  = 10;

    bool allchannels = FALSE;

    float delay = 1.;

    if (status->Param[1][0]>0 && status->Param[2][0]>0 && status->Param[3][0]>0 && status->Param[4][0]>0){
      
      if (status->Param[1][0] == 'a' )
	allchannels = TRUE;
      else if (IsNoDigit(status->Param[1])) {
	channel = atoi(status->Param[1]);
	// Check limits
	if (channel>31 || channel <0) {
	  DoPrompt("channel out of range (0...31)!\n");
	  return 0;
	}
      }
      
      if (IsNoDigit(status->Param[2]) && IsNoDigit(status->Param[3]) && IsNoDigit(status->Param[4])) {
	
	hv1 = atoi(status->Param[2]);
	hv2 = atoi(status->Param[3]);
	dv  = atoi(status->Param[4]);
	
	if ((hv1>0X3FFF || hv1 <0) || (hv2>0X3FFF || hv2 <0)) {
	  DoPrompt("high voltage out of range (Vmin: 0, Vmax: 16383)!\n");
	  return 0;
	}

	if (hv1 > hv2) {
	  DoPrompt("wrong limits (Vmin < Vmax)!\n");
	  return 0;
	}

	if (dv < 1 || dv > abs(hv2-hv1)) {
	  DoPrompt("step size out of range (dVmin: 1, dVmax: |hv2-hv1|)!\n");
	  return 0;
	}
	
	if (status->Param[5][0]>0 && IsNoDigit(status->Param[5])) {
	  if (atof(status->Param[5])<MIN_SWEEP_RATE || atof(status->Param[5])>MAX_SWEEP_RATE) {
	    sprintf(str,"rate out of range (min: %.2f Hz, max: %.2f Hz)!\n",MIN_SWEEP_RATE,MAX_SWEEP_RATE);	    
	    DoPrompt(str);
	    return 0;
	  }
	  else 
	    delay = 1./atof(status->Param[5]);
	}
      }
      else {
	DoPrompt("error: wrong input format - usage: sweep <channel>|<all> <v1> <v2> <dv> [rate]\n");
	return 0;
      }


      StopMonitor();
      
      for (int i=status->FirstBoard;i<=status->LastBoard;i++) 
	for (int j=status->FirstChain;j<=status->LastChain;j++) {
	  if (!allchannels) {
	    
	    for (unsigned int v = hv1; v<=hv2; v+=dv) {
	      if ((hv->GetHVBoard(i))->SetHV(stdout,j,channel,v,rbuf,status->Verbose)==1) {
		sprintf(str,"board %d: high voltage of chain %d channel %d set to %d | 0X%.4X\n",i,j,channel,v,v);
		status->HV[i][j][channel]=v;
		DoPrompt(str);
		UpdateStatus(i,rbuf);
		sPrintStatus(status,str,i);
		DoPrompt(str);
	      }
	      else {
		sprintf(str,"board %d error: could not set hv - check timeout and try again\n",i);
		DoPrompt(str);
		errors++;
	      }

	      usleep((unsigned long)floor(delay*1000000.));
	    }

	  }
	  else {
	    for (int k=0;k<MAX_NUM_CHANNELS;k++) {

	      for (unsigned int v = hv1; v<=hv2; v+=dv) {
		
		if ((hv->GetHVBoard(i))->SetHV(stdout,j,k,v,rbuf,status->Verbose)==1) {
		  sprintf(str,"board %d: high voltage of chain %d channel %d set to %d | 0X%.4X\n",i,j,k,v,v);
		  status->HV[i][j][k]=v;
		  DoPrompt(str);
		  UpdateStatus(i,rbuf);
		  sPrintStatus(status,str,i);
		  DoPrompt(str);
		}
		else {
		  sprintf(str,"board %d error: could not set hv - check timeout and try again\n",i);
		  DoPrompt(str);
		  errors++;
		}
		
		usleep((unsigned long)floor(delay*1000000.));

	      }
	    }
	  }
	}

      StartMonitor();

      if (errors) {
	sprintf(str,"warning %d error(s) => check timeout and try again\n",errors);
	DoPrompt(str);
      }
      else {
	sprintf(str,"no error(s)... success!\n");
	DoPrompt(str);
      }
    }
    else {
      DoPrompt("usage: sweep <channel>|<all> <v1> <v2> <dv> [rate]\n");
      return 0;
    }
    
    return 0;
  
  }
  */

  // Print time and date
  else if (Match(status->Param[0], "date") || Match(status->Param[0], "time")) {

    PrintDateAndTime();
   
    return 0;
  } 


  /*
  // Test I/O - NEVER use this function with a real HV board!!!
  else if (Match(status->Param[0], "testio")) {

    for (int i=status->FirstBoard;i<=status->LastBoard;i++) 
      (hv->GetHVBoard(i))->TestIO();

    return 0;
  } 
  */


  // Set timeout to return from read
  if (Match(status->Param[0], "timeout")) {

    if (!NBoards())
      return 0;

    if (status->Param[1][0]>0) {

      if (!IsNoDigit(status->Param[1])) {
	DoPrompt("wrong input format - usage: timeout <time>\n");
	return 0;
      }
      // Check limits
      else if (atof(status->Param[1]) < MIN_TIMEOUT || atof(status->Param[1]) > MAX_TIMEOUT) {
	sprintf(str,"timeout out of range (min: %.2f s, max: %.2f s)!\n",MIN_TIMEOUT,MAX_TIMEOUT);
	DoPrompt(str);
	return 0;
      }
      else {
	StopMonitor();  
	for (int i=0;i<hv->GetNumberOfBoards();i++) 
	  (hv->GetHVBoard(i))->SetTimeOut((float)atof(status->Param[1]));
	status->fTimeOut = atof(status->Param[1]);	
	sprintf(str,"timeout set to %.2f s\n",status->fTimeOut);

	DoPrompt(str);
	
	StartMonitor();
	return 0;
      }
    }
    else {
      sprintf(str,"usage: timeout <time>\n");
      DoPrompt(str);
    }
    
    return 0;
  }

    
  // Print uptime
  if (Match(status->Param[0], "uptime")) {

    double difftime = GetDiffTime(&StartTime);

    sprintf(str,"%d:%02d:%02d\n",(int)difftime/SECONDS_HOUR,((int)difftime/SECONDS_MINUTE)%SECONDS_MINUTE,(int)difftime%SECONDS_MINUTE);
    DoPrompt(str);

    return 0;
  } 


  // Enable/disable verbosity
  else if (Match(status->Param[0], "verbose")) {

    if (Match(status->Param[1], "on") && status->Param[1][0]) {
      if (status->Verbose == TRUE) {
	sprintf(str,"verbosity is already enabled\n");
	DoPrompt(str);
      }
      else {
	status->Verbose = TRUE;
	sprintf(str,"verbosity enabled\n");
	DoPrompt(str);
      }
    }
    
    else if (Match(status->Param[1], "off") && status->Param[1][0]) {
      if (status->Verbose == FALSE) {
	sprintf(str,"verbosity is already disabled\n");
	DoPrompt(str);
      }
      else {
	status->Verbose = FALSE;
	sprintf(str,"verbosity disabled\n");
	DoPrompt(str);
      }
    }
    
    else
      DoPrompt("usage: verbose <on>|<off>\n");
  
    return 0;
  }
  
  
  // Write reference voltage
  if (Match(status->Param[0], "vref")) {

    if (!NBoards())
      return 0;

    unsigned int voltage = 0;

    if (status->Param[1][0]>0) {
      
      // Binary input
      if (tolower(status->Param[1][0])=='x' && strlen(status->Param[1])>2) {
	if (sPrintHex2Dec(Chop(status->Param[1]+1),&voltage)!=0) {
	    DoPrompt("error: wrong input format - usage: vref <voltage>\n");
	    return 0;
	}
      }
      // Hexadecimal input
      else if (tolower(status->Param[1][0])=='b' && strlen(status->Param[1])>2) {
	if (sPrintBin2Dec(Chop(status->Param[1]+1),&voltage)!=0) {
	  DoPrompt("wrong input format - usage: vref <voltage>\n");
	  return 0;
	}
      }
      // Decimal input
      else if (IsNoDigit(status->Param[1]))
	voltage = atoi(status->Param[1]);
      // Wrong input format
      else {
	DoPrompt("wrong input format - usage: vref <voltage>\n");
	return 0;
      }

      // Check limits
      if (voltage>0X3FFF || voltage <0) {
	DoPrompt("reference voltage out of range (Vmin: 0, Vmax: 16383)!\n");
	return 0;
      }


      StopMonitor();
          
      for (int i=status->FirstBoard;i<=status->LastBoard;i++) 
	for (int j=status->FirstChain;j<=status->LastChain;j++) {
	  if (((hv->GetHVBoard(i))->SetVRef(stdout,j,voltage,rbuf,status->Verbose)==1)) {
	    sprintf(str,"board %d: reference voltage of chain %d set to %d | 0X%.4X\n",hv->GetHVBoard(i)->GetBoardNumber(),j,voltage,voltage);
	    status->VRef[i][j]=voltage;
	    DoPrompt(str);
	    UpdateStatus(i,rbuf);
	    sPrintStatus(status,str,i);
	    DoPrompt(str);
	  } else {
	    sprintf(str,"board %d error: could not set vref - check timeout and try again\n",hv->GetHVBoard(i)->GetBoardNumber());
	    DoPrompt(str);
	  }
	}
      
      StartMonitor();
    }
    else {
      sprintf(str,"usage: vref <voltage>\n");
      DoPrompt(str);
    }

    return 0;
  }

  /*
  // Write to device - NEVER use this function with a real HV board!!!
  if (Match(status->Param[0], "write")) {
    
    if (!NBoards())
      return 0;

    if (status->Param[1][0]>0) {
      
      // Binary input
      if (tolower(status->Param[1][0])=='x' && strlen(status->Param[1])>2) {
	if (sPrintHex2Dec(Chop(status->Param[1]+1),(unsigned int*)wbuf)!=0) {
	  DoPrompt("wrong input format or value out of range!\n");
	  return 0;
	}
      }
      // Hexadecimal input
      else if (tolower(status->Param[1][0])=='b' && strlen(status->Param[1])>2) {
	if (sPrintBin2Dec(Chop(status->Param[1]+1),(unsigned int*)wbuf)!=0) {
	  DoPrompt("wrong input format or value out of range!\n");
	  return 0;
	}
      }
      // Decimal input
      else if (atoi(status->Param[1])>=0 && atoi(status->Param[1])<=255 && IsNoDigit(status->Param[1]))
	wbuf[0] = (unsigned char)atoi(status->Param[1]);
      // Wrong input
      else {
	DoPrompt("wrong input format or value out of range!\n");
	return 0;
      }
      
      if (status->state==active) {
	StopMonitor();
	sprintf(str,"warning: status monitoring deactivated\n");
	DoPrompt(str);
      }
      
      for (int i=status->FirstBoard;i<=status->LastBoard;i++) 
        if ((hv->GetHVBoard(i))->Write(wbuf,1) == 1) {
	  sPrintByteBin(wbuf[0],bdata);
	  sprintf(str,"%d byte(s) written (board %d): %d | 0X%.2X | B%s\n",1,i,wbuf[0],wbuf[0],bdata);
	  DoPrompt(str);
	}
	else {
	  sprintf(str,"error: could not write to board %d\n",i);
	  DoPrompt(str);
	}

    }
    else {
      sprintf(str,"error: no input\n");
      DoPrompt(str);
    }
    return 0;
  }
  */

  // Exit program
  else if(Match(status->Param[0], "exit") || Match(status->Param[0], "quit")) {

    StopMonitor();
    
    ResetAllBoards();

    status->Exit = TRUE;

    pthread_kill(status->HVMonitor, SIGUSR1);
    pthread_kill(status->SocketThread, SIGUSR1);
 
    return 0;
  }
  
  else if (strchr(status->Param[0],'\n')-status->Param[0]==0) {
    return 0;
  }

  else {
    
    if (strchr(status->Param[0],'\n') == 0)
      sprintf(str,"unknown command: %s\n",status->Param[0]);
    else
      sprintf(str,"unknown command: %s",status->Param[0]);
    
    DoPrompt(str);
    
    return 0;

  }
 
  return 0;
}


void ProcessIO::PrintDateAndTime() {

  char str[MAX_COM_SIZE];

  time_t rawtime;
  struct tm * timeinfo;
  
  time(&rawtime);
  timeinfo = localtime(&rawtime); // Get local time
  fflush(stdout);
  sprintf(str,"current date/time is: %s",asctime(timeinfo));
  DoPrompt(str);

}

void ProcessIO::DoPrompt(char* str) {
 
  if (str!=NULL) {
    if (status->NumHVBoards == 0) {
      sprintf(status->Prompt,"HV> %s",str);
      printf(status->Prompt);
      if (status->Log)
	log->LogWrite(status->Prompt);
      sprintf(status->Prompt,"HV> ");
    }
    else if (status->FirstBoard == status->LastBoard) {

      int board = hv->GetHVBoard(status->FirstBoard)->GetBoardNumber();

      if (status->FirstChain == status->LastChain) {
	sprintf(status->Prompt,"HV|C%d|B%d> %s",status->FirstChain,board,str);
	printf(status->Prompt);
	if (status->Log)
	  log->LogWrite(status->Prompt);
	sprintf(status->Prompt,"HV|C%d|B%d> ",status->FirstChain,board);
      }
      else {
	sprintf(status->Prompt,"HV|C%d-%d|B%d> %s",status->FirstChain,status->LastChain,board,str);
	printf(status->Prompt);
	if (status->Log)
	  log->LogWrite(status->Prompt);
	sprintf(status->Prompt,"HV|C%d-%d|B%d> ",status->FirstChain,status->LastChain,board);
      }

    }
    else {
     
      int firstboard = (hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() > hv->GetHVBoard(status->LastBoard)->GetBoardNumber()) ?
	hv->GetHVBoard(status->LastBoard)->GetBoardNumber() : hv->GetHVBoard(status->FirstBoard)->GetBoardNumber();

      int lastboard = (hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() > hv->GetHVBoard(status->LastBoard)->GetBoardNumber()) ?
	hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() : hv->GetHVBoard(status->LastBoard)->GetBoardNumber();

      if (status->FirstChain == status->LastChain) {
	sprintf(status->Prompt,"HV|C%d|B%d-%d> %s",status->FirstChain,firstboard,lastboard,str);
	printf(status->Prompt);
	if (status->Log)
	  log->LogWrite(status->Prompt);
	sprintf(status->Prompt,"HV|C%d|B%d-%d> ",status->FirstChain,firstboard,lastboard);
      }
      else {
	sprintf(status->Prompt,"HV|C%d-%d|B%d-%d> %s",status->FirstChain,status->LastChain,firstboard,lastboard,str);
	printf(status->Prompt);
	if (status->Log)
	  log->LogWrite(status->Prompt);
	sprintf(status->Prompt,"HV|C%d-%d|B%d-%d> ",status->FirstChain,status->LastChain,firstboard,lastboard);
      }
    }
  }
  else {
    if (status->NumHVBoards == 0)
      sprintf(status->Prompt,"HV> ");
    else if (status->FirstBoard == status->LastBoard) {
      
      int board = hv->GetHVBoard(status->FirstBoard)->GetBoardNumber();

      if (status->FirstChain == status->LastChain)
	sprintf(status->Prompt,"HV|C%d|B%d> ",status->FirstChain,board);
      else
	sprintf(status->Prompt,"HV|C%d-%d|B%d> ",status->FirstChain,status->LastChain,board);
    }
    else {

      int firstboard = (hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() > hv->GetHVBoard(status->LastBoard)->GetBoardNumber()) ?
	hv->GetHVBoard(status->LastBoard)->GetBoardNumber() : hv->GetHVBoard(status->FirstBoard)->GetBoardNumber();

      int lastboard = (hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() > hv->GetHVBoard(status->LastBoard)->GetBoardNumber()) ?
	hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() : hv->GetHVBoard(status->LastBoard)->GetBoardNumber();

     if (status->FirstChain == status->LastChain)
       sprintf(status->Prompt,"HV|C%d|B%d-%d> ",status->FirstChain,firstboard,lastboard);
     else 
       sprintf(status->Prompt,"HV|C%d-%d|B%d-%d> ",status->FirstChain,status->LastChain,firstboard,lastboard);
     }
  }

  
}


void ProcessIO::PrintMessage(char *str) {

  char outstr[MAX_COM_SIZE];
  
  fflush(stdout);
  if (str!=NULL) {

    if (status->NumHVBoards == 0) 
      sprintf(outstr,"%s\nHV> ",str); 
    else if (status->FirstBoard == status->LastBoard) {
      
      int board = hv->GetHVBoard(status->FirstBoard)->GetBoardNumber();
      
      if (status->FirstChain == status->LastChain) 
	sprintf(outstr,"%s\nHV|C%d|B%d> ",str,status->FirstChain,board); 
      else 
	sprintf(outstr,"%s\nHV|C%d-%d|B%d> ",str,status->FirstChain,status->LastChain,board); 
    }
    else {
      
      int firstboard = (hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() > hv->GetHVBoard(status->LastBoard)->GetBoardNumber()) ?
	hv->GetHVBoard(status->LastBoard)->GetBoardNumber() : hv->GetHVBoard(status->FirstBoard)->GetBoardNumber();
      
      int lastboard = (hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() > hv->GetHVBoard(status->LastBoard)->GetBoardNumber()) ?
	hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() : hv->GetHVBoard(status->LastBoard)->GetBoardNumber();
      
      if (status->FirstChain == status->LastChain)
	sprintf(outstr,"%s\nHV|C%d|B%d-%d> ",str,status->FirstChain,firstboard,lastboard);
      else 
	sprintf(outstr,"%s\nHV|C%d-%d|B%d-%d> ",str,status->FirstChain,status->LastChain,firstboard,lastboard); 
    }
  }
  else {

    if (status->NumHVBoards == 0) 
      sprintf(outstr,"HV> "); 
    else if (status->FirstBoard == status->LastBoard) {
      
      int board = hv->GetHVBoard(status->FirstBoard)->GetBoardNumber();
      
      if (status->FirstChain == status->LastChain) 
	sprintf(outstr,"HV|C%d|B%d> ",status->FirstChain,board); 
      else 
	sprintf(outstr,"HV|C%d-%d|B%d> ",status->FirstChain,status->LastChain,board); 
    }
    else {
      
      int firstboard = (hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() > hv->GetHVBoard(status->LastBoard)->GetBoardNumber()) ?
	hv->GetHVBoard(status->LastBoard)->GetBoardNumber() : hv->GetHVBoard(status->FirstBoard)->GetBoardNumber();
      
      int lastboard = (hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() > hv->GetHVBoard(status->LastBoard)->GetBoardNumber()) ?
	hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() : hv->GetHVBoard(status->LastBoard)->GetBoardNumber();
      
      if (status->FirstChain == status->LastChain)
	sprintf(outstr,"HV|C%d|B%d-%d> ",status->FirstChain,firstboard,lastboard);
      else 
	sprintf(outstr,"HV|C%d-%d|B%d-%d> ",status->FirstChain,status->LastChain,firstboard,lastboard);
    }


  }

  fflush(stdout);
  fputs(outstr, stdout);
  fflush(stdout);

}


void ProcessIO::PrintMessageToLog(char *str) {

  char outstr[MAX_COM_SIZE];

  if (status->NumHVBoards == 0) {
    sprintf(outstr,"HV|B> %s\n",str);
    log->LogWrite(outstr);
  }
  else if (status->FirstBoard == status->LastBoard) {
    
    int board = hv->GetHVBoard(status->FirstBoard)->GetBoardNumber();
    
    sprintf(outstr,"HV|B%d> %s\n",board,str);
    log->LogWrite(outstr);
  }
  else {
    
    int firstboard = (hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() > hv->GetHVBoard(status->LastBoard)->GetBoardNumber()) ?
      hv->GetHVBoard(status->LastBoard)->GetBoardNumber() : hv->GetHVBoard(status->FirstBoard)->GetBoardNumber();
    
    int lastboard = (hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() > hv->GetHVBoard(status->LastBoard)->GetBoardNumber()) ?
      hv->GetHVBoard(status->FirstBoard)->GetBoardNumber() : hv->GetHVBoard(status->LastBoard)->GetBoardNumber();

    sprintf(outstr,"HV|B%d-%d> %s\n",firstboard,lastboard,str);
   log->LogWrite(outstr);
  }

}


// Print message to screen, log file and socket
void ProcessIO::PrintMessageO(char *Format, ...) {

  char Textbuffer[MAX_COM_SIZE];

  va_list ArgumentPointer;  va_start(ArgumentPointer, Format);
  vsprintf(Textbuffer, Format, ArgumentPointer);

  fputs(Textbuffer, stdout);   fflush(stdout);  // Print to console
  
  if(status->Socket != -1)                      // Print to socket if open
    write(status->Socket,Textbuffer,strlen(Textbuffer)+1); // +1 to transmit '\0'
}


int ProcessIO::NBoards() {

  if (status->NumHVBoards==0)
    DoPrompt("no HV boards available!\n");

  return status->NumHVBoards;
}


int ProcessIO::InitializeHV() {

  int nb = 0;

  if (hv->GetNumberOfBoards())
    printf("Initialization:\n");

  for (int i=0;i<hv->GetNumberOfBoards();i++) {
    printf(" HV board %d (%s):\n",hv->GetHVBoard(i)->GetBoardNumber(),hv->GetHVBoard(i)->GetSerial());
    nb+=hv->GetHVBoard(i)->Init(status->Verbose);
  }

  return nb;
}


void ProcessIO::UpdateStatus(int i, unsigned char* rbuf) {

  if (status->IsUpdated[i])
    status->WC[0][i] = status->WC[1][i];
  else {
    status->WC[0][i] = hv->GetHVBoard(i)->DecodeWrap(rbuf)-1;
    status->IsUpdated[i] = TRUE;
  }
  
  status->WC[1][i] = hv->GetHVBoard(i)->DecodeWrap(rbuf);

  int d = (int)abs(status->WC[0][i]-status->WC[1][i]);

  status->isok[i] = (d==1 || d==7) ? 1 : 0;

  hv->GetHVBoard(i)->DecodeOC(status->OC[i], rbuf);
  status->MR[i] = hv->GetHVBoard(i)->DecodeReset(rbuf);

}


void ProcessIO::StartMonitor() {

  status->state = active;
  status->Stop  = FALSE;

  pthread_kill(status->HVMonitor, SIGUSR1);

}


void ProcessIO::StopMonitor() {

  status->state = stopped;
  status->Stop  = TRUE;

  pthread_kill(status->HVMonitor, SIGUSR1);

}


void ProcessIO::Monitor() {

  char str[MAX_COM_SIZE];

  for (int i=0;i<hv->GetNumberOfBoards() ;i++) {

    if ((hv->GetHVBoard(i))->GetStatus(stdout,rbuf,FALSE)!=1) {
      sprintf(str,"board %d error: could not read status - check timeout and status refresh rate...",hv->GetHVBoard(i)->GetBoardNumber());
      PrintMessage(str);      
    }
    else
      UpdateStatus(i,rbuf);
    
    if (status->MR[i]) {
      sprintf(str,"warning: manual reset of board %d!",hv->GetHVBoard(i)->GetBoardNumber());
      PrintMessage(str);      
      StopMonitor();
      ResetBoard(i);
      ReInitStatusOneBoard(status,i);
      StartMonitor();
    }
    
    if (!status->isok[i]) {
      sprintf(str,"error: wrap counter mismatch board %d (%d,%d)!",hv->GetHVBoard(i)->GetBoardNumber(),status->WC[0][i],status->WC[1][i]);
      PrintMessage(str);      
    }

    for (int j=0;j<MAX_NUM_CHAINS;j++) 
      if (status->OC[i][j]) {
	sprintf(str,"warning: overcurrent in chain %d of board %d!",j,hv->GetHVBoard(i)->GetBoardNumber());
	PrintMessage(str);      
	ResetBoard(i);
	ReInitStatusOneBoard(status,i);
      }
  }
}


void ProcessIO::ResetActiveBoards() {
  
  for (int i=status->FirstBoard;i<=status->LastBoard;i++) {
    if ((hv->GetHVBoard(i))->Reset(stdout,rbuf,status->Verbose)==1) {
      sprintf(str,"software reset done board %d\n",hv->GetHVBoard(i)->GetBoardNumber());
      DoPrompt(str);
      UpdateStatus(i,rbuf);
      sPrintStatus(status,str,i);
      DoPrompt(str);
    }
    else {
      sprintf(str,"board %d error: could not reset - check timeout and try again\n",hv->GetHVBoard(i)->GetBoardNumber());
	DoPrompt(str);
    }    
  }
  
}


void ProcessIO::ResetAllBoards() {
  
  for (int i=0;i<hv->GetNumberOfBoards();i++) {
    if ((hv->GetHVBoard(i))->Reset(stdout,rbuf,status->Verbose)==1) {
      sprintf(str,"software reset done board %d\n",hv->GetHVBoard(i)->GetBoardNumber());
      DoPrompt(str);
      UpdateStatus(i,rbuf);
      sPrintStatus(status,str,i);
      DoPrompt(str);
    }
    else {
      sprintf(str,"board %d error: could not reset - check timeout and try again\n",hv->GetHVBoard(i)->GetBoardNumber());
      DoPrompt(str);
    }    
  }
  
}


void ProcessIO::ResetBoard(int i) {
    
  if ((hv->GetHVBoard(i))->Reset(stdout,rbuf,status->Verbose)==1) {
    sprintf(str,"software reset done board %d",hv->GetHVBoard(i)->GetBoardNumber());
    PrintMessage(str);
    UpdateStatus(i,rbuf);
    sPrintStatus(status,str,i);
    PrintMessage(Chop(str));
  }
  else {
    sprintf(str,"board %d error: could not reset - check timeout and try again",hv->GetHVBoard(i)->GetBoardNumber());
    PrintMessage(str);
  }    
  
}


int ProcessIO::SaveHVSettings(char* filename) {

  char str[MAX_COM_SIZE];

  FILE *fptr;

  time_t time_now_secs;
  struct tm *time_now;
  
  time(&time_now_secs);
  time_now = localtime(&time_now_secs);

  if ((fptr = fopen(filename,"w")) == NULL) {
    sprintf(str,"error: could not open file \"%s\"\n", filename);
    return 0;
  }
  
  fprintf(fptr,"\n********** HV settings written by hvutil %s, %04d %02d %02d, %02d:%02d:%02d **********\n\n",
	  HV_CONTROL_VERSION, 
	  1900 + time_now->tm_year, 
	  1 + time_now->tm_mon,
	  time_now->tm_mday,
	  time_now->tm_hour,
	  time_now->tm_min,
	  time_now->tm_sec);
  
  for (int i=0;i<status->NumHVBoards;i++) {
    fprintf(fptr,"Device: %s\n\n",status->fUSBDevice[i]);
    
    for (int j=0;j<MAX_NUM_CHAINS;j++) {
      for (int k=0;k<4;k++) {
	for (int l=0;l<8;l++) 
	  fprintf(fptr,"%5d ",status->HV[i][j][k*8+l]);
	fprintf(fptr,"\n");
      }
      fprintf(fptr,"\n");
    }
  }
   
  fclose(fptr);

  return 1;
}


int ProcessIO::IsBoard(int board) {

  for (int i=0;i<MAX_NUM_HVBOARDS;i++)
    if (board == status->USBDeviceNumber[i])
      return 1;

  return 0;
}


int ProcessIO::GetBoardIdx(int board) {

  for (int i=0;i<MAX_NUM_HVBOARDS;i++)
    if (board == status->USBDeviceNumber[i])
      return i;

  return -1;
}
