#include <valarray>

#include "Dim.h"
#include "Event.h"
#include "Shell.h"
#include "StateMachineDim.h"
#include "Connection.h"
#include "Configuration.h"
#include "Console.h"

#include "tools.h"

#include "LocalControl.h"

#include "HeadersFTM.h"
#include "HeadersDrive.h"
#include "HeadersRateScan.h"
#include "HeadersRateControl.h"

namespace ba    = boost::asio;
namespace bs    = boost::system;
namespace dummy = ba::placeholders;

using namespace std;

// ------------------------------------------------------------------------

#include "DimDescriptionService.h"
#include "DimState.h"

// ------------------------------------------------------------------------

class StateMachineRateControl : public StateMachineDim//, public DimInfoHandler
{
private:
    bool fTriggerOn;

    vector<bool> fBlock;

    DimVersion fDim;
    DimDescribedState fDimFTM;
    DimDescribedState fDimRS;
    DimDescribedState fDimDrive;

    DimDescribedService fDimThreshold;

    float  fTargetRate;
    float  fTriggerRate;

    uint16_t fThresholdMin;
    uint16_t fThresholdReference;

    deque<pair<Time,float>> fCurrents;

    bool fVerbose;
    bool fCalibrateByCurrent;

    uint64_t fCounter;

    Time fCalibrationTimeStart;

    bool CheckEventSize(const EventImp &evt, size_t size)
    {
        if (size_t(evt.GetSize())==size)
            return true;

        if (evt.GetSize()==0)
            return false;

        ostringstream msg;
        msg << evt.GetName() << " - Received event has " << evt.GetSize() << " bytes, but expected " << size << ".";
        Fatal(msg);
        return false;
    }

    vector<uint16_t> fThresholds;

    void PrintThresholds(const FTM::DimStaticData &sdata)
    {
        //if (!fVerbose)
        //    return;

        if (fThresholds.size()==0)
            return;

        Out() << "Min. DAC=" << fThresholdMin << endl;

        int t=0;
        for (t=0; t<160; t++)
            if (sdata.fThreshold[t]!=fThresholds[t])
                break;

        if (t==160)
            return;

        for (int j=0; j<10; j++)
        {
            for (int k=0; k<4; k++)
            {
                for (int i=0; i<4; i++)
                    if (fThresholds[i+k*4+j*16]!=fThresholdMin)
                        Out() << setw(3) << fThresholds[i+k*4+j*16] << " ";
                    else
                        Out() << " -  ";
                Out() << "   ";
            }
            Out() << endl;
        }
        Out() << endl;
    }

    void Step(int idx, float step)
    {
        uint16_t diff = fThresholds[idx]+int16_t(truncf(step));
        if (diff<fThresholdMin)
            diff=fThresholdMin;

        if (diff==fThresholds[idx])
            return;

        if (fVerbose)
        {
            Out() << idx/40 << "|" << (idx/4)%10 << "|" << idx%4;
            Out() << (step>0 ? " += " : " -= ");
            Out() << fabs(step) << " (" << diff << ")" << endl;
        }

        const uint32_t val[2] = { idx,  diff };
        DimClient::sendCommandNB("FTM_CONTROL/SET_THRESHOLD", (void*)val, 8);

        fBlock[idx/4] = true;
    }

    void ProcessPatches(const FTM::DimTriggerRates &sdata)
    {

        // Caluclate Median and deviation
        vector<float> medb(sdata.fBoardRate, sdata.fBoardRate+40);
        vector<float> medp(sdata.fPatchRate, sdata.fPatchRate+160);

        sort(medb.begin(), medb.end());
        sort(medp.begin(), medp.end());

        vector<float> devb(40);
        for (int i=0; i<40; i++)
            devb[i] = fabs(sdata.fBoardRate[i]-medb[i]);

        vector<float> devp(160);
        for (int i=0; i<160; i++)
            devp[i] = fabs(sdata.fPatchRate[i]-medp[i]);

        sort(devb.begin(), devb.end());
        sort(devp.begin(), devp.end());

        double mb = (medb[19]+medb[20])/2;
        double mp = (medp[79]+medp[80])/2;

        double db = devb[27];
        double dp = devp[109];

        // If any is zero there is something wrong
        if (mb==0 || mp==0 || db==0 || dp==0)
            return;

        if (fVerbose)
        {
            Out() << "Patch: Median=" << mp << " Dev=" << dp << endl;
            Out() << "Board: Median=" << mb << " Dev=" << db << endl;
        }

        for (int i=0; i<40; i++)
        {
            if (fBlock[i])
            {
                fBlock[i] = false;
                continue;
            }

            int maxi = -1;

            const float dif = fabs(sdata.fBoardRate[i]-mb)/db;
            if (dif>5)
            {
                if (fVerbose)
                    Out() << "B" << i << ": " << dif << endl;

                float max = sdata.fPatchRate[i*4];
                maxi = 0;

                for (int j=1; j<4; j++)
                    if (sdata.fPatchRate[i*4+j]>max)
                    {
                        max = sdata.fPatchRate[i*4+j];
                        maxi = j;
                    }
            }

            for (int j=0; j<4; j++)
            {
                // For the noise pixel correct down to median+3*deviation
                if (maxi==j)
                {
                    // This is the step which has to be performed to go from
                    // a NSB rate of sdata.fPatchRate[i*4+j]


                    const float step = (log10(sdata.fPatchRate[i*4+j])-log10(mp+5*dp))/0.039;
                    //  * (dif-5)/dif
                    Step(i*4+j, step);
                    continue;
                }

                // For pixels below the meadian correct also back to median+3*deviation
                if (sdata.fPatchRate[i*4+j]<mp)
                {
                    const float step = (log10(sdata.fPatchRate[i*4+j])-log10(mp+3*dp))/0.039;
                    Step(i*4+j, step);
                    continue;
                }

                const float step =  -1.5*(log10(mp+dp)-log10(mp))/0.039;
                Step(i*4+j, step);
            }
        }
    }

    int ProcessCamera(const FTM::DimTriggerRates &sdata)
    {
        if (fCounter++==0)
            return GetCurrentState();

        // Caluclate Median and deviation
        vector<float> medb(sdata.fBoardRate, sdata.fBoardRate+40);

        sort(medb.begin(), medb.end());

        vector<float> devb(40);
        for (int i=0; i<40; i++)
            devb[i] = fabs(sdata.fBoardRate[i]-medb[i]);

        sort(devb.begin(), devb.end());

        double mb = (medb[19]+medb[20])/2;
        double db = devb[27];

        // If any is zero there is something wrong
        if (mb==0 || db==0)
        {
            Warn("The median or the deviation of all board rates is zero... cannot calibrate.");
            return GetCurrentState();
        }

        double avg = 0;
        int    num = 0;

        for (int i=0; i<40; i++)
        {
            if ( fabs(sdata.fBoardRate[i]-mb)<2.5*db)
            {
                avg += sdata.fBoardRate[i];
                num++;
            }
        }

        fTriggerRate = avg/num * 40;

        if (fVerbose)
        {
            Out() << "Board:  Median=" << mb << " Dev=" << db << endl;
            Out() << "Camera: " << fTriggerRate << " (" << sdata.fTriggerRate << ", n=" << num << ")" << endl;
            Out() << "Target: " << fTargetRate << endl;
        }

        if (sdata.fTriggerRate<fTriggerRate)
            fTriggerRate = sdata.fTriggerRate;

        // ----------------------

        /*
        if (avg>0 && avg<fTargetRate)
        {
            // I am assuming here (and at other places) the the answer from the FTM when setting
            // the new threshold always arrives faster than the next rate update.
            fThresholdMin = fThresholds[0];
            Out() << "Setting fThresholdMin to " << fThresholds[0] << endl;
        }
        */

        if (fTriggerRate>0 && fTriggerRate<fTargetRate)
        {
            fThresholds.assign(160, fThresholdMin);

            const RateControl::DimThreshold data = { fThresholdMin, fCalibrationTimeStart.Mjd(), Time().Mjd() };
            fDimThreshold.setQuality(0);
            fDimThreshold.Update(data);

            ostringstream out;
            out << setprecision(3);
            out << "Measured rate " << fTriggerRate << "Hz below target rate " << fTargetRate << "... mininum threshold set to " << fThresholdMin;
            Info(out);

            return RateControl::State::kGlobalThresholdSet;
        }

        // This is a step towards a threshold at which the NSB rate is equal the target rate
        // +1 to avoid getting a step of 0
        const float step = (log10(fTriggerRate)-log10(fTargetRate))/0.039 + 1;

        const uint16_t diff = fThresholdMin+int16_t(truncf(step));
        if (diff<=fThresholdMin)
        {
            const RateControl::DimThreshold data = { fThresholdMin, fCalibrationTimeStart.Mjd(), Time().Mjd() };
            fDimThreshold.setQuality(1);
            fDimThreshold.Update(data);

            ostringstream out;
            out << setprecision(3);
            out << "Next step would be 0... mininum threshold set to " << fThresholdMin;
            Info(out);

            return RateControl::State::kGlobalThresholdSet;
        }

        if (fVerbose)
        {
            //Out() << idx/40 << "|" << (idx/4)%10 << "|" << idx%4;
            Out() << fThresholdMin;
            Out() << (step>0 ? " += " : " -= ");
            Out() << step << " (" << diff << ")" << endl;
        }

        const uint32_t val[2] = { -1,  diff };
        DimClient::sendCommandNB("FTM_CONTROL/SET_THRESHOLD", (void*)val, 8);

        fThresholdMin = diff;

        return GetCurrentState();
    }

    int HandleStaticData(const EventImp &evt)
    {
        if (!CheckEventSize(evt, sizeof(FTM::DimStaticData)))
            return GetCurrentState();

        const FTM::DimStaticData &sdata = *static_cast<const FTM::DimStaticData*>(evt.GetData());
        fTriggerOn = sdata.HasTrigger();

        PrintThresholds(sdata);

        fThresholds.assign(sdata.fThreshold, sdata.fThreshold+160);

        return GetCurrentState();
    }

    int HandleTriggerRates(const EventImp &evt)
    {
        if (fThresholds.size()==0)
            return GetCurrentState();

        if (GetCurrentState()<=RateControl::State::kConnected ||
            GetCurrentState()==RateControl::State::kGlobalThresholdSet)
            return GetCurrentState();

        if (!CheckEventSize(evt, sizeof(FTM::DimTriggerRates)))
            return GetCurrentState();

        const FTM::DimTriggerRates &sdata = *static_cast<const FTM::DimTriggerRates*>(evt.GetData());

        if (GetCurrentState()==RateControl::State::kSettingGlobalThreshold && !fCalibrateByCurrent)
            return ProcessCamera(sdata);

        if (GetCurrentState()==RateControl::State::kInProgress)
            ProcessPatches(sdata);

        return GetCurrentState();
    }

    int HandleCalibratedCurrents(const EventImp &evt)
    {
        // Check if received event is valid
        if (!CheckEventSize(evt, (416+6)*4))
            return GetCurrentState();

        // Record only currents when the drive is tracking to avoid
        // bias from the movement
        if (fDimDrive.state()<Drive::State::kTracking)
            return GetCurrentState();

        // Get time and median current (FIXME: check N?)
        const Time &time = evt.GetTime();
        const float med  = evt.Get<float>(416*4+4+4);

        // Keep all median currents of the past 10 seconds
        fCurrents.push_back(make_pair(time, med));
        while (!fCurrents.empty())
        {
            if (time-fCurrents.front().first<boost::posix_time::seconds(10))
                break;

            fCurrents.pop_front();
        }

        // If we are not doing a calibration no further action necessary
        if (!fCalibrateByCurrent)
            return GetCurrentState();

        if (GetCurrentState()!=RateControl::State::kSettingGlobalThreshold)
            return GetCurrentState();

        // We want at least 8 values for averaging
        if (fCurrents.size()<8)
            return GetCurrentState();

        // Calculate avera and rms of median
        double avg = 0;
        double rms = 0;
        for (auto it=fCurrents.begin(); it!=fCurrents.end(); it++)
        {
            avg += it->second;
            rms += it->second*it->second;
        }
        avg /= fCurrents.size();
        rms /= fCurrents.size();

        rms = sqrt(rms-avg*avg);

        fThresholdMin = 36.0833*pow(avg, 0.638393)+184.037;
        fThresholdReference = fThresholdMin;
        fThresholds.assign(160, fThresholdMin);

        const RateControl::DimThreshold data = { fThresholdMin, fCalibrationTimeStart.Mjd(), Time().Mjd() };
        fDimThreshold.setQuality(0);
        fDimThreshold.Update(data);

        ostringstream out;
        out << setprecision(3);
        out << "Measured average current " << avg << "uA +- " << rms << "uA [N=" << fCurrents.size() << "]... mininum threshold set to " << fThresholdMin;
        Info(out);

        return RateControl::State::kGlobalThresholdSet;
    }

    int Calibrate()
    {
        if (!fTriggerOn)
        {
            Info("Physics trigger not enabled... CALIBRATE command ignored.");
            return RateControl::State::kGlobalThresholdSet;
        }

        const int32_t val[2] = { -1, fThresholdReference };
        Dim::SendCommand("FTM_CONTROL/SET_THRESHOLD", val);

        fThresholds.assign(160, fThresholdReference);

        fThresholdMin = fThresholdReference;
        fTriggerRate  = -1;
        fCounter      = 0;

        fCalibrateByCurrent = false;
        fCalibrationTimeStart = Time();

        ostringstream out;
        out << "Rate calibration started at a threshold of " << fThresholdReference << " with a target rate of " << fTargetRate << " Hz";
        Info(out);

        return RateControl::State::kSettingGlobalThreshold;
    }

    int CalibrateByCurrent()
    {
        if (!fTriggerOn)
        {
            Info("Physics trigger not enabled... CALIBRATE command ignored.");
            return RateControl::State::kGlobalThresholdSet;
        }

        cout << "-1-" << endl;

        if (fDimDrive.state()<Drive::State::kMoving)
            Warn("Drive not even moving...");

        cout << "-2-" << endl;

        fCounter = 0;
        fCalibrateByCurrent = true;
        fCalibrationTimeStart = Time();

        cout << "-3-" << endl;

        ostringstream out;
        out << "Rate calibration by current " << fThresholdReference << " with a target rate of " << fTargetRate << " Hz";
        Info(out);

        cout << "START" << endl;

        return RateControl::State::kSettingGlobalThreshold;
    }

    int StopRC()
    {
        return RateControl::State::kConnected;
    }

    int SetMinThreshold(const EventImp &evt)
    {
        if (!CheckEventSize(evt, 4))
            return kSM_FatalError;

        // FIXME: Check missing

        fThresholdReference = evt.GetUShort();

        return GetCurrentState();
    }

    int SetTargetRate(const EventImp &evt)
    {
        if (!CheckEventSize(evt, 4))
            return kSM_FatalError;

        fTargetRate = evt.GetFloat();

        return GetCurrentState();
    }

    int Print() const
    {
        cout << "PRINT" << endl;
        Out() << fDim << endl;
        Out() << fDimFTM << endl;
        Out() << fDimRS << endl;
        Out() << fDimDrive << endl;

        return GetCurrentState();
    }

    int SetVerbosity(const EventImp &evt)
    {
        if (!CheckEventSize(evt, 1))
            return kSM_FatalError;

        fVerbose = evt.GetBool();

        return GetCurrentState();
    }

    int Execute()
    {
        if (!fDim.online())
            return RateControl::State::kDimNetworkNA;

        // All subsystems are not connected
        if (fDimFTM.state()<FTM::State::kConnected || fDimDrive.state()<Drive::State::kConnected)
            return RateControl::State::kDisconnected;

        const bool inprog = fTriggerOn && fDimRS.state()<RateScan::State::kConfiguring;

        switch (GetCurrentState())
        {
        case RateControl::State::kSettingGlobalThreshold:
            return RateControl::State::kSettingGlobalThreshold;

        case RateControl::State::kGlobalThresholdSet:
            if (!inprog)
                return RateControl::State::kGlobalThresholdSet;

        case RateControl::State::kInProgress:
            if (inprog)
                return RateControl::State::kInProgress;
        }

        return RateControl::State::kConnected;
    }

public:
    StateMachineRateControl(ostream &out=cout) : StateMachineDim(out, "RATE_CONTROL"),
        fTriggerOn(false), fBlock(40),
        fDimFTM("FTM_CONTROL"),
        fDimRS("RATE_SCAN"),
        fDimDrive("DRIVE_CONTROL"),
        fDimThreshold("RATE_CONTROL/THRESHOLD", "S:1;D:1;D:1",
                      "Resulting threshold after calibration"
                      "|threshold[dac]:Resulting threshold from calibration"
                      "|begin[mjd]:Start time of calibration"
                      "|end[mjd]:End time of calibration")
    {
        // ba::io_service::work is a kind of keep_alive for the loop.
        // It prevents the io_service to go to stopped state, which
        // would prevent any consecutive calls to run()
        // or poll() to do nothing. reset() could also revoke to the
        // previous state but this might introduce some overhead of
        // deletion and creation of threads and more.

        fDim.Subscribe(*this);
        fDimFTM.Subscribe(*this);
        fDimRS.Subscribe(*this);
        fDimDrive.Subscribe(*this);

        Subscribe("FTM_CONTROL/TRIGGER_RATES")
            (bind(&StateMachineRateControl::HandleTriggerRates, this, placeholders::_1));
        Subscribe("FTM_CONTROL/STATIC_DATA")
            (bind(&StateMachineRateControl::HandleStaticData,   this, placeholders::_1));
        Subscribe("FEEDBACK/CALIBRATED_CURRENTS")
            (bind(&StateMachineRateControl::HandleCalibratedCurrents, this, placeholders::_1));

        // State names
        AddStateName(RateControl::State::kDimNetworkNA, "DimNetworkNotAvailable",
                     "The Dim DNS is not reachable.");

        AddStateName(RateControl::State::kDisconnected, "Disconnected",
                     "The Dim DNS is reachable, but the required subsystems are not available.");

        AddStateName(RateControl::State::kConnected, "Connected",
                     "All needed subsystems are connected to their hardware, no action is performed.");

        AddStateName(RateControl::State::kSettingGlobalThreshold, "Calibrating",
                     "A global minimum thrshold is currently determined.");

        AddStateName(RateControl::State::kGlobalThresholdSet, "GlobalThresholdSet",
                     "A global threshold has ben set, waiting for the trigger to be switched on.");

        AddStateName(RateControl::State::kInProgress, "InProgress",
                     "Rate control in progress.");

        AddEvent("CALIBRATE")
            (bind(&StateMachineRateControl::Calibrate, this))
            ("Start a search for a reasonable minimum global threshold");

        AddEvent("CALIBRATE_BY_CURRENT")
            (bind(&StateMachineRateControl::CalibrateByCurrent, this))
            ("Set the global threshold from the median current");

        AddEvent("STOP", RateControl::State::kSettingGlobalThreshold, RateControl::State::kGlobalThresholdSet, RateControl::State::kInProgress)
            (bind(&StateMachineRateControl::StopRC, this))
            ("Stop a calibration or ratescan in progress");

        AddEvent("SET_MIN_THRESHOLD", "I:1")
            (bind(&StateMachineRateControl::SetMinThreshold, this, placeholders::_1))
            ("Set a minimum threshold at which th rate control starts calibrating");

        AddEvent("SET_TARGET_RATE", "F:1")
            (bind(&StateMachineRateControl::SetTargetRate, this, placeholders::_1))
            ("Set a target trigger rate for the calibration");

        AddEvent("PRINT")
            (bind(&StateMachineRateControl::Print, this))
            ("Print current status");

        AddEvent("SET_VERBOSE", "B")
            (bind(&StateMachineRateControl::SetVerbosity, this, placeholders::_1))
            ("set verbosity state"
             "|verbosity[bool]:disable or enable verbosity for received data (yes/no), except dynamic data");

    }

    int EvalOptions(Configuration &conf)
    {
        fVerbose = !conf.Get<bool>("quiet");

        fThresholdReference = 300;
        fThresholdMin       = 300;
        fTargetRate         =  75;

        return -1;
    }
};

// ------------------------------------------------------------------------

#include "Main.h"

template<class T>
int RunShell(Configuration &conf)
{
    return Main::execute<T, StateMachineRateControl>(conf);
}

void SetupConfiguration(Configuration &conf)
{
    po::options_description control("Rate control options");
    control.add_options()
        ("quiet,q",       po_bool(),  "Disable printing more informations during rate control.")
       //("max-wait",   var<uint16_t>(150), "The maximum number of seconds to wait to get the anticipated resolution for a point.")
       // ("resolution", var<double>(0.05) , "The minimum resolution required for a single data point.")
        ;

    conf.AddOptions(control);
}

/*
 Extract usage clause(s) [if any] for SYNOPSIS.
 Translators: "Usage" and "or" here are patterns (regular expressions) which
 are used to match the usage synopsis in program output.  An example from cp
 (GNU coreutils) which contains both strings:
  Usage: cp [OPTION]... [-T] SOURCE DEST
    or:  cp [OPTION]... SOURCE... DIRECTORY
    or:  cp [OPTION]... -t DIRECTORY SOURCE...
 */
void PrintUsage()
{
    cout <<
        "The ratecontrol program is a keep the rate reasonable low.\n"
        "\n"
        "Usage: ratecontrol [-c type] [OPTIONS]\n"
        "  or:  ratecontrol [OPTIONS]\n";
    cout << endl;
}

void PrintHelp()
{
    Main::PrintHelp<StateMachineRateControl>();

    /* Additional help text which is printed after the configuration
     options goes here */

    /*
     cout << "bla bla bla" << endl << endl;
     cout << endl;
     cout << "Environment:" << endl;
     cout << "environment" << endl;
     cout << endl;
     cout << "Examples:" << endl;
     cout << "test exam" << endl;
     cout << endl;
     cout << "Files:" << endl;
     cout << "files" << endl;
     cout << endl;
     */
}

int main(int argc, const char* argv[])
{
    Configuration conf(argv[0]);
    conf.SetPrintUsage(PrintUsage);
    Main::SetupConfiguration(conf);
    SetupConfiguration(conf);

    if (!conf.DoParse(argc, argv, PrintHelp))
        return 127;

    if (!conf.Has("console"))
        return RunShell<LocalStream>(conf);

    if (conf.Get<int>("console")==0)
        return RunShell<LocalShell>(conf);
    else
        return RunShell<LocalConsole>(conf);

    return 0;
}
