/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MPedestalCam                                                            //
//                                                                         //
// Hold the Pedestal information for all pixels in the camera              //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MPedestalCam.h"
#include "MPedestalPix.h"

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"


ClassImp(MPedestalCam);

// --------------------------------------------------------------------------
//
// Default constructor. Creates a MPedestalPix object for each pixel
//
MPedestalCam::MPedestalCam(const char *name, const char *title)
{
    fName  = name  ? name  : "MPedestalCam";
    fTitle = title ? title : "Storage container for all Pedestal Information in the camera";

    fArray = new TClonesArray("MPedestalPix", 577);

    for (int i=0; i<577; i++)
        new ((*fArray)[i]) MPedestalPix;
}

// --------------------------------------------------------------------------
//
// Delete the array conatining the pixel pedest information
//
MPedestalCam::~MPedestalCam()
{
    delete fArray;
}

// --------------------------------------------------------------------------
//
// Set the size of the camera
//
inline void MPedestalCam::InitSize(const UInt_t i)
{
    fArray->ExpandCreate(i);
}

// --------------------------------------------------------------------------
//
// Get the size of the MPedestalCam
//
inline Int_t MPedestalCam::GetSize() const
{
    return fArray->GetEntries();
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
inline MPedestalPix &MPedestalCam::operator[](Int_t i)
{
    return *(MPedestalPix*)fArray->UncheckedAt(i);
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
inline MPedestalPix &MPedestalCam::operator[](Int_t i) const
{
    return *(MPedestalPix*)fArray->UncheckedAt(i);
}

// --------------------------------------------------------------------------
//
// Check if position i is inside bounds
//
Bool_t MPedestalCam::CheckBounds(Int_t i)
{
    return i < fArray->GetEntriesFast();
} 

void MPedestalCam::Clear(Option_t *o)
{
    fArray->ForEach(TObject, Clear)();
}

void MPedestalCam::Print(Option_t *o="") const
{
    *fLog << all << GetDescriptor() << ":" << endl;
    int id = 0;

    TIter Next(fArray);
    MPedestalPix *pix;
    while ((pix=(MPedestalPix*)Next()))
    {
        id++;

        if (!pix->IsValid())
            continue;

        *fLog << id-1 << ": ";
        *fLog << pix->GetMean() << " " << pix->GetSigma() << " ";
        *fLog << pix->GetMeanRms() << " " << pix->GetSigmaRms() << endl;
    }
}

Float_t MPedestalCam::GetMeanMin(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t minval = (*this)[0].GetMean();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetMean();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval < minval)
            minval = testval;
    }
    return minval;
}

Float_t MPedestalCam::GetMeanMax(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t maxval = (*this)[0].GetMean();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetMean();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval > maxval)
            maxval = testval;
    }
    return maxval;
}
