
/********************************************************************\

  Name:         HV.cc

  Created by:   Sebastian Commichau, November 2008
                commichau@phys.ethz.ch

  Contents:     Main class for HV supply
                
\********************************************************************/

#include "HV.h"
#include "ProcessIO.h" // Must be not in HV.h to avoid problem with declaring class ProcessIO

// Constructor
HVBoard::HVBoard(int DeviceNumber, char *DeviceName, class ProcessIO *PIO) {
   
  char Buffer[BUFFER_LENGTH];
  struct termios tio;
 
  m = PIO;

  SetTimeOut(m->fTimeOut);
  BoardNumber = DeviceNumber;
  BoardName = DeviceName;
  
  // Create DIM services
  snprintf(Buffer, sizeof(Buffer), SERVER_NAME"/NAME/ID%.2d", BoardNumber);
  Name = new DimService (Buffer, BoardName);

  for (int i=0; i<NUM_CHAINS; i++) {
    for (int j=0; j<NUM_CHANNELS; j++) {
      snprintf(Buffer, sizeof(Buffer), SERVER_NAME"/VOLT/ID%.2d/%.2d-%.3d", BoardNumber, i, j);
      BiasVolt[i][j] = new DimService (Buffer, HVV[i][j]);
    }
    Overcurrent[i] = false;
  }
  ResetButton = false;
  WrapOK = true;
  LastWrapCount = -1;
  ErrorCount = 0;

  ClearVoltageArrays();

  // Open device (do not warn on non-existing device)
  snprintf(Buffer, BUFFER_LENGTH, "/dev/%s",DeviceName);
  if ((fDescriptor = open(Buffer, O_RDWR|O_NOCTTY|O_NDELAY)) == -1) {
    if(errno != 2) m->PrintMessage(All, "Error: Could not open device %d/%s (%s)\n", DeviceNumber,DeviceName, strerror(errno));
    return;
  }

  // Get current serial port settings
  if (tcgetattr(fDescriptor, &tio) == -1) {
    m->PrintMessage(All, "Error: tcgetattr() failed (%d/%s)\n", errno, strerror(errno));
    return;   
  }

  // Set baudrate and raw mode
  if (cfsetspeed(&tio, BAUDRATE) == -1) m->PrintMessage(All, "Error: Could not set baud rate (%s)\n", strerror(errno));
  cfmakeraw(&tio);
  if (tcsetattr(fDescriptor, TCSANOW, &tio ) == -1) m->PrintMessage(All, "Error: tcsetattr() failed (%s)\n", strerror(errno));

  //  Synchronize HV board (if fails, closes device and sets fDescriptor to -2) 
  unsigned char wbuf = REG_STATUS;
  int trial = 0, ret;
  
  while(++trial<=3) {
    if((ret = Communicate(&wbuf, 1)) == 1) {
      Reset();
      return;
    }
    if (ret==0) break;
    wbuf = 0;
  }
  close(fDescriptor);
  fDescriptor = -2;
  return;
}

// Destructor (Resetting board)
HVBoard::~HVBoard() {
  if(fDescriptor >= 0) {
    Reset();
    close(fDescriptor);
  }

  delete Name;
  for (int i=0; i<NUM_CHAINS; i++) {
    for (int j=0; j<NUM_CHANNELS; j++) {
      delete BiasVolt[i][j];
    }
  }
}


// Communicate: Write and read from HV Board until fTimeOut has been reached 
//
// Returns: 0 error, 1 success, -1 fTimeOut exceeded
int HVBoard::Communicate(unsigned char* wbuf, int Bytes) {

  unsigned char rbuf;
  int ret;
  fd_set SelectDescriptor;
  
  // === Write data ===
  if ((ret=write(fDescriptor, wbuf, Bytes)) < Bytes) {
    if (ret == -1) m->PrintMessage(All, "Could not write data to HV board (%s)\n", strerror(errno));
    else m->PrintMessage(All, "Could write only %d of %d bytes to HV board\n", ret, Bytes);
    ErrorCount++;
    return 0;
  }

  // === Try to read until time-out ===
  FD_ZERO(&SelectDescriptor);   FD_SET(fDescriptor, &SelectDescriptor);
  struct timeval WaitTime = {(long) fTimeOut, (long) ((fTimeOut-(long) fTimeOut)*1e6)};
  if (select(fDescriptor+1, &SelectDescriptor, NULL, NULL, &WaitTime)==-1) {
    m->PrintMessage(All, "Error with select() (%s)\n", strerror(errno));
    return 0;
  }
  // Time-out expired?
  if (!FD_ISSET(fDescriptor, &SelectDescriptor)) return -1;

  // Read error?    
  if ((ret = read(fDescriptor, &rbuf, 1)) == -1) {
    m->PrintMessage(All, "Read error (%s)\n", strerror(errno));
    ErrorCount++;
    return 0;
  }

  // === Update status information ===
  if (LastWrapCount != -1) {
    if ((LastWrapCount+1)%8 == (rbuf & 0X07)) WrapOK = true;
    else WrapOK = false;
  }
  LastWrapCount = rbuf & 0X07;  
  for (int i=0; i<NUM_CHAINS; i++) Overcurrent[i] = (rbuf & (0X08 << i));
  ResetButton = (bool) (rbuf & 0X80);

  return 1;
}


/* Reset HV board - uses TRead() and has same return values */
int HVBoard::Reset() {
  
  unsigned char wbuf[] = {REG_RESET,0,0};
  int ret;
  
  if((ret = Communicate(wbuf, 3)) == 1) {
    ClearVoltageArrays();
    ErrorCount = 0;
  }  
  return ret;
}


/* Read status register - uses TRead() and has same return values */
int HVBoard::GetStatus() {

  unsigned char wbuf[] = {REG_STATUS,0,0};
  
  return Communicate(wbuf, 3);
}


/* Set high voltage - uses TRead() and has same return values */
int HVBoard::SetHV(int chain, unsigned int channel, int hv) {

  unsigned char wbuf[] = {0,0,0};
  int ret;
  
  if (hv<0 || hv > 0x3FFF) {
    m->PrintMessage("Error: Voltage out of range 0-0x3FFF\n"); 
    return 0;
  }
  
  switch (chain) {
    case 0: wbuf[0] = REG_HV0; break;
    case 1: wbuf[0] = REG_HV1; break;
    case 2: wbuf[0] = REG_HV2; break;
    case 3: wbuf[0] = REG_HV3; break;

    default : m->PrintMessage(" Error: Chain %d does not exist\n",chain); return 0;
  }
 
  // Assemble bytes
  wbuf[0] |= (unsigned char)((channel >> 2) & 0X00000007);  // Add address [A4-A3]
  wbuf[1] |= (unsigned char)((hv >> 8) & 0X000000FF);       // Add [D13-D8]
  wbuf[1] |= (unsigned char)((channel << 6)  & 0X000000C0); // Add [A1-A0]
  wbuf[2] |= (unsigned char)(hv & 0X000000FF);              // Add [D7-D0]
   
  if ((ret = Communicate(wbuf, 3)) == 1) HV[chain][channel] = hv;
  return ret;
}


// Set all voltages of board to zero
void HVBoard::ClearVoltageArrays() {

  for (int j=0; j<NUM_CHAINS; j++) {
    for (int k=0; k<NUM_CHANNELS; k++){
      HV[j][k] = 0;
      HVV[j][k] = 0.0;
      
      // Update DIM services
      BiasVolt[j][k]->updateService();
    }
  }
}


// Set time-out to wait for read
void HVBoard::SetTimeOut(double t) {

  if (t >= MIN_TIMEOUT && t <= MAX_TIMEOUT) fTimeOut  = t;
  else fTimeOut  = 1.0;
}
