
/********************************************************************\

  ProcessIO.cc

  Main class processing user input
 
  Sebastian Commichau, Sabrina Stark-Schneebeli, Oliver Grimm

\********************************************************************/

#include "ProcessIO.h"

static char* state_str[]    = {"active", "stopped", "n.a."};


ProcessIO::ProcessIO(const char *ConfigFile): 
    	    DimCommand((char *) SERVER_NAME"/Command", (char *) "C"),
    	    EvidenceServer(SERVER_NAME) {

  // Get program start time
  time (&StartTime);

  // Create DIM text output service
  Textout = new DimService (SERVER_NAME"/Textout", (char *) "");

  // Initialize status variables
  state     = active;
  Exit      = false;
  CmdFromSocket = false;
  Socket    = -1;
  
  NumHVBoards = 0;
  FirstBoard  = 0;
  LastBoard   = -1;
  FirstChain  = 0;
  LastChain   = NUM_CHAINS-1;

  // Get configuration data
  char *Boards = GetConfig(SERVER_NAME " Boards");
  fPixMapTable = GetConfig(SERVER_NAME " PixMapTable");
  fTimeOut = atof(GetConfig(SERVER_NAME " TimeOut"));
  fStatusRefreshRate = atof(GetConfig(SERVER_NAME " StatusRefreshRate"));
  fCCPort = atoi(GetConfig(SERVER_NAME " CCPort"));
  DACMin = atoi(GetConfig(SERVER_NAME " DACMin"));
  DACMax = atoi(GetConfig(SERVER_NAME " DACMax"));
  fHVCalibOffset = atof(GetConfig(SERVER_NAME " HVCalibOffset"));
  fHVCalibSlope = atof(GetConfig(SERVER_NAME " HVCalibSlope"));
  fHVMaxDiff = atoi(GetConfig(SERVER_NAME " HVMaxDiff"));

  if (fStatusRefreshRate < MIN_RATE || fStatusRefreshRate > MAX_RATE)  fStatusRefreshRate = 1;

  // Open HV devices
  fHVBoard = new HVBoard* [strlen(Boards)]; 
  char *Token = strtok(Boards, " \t");
  while (Token != NULL) {
    fHVBoard[NumHVBoards] = new HVBoard(NumHVBoards, Token, this);
    if(fHVBoard[NumHVBoards]->fDescriptor >= 0) {
       PrintMessage("Synchronized and reset board %d (%s)\n",NumHVBoards,fHVBoard[NumHVBoards]->BoardName);
       NumHVBoards++;
    }
    else {
      PrintMessage(All, "Failed to synchronize board %d (%s)\n",NumHVBoards,fHVBoard[NumHVBoards]->BoardName);
      delete fHVBoard[NumHVBoards];
    }
	Token = strtok(NULL, " \t");
  }
  LastBoard = NumHVBoards-1;
  
  // Create instances
  calib  = new HVCalib(this);
  pm 	 = new PixelMap(fPixMapTable);  
}


ProcessIO::~ProcessIO() {
  
  for (int i=0; i<NumHVBoards; i++) delete fHVBoard[i];
  delete[] fHVBoard;
    
  delete pm;	    delete calib;
  delete Textout;  
}


// Process user input
void ProcessIO::CommandControl(char *Command) {

  if (strlen(Command)==0) return;  // Ignore empty commands

  if(Command[0]=='.') {   // Shell command
    system(&(Command[1]));
    return;
  }

  for(int i=0; i<MAX_NUM_TOKEN; i++) Param[i] = "";  // All pointers point initially to empty string
  NParam = ParseInput(Command, Param);

  // Adress board
  if (Match(Param[0], "board")) {
    
    if (!NumHVBoards) return;

    // Print list of boards
    if (NParam == 1) {
      for (int i=0; i<NumHVBoards; i++) {
        PrintMessage("Board %d: %s\n", fHVBoard[i]->GetBoardNumber(), fHVBoard[i]->BoardName);
      }
      return;
    }

    //Select board(s)
    if (Match(Param[1],"all")) {
      FirstBoard = 0;
      LastBoard = NumHVBoards-1;
    } 
    else if (NParam==2 && atoi(Param[1])>=0 && atoi(Param[1])<NumHVBoards) {
      FirstBoard = atoi(Param[1]);
      LastBoard = FirstBoard;
    }
    else if (NParam==3 && atoi(Param[1])>=0 && atoi(Param[1])<NumHVBoards && 
           atoi(Param[2])>0 && atoi(Param[2])<NumHVBoards) {
      FirstBoard = atoi(Param[1]);
      LastBoard = atoi(Param[2]);
    }
    else PrintMessage("Cannot address board(s), out of range.\n");

    return;
  } 


  // Adress chains
  else if (Match(Param[0], "chain")) {
    
    if (!NumHVBoards) return;
    
    if (Match(Param[1],"all")) {
      FirstChain = 0;
      LastChain = 3;
    } 
    else if (NParam==2 && atoi(Param[1])>=0 && atoi(Param[1])<NUM_CHAINS) {
      FirstChain = atoi(Param[1]);
      LastChain = FirstChain;
    }
    else if (NParam==3 && atoi(Param[1])>=0 && atoi(Param[1])<NUM_CHAINS && 
           atoi(Param[2])>0 && atoi(Param[2])<NUM_CHAINS) {
      FirstChain = atoi(Param[1]);
      LastChain = atoi(Param[2]);
    }
    else PrintMessage("Cannot address chain(s), out of range.\n");
        
    return;
  } 


  // Print HV utility configuration
  else if (Match(Param[0], "config")) {
    PrintMessage( " Pixel map table:   %s\n"
    	    	  " %d USB devices:\n", fPixMapTable,
		  NumHVBoards);  
    for (int i=0; i<NumHVBoards; i++) { 
      PrintMessage(" Board %d: %s\n", i, fHVBoard[i]->BoardName);
    }
    PrintMessage( " TimeOut:           %.2f s\n"
    	    	  " StatusRefreshRate: %.2f Hz\n"
    	    	  " CCPort:            %d\n"
    	    	  " DACMin value:      %d\n"
    	    	  " DACMax value:      %d\n"
    	    	  " HVCalibOffset :    %f\n"
    	    	  " HVCalibSlope :     %f\n"
    	    	  " HVMaxDiff :        %u\n", fTimeOut,
		  fStatusRefreshRate, fCCPort, DACMin,
		  DACMax, fHVCalibOffset, fHVCalibSlope,
		  fHVMaxDiff);

    return;
  }


  // Print help
  if (Match(Param[0], "help")) {
    PrintMessage(" board <i>|<i> <j>|<all>           Address board i, boards i-j or all boards or list boards\n"
    	" chain <i>|<i> <j>|<all>           Address chain i, chains i-j or all chains\n"
    	" hv <PXL id>|<ch>|<all> <v>        Set bias of pixel ID, ch. or all ch. of active chain(s)/board(s)\n"
    	" hvdiff <PXL id>|<ch>|<all> <diff> Change bias by <diff>\n"
		" status [dac]                      Show status information (DAC values if requested)\n"
		" config                            Print configuration\n"
		" load <file>                       Load and set bias settings from <file>\n"
		" save <file>                       Save current bias settings to [file]\n"
		" exit                              Exit program\n"
		" rate <rate>                       Set status refresh rate to <rate> [Hz]\n"
		" timeout <time>                    Set timeout to return from read to <time> [s]\n"
		" reset                             Reset active bias board\n"
		" start                             Start bias status monitor\n"
		" stop                              Stop bias status monitor - not recommended!\n"
		" uptime                            Get program uptime [h:m:s]\n"
		" help                              Print help\n"
		" .<cmd>                   	     Execute shell command <cmd>");

    return;
  } 


  // Set new bias voltage (if no boards available, simply returns OK)
  // First reponse to socket should be 'OK' if no error occurred.
  if (Match(Param[0], "hv") || Match(Param[0], "hvdiff")) {

    int hvoltage, DACValue, Errors=0, Board=-1, Chain=-1, Channel=-1;
    double hvoltageV;
    bool SetDac;
    
    // Need two or parameters
    if (NParam<3 || NParam>4) {
      PrintMessage("Usage: hv/hvdiff <channel>|<all> <voltage> [dac]\n");
      return;
    }

    // Evaluate voltage parameter
    if (NParam==4 && Match(Param[3], "dac")) {
      SetDac = true;
      if (!ConvertToInt(Param[2], &hvoltage)) {
        PrintMessage("Error: Wrong number format for DAC voltage setting\n");
        return;
      } 
    }
    else {
      SetDac = false;
      if (!ConvertToDouble(Param[2], &hvoltageV)) {
        PrintMessage("Error: Wrong number format for voltage setting\n");
        return;
      }
    } 

    // Evaluate pixel or channel parameter    
    if(pm->Pixel_to_HVboard(Param[1]) != 999999999) {
      Board = pm->Pixel_to_HVboard(Param[1]);
      Chain = pm->Pixel_to_HVchain(Param[1]);
      Channel = pm->Pixel_to_HVchannel(Param[1]);
    }
    else if (!Match(Param[1], "all") && !ConvertToInt(Param[1], &Channel)) {
      PrintMessage("Error: Wrong channel identification\n");
      return;
    }

    for (int i=FirstBoard; i<=LastBoard; i++) {
      if (i!=Board && Board!=-1) continue;
    for (int j=FirstChain; j<=LastChain; j++) {
        if (j!=Chain && Chain!=-1) continue;
	for (int k=0; k<NUM_CHANNELS; k++) {
	  if (k!=Channel && Channel!=-1) continue;

	  // hvdiff is ignored if DAC value is zero
	  if ((strlen(Param[0])>2 || isdigit(*Param[2])==0) && (fHVBoard[i]->HV[j][k] == 0)) continue;

	  // Determine new DAC values
	  if (!SetDac){
		if (strlen(Param[0])>2 || isdigit(*Param[2])==0) fHVBoard[i]->HVV[j][k] += hvoltageV; // hvdiff
	    else fHVBoard[i]->HVV[j][k] = hvoltageV;
	    DACValue = calib->HVToDAC(fHVBoard[i]->HVV[j][k], i, j, k);
	  }
	  else {
	    if(strlen(Param[0])>2 || isdigit(*Param[2])==0) DACValue = fHVBoard[i]->HV[j][k] + hvoltage; // hvdiff
	    else DACValue = hvoltage;
	  }
	  
	  // Set new voltage (if DAC value, update calibrated value)
	  if (!RampVoltage(DACValue, i, j, k)) Errors++;
	  if (SetDac) fHVBoard[i]->HVV[j][k] = calib->DACToHV(fHVBoard[i]->HV[j][k], i, j, k);

      // Update DIM services
      fHVBoard[i]->BiasVolt[j][k]->updateService();

	} // Channels	
    } // Chains
    } // Boards

    if (Errors > 0) PrintMessage("ERROR - Errors on %d channel(s) occurred\n", Errors);
    else PrintMessage("OK - no errors\n");
    
    return;
  }


  // Load HV settings from file
  else if (Match(Param[0], "load")) {

    char Buffer[MAX_COM_SIZE];
    int NBoards = 0, Errors = 0, Chain, Channel;
    unsigned int DACValue;
    FILE *File;
    
    if (NParam != 2) {
      PrintMessage("Usage: load <file>\n");
      return;
    }
    
    if ((File=fopen(Param[1], "r")) == NULL) {
      PrintMessage("Error: Could not open file '%s' (%s)\n", Param[1], strerror(errno));
      return;
    }
	
    while (fgets(Buffer, sizeof(Buffer), File) != NULL) {
      for (int Board=0; Board<NumHVBoards; Board++) {
	if (Match(fHVBoard[Board]->BoardName, Buffer)) {
	  PrintMessage("Found bias settings for board %d (%s)\n\r",fHVBoard[Board]->GetBoardNumber(), fHVBoard[Board]->BoardName);

	  Chain = 0;  Channel = 0;
	  while (fscanf(File, "%u", &DACValue)==1 && Chain<NUM_CHAINS) {
    	    if (!RampVoltage(DACValue, Board, Chain, Channel)) {
	      Errors++;
	      PrintMessage("Error: Could not ramp chain %d, channel %d\n", Chain, Channel);
    	    }
	    else {
	      PrintMessage("Ramped chain %d, channel %d to %u (%.2f V)                         \r",
	      	 Chain,Channel,DACValue,calib->DACToHV(DACValue,Board,Chain,Channel));
    	    }
	    fHVBoard[Board]->HVV[Chain][Channel] = calib->DACToHV(fHVBoard[Board]->HV[Chain][Channel], Board, Chain, Channel);

	    // Update DIM services
    	    fHVBoard[Board]->BiasVolt[Chain][Channel]->updateService();

	    if(++Channel == NUM_CHANNELS) {
	      Chain++;
	      Channel = 0;
	    }
	  }
	  if (ferror(File) != 0) {
	    PrintMessage("Error reading DAC value from file, terminating. (%s)\n",strerror(errno));
    	    return;
	  }
	  else PrintMessage("\nFinished updating board\n");
    	  NBoards++;
	}
      } // Loop over boards
    } // while()
    	    
    if (NBoards != NumHVBoards) {
      PrintMessage("Warning: Could not load bias settings for all connected HV boards\n");
    }
    else if (Errors == 0) PrintMessage("Success: Read bias settings for all connected HV boards\n");
    if (Errors != 0) PrintMessage("Warning: Errors on %d channel(s) occurred\n", Errors);
    
    if (fclose(File) != 0) PrintMessage("Error: Could not close file '%s'\n",Param[1]);

    return;    
  }
	   

  // Set status refresh rate
  if (Match(Param[0], "rate")) {

    double Rate;
    
    if (NParam != 2) {
      PrintMessage("Usage: rate <Hz>\n");
      return;
    }
    
    if (!ConvertToDouble(Param[1], &Rate)) {
       PrintMessage("Error: Wrong number format\n");
       return;   
    }

    // Check limits
    if (Rate<MIN_RATE || Rate>MAX_RATE) {
      PrintMessage("Refresh rate out of range (min: %.2f Hz, max: %.2f Hz)\n", MIN_RATE, MAX_RATE);
      return;
    }

    fStatusRefreshRate = Rate;
    PrintMessage("Refresh rate set to %.2f Hz\n", fStatusRefreshRate);

    return;
  }
  
  // Reset
  if (Match(Param[0], "reset")) {

    if (!NumHVBoards) return;
    for (int i=FirstBoard; i<=LastBoard; i++) ResetBoard(i);
    return;
  }


  // Save HV settings of all boards
  else if (Match(Param[0], "save")) {

    FILE *File;
    time_t time_now_secs;
    struct tm *Time;

    if (NParam != 2) {
      PrintMessage("Usage: save <Filename>\n");
      return;
    }
  
    time(&time_now_secs);
    Time = localtime(&time_now_secs);

    if ((File = fopen(Param[1], "w")) == NULL) {
      PrintMessage("Error: Could not open file '%s' (%s)\n", Param[1], strerror(errno));
      return;
    }
  
    fprintf(File,"********** Bias settings, %04d %02d %02d, %02d:%02d:%02d **********\n\n",
	    1900 + Time->tm_year, 1 + Time->tm_mon,
	    Time->tm_mday, Time->tm_hour, Time->tm_min, Time->tm_sec);
  
    for (int i=0; i<NumHVBoards; i++) {
      fprintf(File, "%s\n\n", fHVBoard[i]->BoardName);

      for (int j=0; j<NUM_CHAINS; j++) {
	for (int k=0; k<NUM_CHANNELS; k++) fprintf(File,"%5d ",fHVBoard[i]->HV[j][k]);
	fprintf(File, "\n");
      }
      fprintf(File, "\n");
    }

    if (fclose(File) != 0) {
      PrintMessage("Error: Could not close file '%s' (%s)\n", Param[1], strerror(errno));  
    }

    return;
  }


  // Start monitoring
  else if (Match(Param[0], "start")) {

    state = active;
    pthread_kill(HVMonitor, SIGUSR1);
    PrintMessage(All, "Status monitoring activated\n");
    return;  
  }

 
  // Print status
  else if (Match(Param[0], "status")) {
    
    PrintMessage("\n Status monitor: %s\n", state_str[state]);
    PrintMessage(" Status refresh rate [Hz]: %.2f\n", fStatusRefreshRate);
    PrintMessage(" Socket state: %s\n", Socket==-1 ? "disconnected":"connected");
    PrintMessage(" Total number of boards: %d\n", NumHVBoards);
    PrintMessage(" Active boards: %d\n\n", LastBoard - FirstBoard + 1);

    for (int i=FirstBoard; i<=LastBoard; i++) {
      PrintMessage(" BOARD %d (%s)   Wrap counter: %s (%d)  Manual reset: %s\n    Time-out: %.2f s  Error count: %d\n\n",
	fHVBoard[i]->GetBoardNumber(), fHVBoard[i]->BoardName,
	fHVBoard[i]->WrapOK ? "ok":"error",
	fHVBoard[i]->LastWrapCount, 
	fHVBoard[i]->ResetButton ? "yes" : "no",
	fHVBoard[i]->fTimeOut,
	fHVBoard[i]->ErrorCount);

      for (int j=FirstChain; j<=LastChain; j++) {
	PrintMessage("  CHAIN %d     Over-current: %s\n", j, fHVBoard[i]->Overcurrent[j] ? "yes" : "no");
	for (int k=0;k<4;k++) {
          PrintMessage("\r");
	  for (int l=0;l<8;l++) {
	    if(NParam == 2) PrintMessage("%5d ",fHVBoard[i]->HV[j][k*8+l]);
    	    else PrintMessage("%#5.2f ",fHVBoard[i]->HVV[j][k*8+l]);
    	  }
	  PrintMessage("\n");
	}
      }
    }

    return;
  } 


  // Stop monitoring
  else if (Match(Param[0], "stop")) {

    state = stopped;
    pthread_kill(HVMonitor, SIGUSR1);
    PrintMessage(All, "Status monitor stopped\n");

    return;
  }


  // Set timeout to return from read
  if (Match(Param[0], "timeout")) {

    double Timeout;
    
    if (!NumHVBoards) return;

    if (NParam != 2) {
      PrintMessage("Usage: timeout <secs>\n");
      return;
    }
    
    if (!ConvertToDouble(Param[1], &Timeout)) {
       PrintMessage("Error: Wrong number format\n");
       return;   
    }

    for (int i=0; i<NumHVBoards; i++) fHVBoard[i]->SetTimeOut(Timeout);
    PrintMessage("Timeout set to %.2f s for all boards\n", Timeout);
    
    return;
  }

    
  // Print uptime
  if (Match(Param[0], "uptime")) {
    time_t ActualT;
    time (&ActualT);

    PrintMessage("%d:%02d:%02d\n", (int) difftime(ActualT, StartTime)/3600, ((int) difftime(ActualT, StartTime)/60)%60, (int) difftime(ActualT, StartTime)%60);

    return;
  } 


  // Exit program
  else if(Match(Param[0], "exit")) {

    if (CmdFromSocket) {
      PrintMessage("Exit command not allowed over socket.\n");
      return;
    }  	

    Exit = true;
    pthread_kill(HVMonitor, SIGUSR1);
    pthread_kill(SocketThread, SIGUSR1);
 
    return;
  }
  
  PrintMessage("Unknown command '%s'\n", Param[0]);
 
  return;
}


// Print message to selected target
void ProcessIO::PrintMessage(MsgTarget Target, const char *Format, ...) {
  va_list ArgumentPointer;
  va_start(ArgumentPointer, Format); 
  DoPrintMessage(Format, ArgumentPointer, Target);
  va_end(ArgumentPointer);
}

// Print message to log file, and screen or socket (depending on command origin)
void ProcessIO::PrintMessage(const char *Format, ...) {
  va_list ArgumentPointer;
  va_start(ArgumentPointer, Format);
  if (CmdFromSocket) DoPrintMessage(Format, ArgumentPointer, Client);
  else DoPrintMessage(Format, ArgumentPointer, Console);
  va_end(ArgumentPointer);
}

// Function doing the actual printing work
// Be careful with overloading variadic functions!
void ProcessIO::DoPrintMessage(const char *Format, va_list ArgumentPointer, MsgTarget Target) {

  static char Textbuffer[MAX_COM_SIZE]; // static because of DIM service

  memset(Textbuffer, 0, sizeof(Textbuffer));  
  vsnprintf(Textbuffer, sizeof(Textbuffer), Format, ArgumentPointer);
  
  // Print to console
  if(Target & Console) {
    if(strlen(Textbuffer)>0 && Textbuffer[strlen(Textbuffer)-1]=='\n') {
      printf("\r%s%s", Textbuffer, Prompt);   // New prompt
    }
    else printf("%s", Textbuffer);
    fflush(stdout);
  }

  // Send to DIM service
  Textout->updateService(Textbuffer); 

  // Print to socket
  if((Target & Client) && Socket!=-1) {
    write(Socket, Textbuffer, strlen(Textbuffer));
  }

  // Send to log
  if(Target & Log) {
    char *Buf;
    if (asprintf(&Buf, "%s %s", SERVER_NAME, Textbuffer) != -1) { 
      DimClient::sendCommand("DColl/Log", Buf);
      free(Buf);
    }
    else DimClient::sendCommand("DColl/Log", SERVER_NAME" asprintf() failed"); 
  }
}


// Ramp to new voltage with maximum step size given in fHVMaxDiff
// No ramping when decreasing voltage
bool ProcessIO::RampVoltage(unsigned int Target, int Board, int Chain, int Channel) {

  while (fHVBoard[Board]->HV[Chain][Channel] != (int) Target) {	  
    int Diff = Target - fHVBoard[Board]->HV[Chain][Channel];
    if (Diff > (int) fHVMaxDiff) Diff = fHVMaxDiff;

    if (fHVBoard[Board]->SetHV(Chain, Channel, fHVBoard[Board]->HV[Chain][Channel]+Diff) != 1) {
      State(ERROR, "Could not set bias of board %d, chain %d, channel %d. Skipping channel\n",fHVBoard[Board]->GetBoardNumber(),Chain,Channel);
      return false;
    }
  }

  return true;
}


// Check board status (ignore board if it has more than 10 read/write errors)
void ProcessIO::Monitor() {

  static bool Warned = false;
  
  for (int i=0; i<NumHVBoards; i++) {
    if (fHVBoard[i]->ErrorCount > 10) {
      if (!Warned) {
        Warned = true;
        PrintMessage(All, "Warning: Some board has many read/write errors, status monitor disabled\n");
        State(WARN, "Warning: Some board has many read/write errors, status monitor disabled\n");
      }
      continue;
    }

    if (fHVBoard[i]->GetStatus() != 1) {
      PrintMessage(All, "Error: Monitor could not read status of board %d\n", fHVBoard[i]->GetBoardNumber());
      State(ERROR, "Error: Monitor could not read status of board %d\n", fHVBoard[i]->GetBoardNumber());
    }
    
    if (fHVBoard[i]->ResetButton) {
      PrintMessage(All, "Manual reset of board %d\n",fHVBoard[i]->GetBoardNumber());
      State(INFO, "Manual reset of board %d\n",fHVBoard[i]->GetBoardNumber());
      ResetBoard(i);
    }
    
    if (!fHVBoard[i]->WrapOK) {
      PrintMessage(All, "Error: Wrap counter mismatch board %d\n",fHVBoard[i]->GetBoardNumber());
      State(ERROR, "Error: Wrap counter mismatch board %d\n",fHVBoard[i]->GetBoardNumber());
    }

    for (int j=0; j<NUM_CHAINS; j++) {
      if (fHVBoard[i]->Overcurrent[j]) {
		PrintMessage(All, "Warning: Overcurrent in chain %d of board %d\n",j,fHVBoard[i]->GetBoardNumber());
		State(WARN, "Warning: Overcurrent in chain %d of board %d\n",j,fHVBoard[i]->GetBoardNumber());
		ResetBoard(i);
      }
    }
    
  }
}


// Send reset to board and clear voltage arrays
void ProcessIO::ResetBoard(int i) {
    
  if (fHVBoard[i]->Reset() == 1) {
    PrintMessage("Reset of board %d\n", fHVBoard[i]->GetBoardNumber());
    PrintBoardStatus(i);
  }
  else PrintMessage("Error: Could not reset board %d\n",fHVBoard[i]->GetBoardNumber());
}


// Print current board status
void ProcessIO::PrintBoardStatus(int i) {

  PrintMessage("Status board %d (%s): MR %s OC0 %s OC1 %s OC2 %s OC3 %s WC %s (%d)\n",
    	  fHVBoard[i]->GetBoardNumber(), fHVBoard[i]->BoardName,
	  fHVBoard[i]->ResetButton ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[0] ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[1] ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[2] ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[3] ? "yes" : "no",
	  fHVBoard[i]->WrapOK ? "ok":"error", fHVBoard[i]->LastWrapCount);
}


// Parse command line for white space and double-quote separated tokens 
int ProcessIO::ParseInput(char* Command, const char *Param[]) {
  int Count=0;
   
  while(Count<MAX_NUM_TOKEN) {
    while (isspace(*Command)) Command++; // Ignore initial white spaces
    if(*Command=='\0') break;
    if (*Command == '\"') {
      Param[Count] = ++Command;
      while(*Command!='\"' && *Command!='\0') Command++;
    }
    else {
      Param[Count] = Command;
      while(!isspace(*Command) && *Command!='\0') Command++;
    }
    if(*Command != '\0') *Command++ = '\0';
    Count++;
  }
  return Count;
}

//
// Command handling (the only command is 'BIAS/Command')
//
void ProcessIO::commandHandler() {

    // Log command
    PrintMessage(Log, "DIM> %s\n", getString());

    // Process command     
    pthread_mutex_lock(&control_mutex);
    CommandControl(getString());
    pthread_mutex_unlock(&control_mutex);
}

// Check if two strings match (min 1 character must match)
bool Match(const char *str, const char *cmd) {
  return strncasecmp(str,cmd,strlen(str)==0 ? 1:strlen(str)) ? false:true;
}


// Convert string to double
// Returns false if conversion did not stop on whitespace or EOL character
bool ConvertToDouble(const char *String, double *Result) {

  char *EndPointer;
  
  *Result = strtod(String, &EndPointer);
  if(!isspace(*EndPointer) && *EndPointer!='\0') return false;
  return true;
}


// Convert string to int
// Returns false if conversion did not stop on whitespace or EOL character
bool ConvertToInt(const char *String, int *Result) {

  char *EndPointer;
  
  *Result = (int) strtol(String, &EndPointer, 0);
  if(!isspace(*EndPointer) && *EndPointer!='\0') return false;
  return true;
}
