
/* ============================================================ 

Edd - Evidence Data Display

Qt-based graphical user interface for the Evidence contron system

Edd_Indicator changes its background colour in case it display
a DIM status service

February 2010, Oliver Grimm

============================================================ */

#include "Edd.h"

Qt::GlobalColor LineColors[] = {Qt::black, Qt::blue, Qt::red, Qt::green, Qt::white,
   	Qt::darkRed, Qt::darkGreen, Qt::darkBlue, Qt::cyan,
    	Qt::darkCyan, Qt::magenta, Qt::darkMagenta,
	Qt::gray, Qt::darkGray, Qt::lightGray};

class GUI *Handler;


// History chooser function (opens plot for numeric data, TextHist for all other)
QWidget *OpenHistory(char *Service) {

  char *Name, *Format;
  DimBrowser Browser;
  
  //if (Browser.getServices(Service) == 0) return NULL;
  Browser.getServices(Service);
  if (Browser.getNextService(Name, Format) != DimSERVICE) return NULL;
  
  if (strlen(Format) == 1 && *Format != 'C') return new Edd_Plot(Service);
  else return new Edd_TextHist(Service);
}


//////////////////////////////////////////
// Text display for arbitary DIM service//
//////////////////////////////////////////

// Constructor
Edd_Indicator::Edd_Indicator(QString DIMService, QWidget *P): QLineEdit(P) {

  // Widget properties
  setReadOnly(true);
  setMaximumWidth(100);
  ShowAsTime = false;
  
  // Connect to DIM handler
  if (connect(Handler, SIGNAL(YEP(DimInfo*, int, QByteArray, QString)), SLOT(Update(DimInfo*, int, QByteArray, QString))) == false) {
    printf("Failed connection for %s\n", DIMService.toAscii().data());
  }

  // Context menu
  Menu = new QMenu(this);
  Menu->addAction("Open new history", this, SLOT(MenuOpenHistory()));
  Menu->addAction("Copy service", this, SLOT(MenuCopyService()));
  Menu->addAction("Copy data", this, SLOT(MenuCopyData()));

  // DIM client
  Data = new DimStampedInfo(DIMService.toAscii().data(), INT_MAX, NO_LINK, Handler);
}

// Destructor
Edd_Indicator::~Edd_Indicator() {
  delete Data;
}

// Update widget
void Edd_Indicator::Update(DimInfo *Info, int Time, QByteArray Array, QString Text) {

  if (Info != Data) return;
    
  QPalette Pal = palette();  

  // Check if service available
  if (Time == -1) {
    setText("n/a");
	setStatusTip(QString("%1:  unavailable").arg(Info->getName()));
    Pal.setColor(QPalette::Base, Qt::lightGray);
  }
  else {
    // Backgound colour determined by last byte
    switch (Array[Array.size()]) {
      case 0:  Pal.setColor(QPalette::Base, Qt::white); break;
      case 1:  Pal.setColor(QPalette::Base, Qt::yellow); break;
      case 2:  Pal.setColor(QPalette::Base, Qt::red); break;
      case 3:  Pal.setColor(QPalette::Base, Qt::red); break;
      default: break;
    }
	
	if (!ShowAsTime) setText(Text);
	else setText(QDateTime::fromTime_t(Text.toInt()).toString());
	
    // Update status tip
    setStatusTip(QString("%1:  Last update %2  Format '%3'").arg(Info->getName(), QDateTime::fromTime_t(Time).toString()).arg(Info->getFormat()));
  }
  
  setPalette(Pal);
}

// Open plot if mouse release within widget
void Edd_Indicator::mouseReleaseEvent(QMouseEvent *Event) {

  if (Event->button()!=Qt::LeftButton || !contentsRect().contains(Event->pos())) return;

  // Check if last history plot still open, then raise
  foreach (QWidget *Widget, QApplication::allWidgets()) {
    if (Widget == LastPlot) {
      Widget->activateWindow();
      Widget->raise();
      return;
    }
  }

  // If not, open new plot
  Edd_Indicator::MenuOpenHistory();
}

// Handling of mouse press event: Register start position for drag
void Edd_Indicator::mousePressEvent(QMouseEvent *Event) {

  if (Event->button() == Qt::LeftButton) dragStart = Event->pos();
}

// Handling of dragging (Drag and MimeData will be deleted by Qt)
void Edd_Indicator::mouseMoveEvent(QMouseEvent *Event) {

  if ((Event->buttons() & Qt::LeftButton) == 0) return;
  if ((Event->pos()-dragStart).manhattanLength() < QApplication::startDragDistance()) return;

  QDrag *Drag = new QDrag(this);
  QMimeData *MimeData = new QMimeData;
  MimeData->setText(QString(Data->getName()));
  Drag->setMimeData(MimeData);
  Drag->exec();
}

//
// Opening context menu
//
void Edd_Indicator::contextMenuEvent(QContextMenuEvent *Event) {

  Menu->exec(Event->globalPos());
}

// Menu: Open history plot
void Edd_Indicator::MenuOpenHistory() {
  
  LastPlot = OpenHistory(Data->getName());
  if (LastPlot != NULL) LastPlot->show();
}

// Menu: Copy service name
void Edd_Indicator::MenuCopyService() {
  
  QApplication::clipboard()->setText(QString(Data->getName()));
}

// Menu: Copy data
void Edd_Indicator::MenuCopyData() {
  
  QApplication::clipboard()->setText(text());
}

//////////////////////////////////
// History plot for DIM service //
//////////////////////////////////

//
// Constructor
//
Edd_Plot::Edd_Plot(QString DIMService, QWidget *P): QwtPlot(P), EvidenceHistory() {

  setAcceptDrops(true);
  setAttribute(Qt::WA_DeleteOnClose);
  
  // Graph properties
  setAutoReplot(false);
  setCanvasBackground(QColor(Qt::yellow));
  setAxisScaleDraw(QwtPlot::xBottom, new TimeScale());

  Zoomer = new QwtPlotZoomer(QwtPlot::xBottom,QwtPlot::yLeft,canvas());
  connect(Zoomer, SIGNAL(zoomed(const QwtDoubleRect &)), this, SLOT(HandleZoom(const QwtDoubleRect &)));
  Panner = new QwtPlotPanner(canvas());
  Panner->setMouseButton(Qt::LeftButton, Qt::ShiftModifier);
  Grid = new QwtPlotGrid;
  Grid->setMajPen(QPen(Qt::gray, 0, Qt::DotLine));
  Grid->attach(this);
  Legend = new QwtLegend();
  Legend->setItemMode(QwtLegend::ClickableItem);
  insertLegend(Legend, QwtPlot::TopLegend);

  connect(this, SIGNAL(legendClicked (QwtPlotItem *)), SLOT(LegendClicked(QwtPlotItem *)));
  
  // Connect to DIM handler
  if (connect(Handler, SIGNAL(YEP(DimInfo *, int, QByteArray, QString)), SLOT(Update(DimInfo *, int, QByteArray, QString))) == false) {
    printf("Failed connection for %s\n", DIMService.toAscii().data());
  }

  // Context menu
  Menu = new QMenu(this);
  YLogAction = Menu->addAction("y scale log", this, SLOT(UpdatePlot()));
  YLogAction->setCheckable(true);
  NormAction = Menu->addAction("Normalize", this, SLOT(UpdatePlot()));
  NormAction->setCheckable(true);
  StyleAction = Menu->addAction("Draw dots", this, SLOT(UpdatePlot()));
  StyleAction->setCheckable(true);
  Menu->addAction("Zoom out", this, SLOT(MenuZoomOut()));
  Menu->addAction("Single trace", this, SLOT(MenuSingleTrace()));
  Menu->addSeparator();
  Menu->addAction("Save as ASCII", this, SLOT(MenuSaveASCII()));
  Menu->addAction("Save plot", this, SLOT(MenuSave()));
  Menu->addAction("Print plot", this, SLOT(MenuPrint()));
  Menu->addSeparator();
  Menu->addAction("Paste service", this, SLOT(MenuPasteService()));

  // DIM client
  if (!DIMService.isEmpty()) AddService(DIMService);
}

//
// Destructor (items with parent widget are automatically deleted)
//
Edd_Plot::~Edd_Plot() {

  for (int i=0; i<Items.size(); i++) {
    delete Items[i].LiveData;
    delete Items[i].Signal;
  }  
  delete Grid;
}

//
// Add history service to plot
//
void Edd_Plot::AddService(QString Name) {

  // Lock before accessing Items list
  QMutexLocker Locker(&Mutex);

  // Check if already subscribed to service
  for (int i=0; i<Items.size(); i++) {
    if (Name == Items[i].LiveData->getName()) {
      QMessageBox::warning(this, "Edd Message",Name+".hist already present",QMessageBox::Ok);
      return;
    }
  }  
  
  // Generate new curve and subscribe to service
  struct PlotItem New;
  New.Signal = new QwtPlotCurve;
  New.Signal->attach(this);
  New.Signal->setTitle(Name+".hist");
  New.Signal->setPen(QColor(LineColors[Items.size() % (sizeof(LineColors)/sizeof(Qt::GlobalColor))]));
  New.SizeLimit = 5000;
  New.LiveData = new DimStampedInfo(Name.toAscii().data(), NO_LINK, Handler); 

  Items.append(New);
}

// Update widget (must happen in GUI thread)
void Edd_Plot::Update(DimInfo *Info, int Time, QByteArray, QString Text) {

  // Check if service available
  if (Time == -1) {
	setStatusTip(QString("%1:  unavailable").arg(Info->getName()));
  }

  // Lock before accessing Items list
  QMutexLocker Locker(&Mutex);

  // Determine which plot item this call belongs to
  int ItemNo;
  for (ItemNo=0; ItemNo<Items.size(); ItemNo++) if (Info == Items[ItemNo].LiveData) {
  
	// If size limit reached, clear buffer
    if (Items[ItemNo].x.size() > Items[ItemNo].SizeLimit) {
	  Items[ItemNo].x.clear();
	  Items[ItemNo].y.clear();
    }

	// If buffer empty, request new history buffer
    if (Items[ItemNo].x.isEmpty()) {
	  int Time, Size;
	  void *Data;

	  if (GetHistory(Items[ItemNo].LiveData->getName())) {
     	double Smallest = DBL_MAX, Largest = DBL_MIN;
		double Number=0;
		while (Next(Time, Size, Data)) {
		  switch (*(Info->getFormat())) {
    		case 'I':  Number = *(int *) Data;   break;
    		case 'S':  Number = *(short *) Data;   break;
    		case 'F':  Number = *(float *) Data;   break;
    		case 'D':  Number = *(double *) Data;   break;
    		case 'X':  Number = *(long long *) Data;   break;
    		default: break;
		  }

		  Items[ItemNo].x.append(Time);
		  Items[ItemNo].y.append(Number);
		  
      	  if (Largest < Items[ItemNo].y.last()) Largest = Items[ItemNo].y.last();
    	  if (Smallest > Items[ItemNo].y.last()) Smallest = Items[ItemNo].y.last();
		}

    	Items[ItemNo].Smallest = Smallest;
    	Items[ItemNo].Largest = Largest;

		// Local buffer always at least twice as large as history
		if (Items[ItemNo].SizeLimit < 2*Items[ItemNo].x.size()) {
		  Items[ItemNo].SizeLimit = 2*Items[ItemNo].x.size();
		}		
	  }
	}

    // Append data
    Items[ItemNo].x.append(Time);
    Items[ItemNo].y.append(atof(Text.toAscii().data()));

    // Update largest and smallest value    
    if (Items[ItemNo].y.last() > Items[ItemNo].Largest) Items[ItemNo].Largest = Items[ItemNo].y.last();
    if (Items[ItemNo].y.last() < Items[ItemNo].Smallest) Items[ItemNo].Smallest = Items[ItemNo].y.last();    

    // Update status tip
    QDateTime Timex = QDateTime::fromTime_t(Time); 
    StatusTip = QString("%1:  Last update %2  Format '%3'").arg(Info->getName(), Timex.toString()).arg(Info->getFormat());
  }

  Locker.unlock();

  UpdatePlot();
}

//
// Update all curves in plot
//
void Edd_Plot::UpdatePlot() {

  static QwtSymbol Symbol, Sym1;
  Symbol.setStyle(QwtSymbol::Ellipse);
  Symbol.setSize(4);

  if (!YLogAction->isChecked()) {
    setAxisScaleEngine(QwtPlot::yLeft, new QwtLinearScaleEngine);
  }
  else setAxisScaleEngine(QwtPlot::yLeft, new QwtLog10ScaleEngine);

  // Lock before accessing Items list
  QMutexLocker Locker(&Mutex);

  setStatusTip(StatusTip);
  
  for (int ItemNo=0; ItemNo<Items.size(); ItemNo++) {

    if (StyleAction->isChecked()) Items[ItemNo].Signal->setSymbol(Symbol);
    else Items[ItemNo].Signal->setSymbol(Sym1);

    int DataPoints = Items[ItemNo].x.size();	
	if (DataPoints == 0) continue;
 
    // Normalize y scale if requested
    double *y = new double [DataPoints];
    for (int i=0; i<DataPoints; i++) {
      y[i] = Items[ItemNo].y[i];

      if (NormAction->isChecked()) {
	    if (Items[ItemNo].Smallest != Items[ItemNo].Largest) {
		  y[i] = (y[i] - Items[ItemNo].Smallest)/(Items[ItemNo].Largest-Items[ItemNo].Smallest);
		}
	    else y[i] = 1;
      }  
    }

    // Plot data
    Items[ItemNo].Signal->setData(Items[ItemNo].x.data(), y, DataPoints);
    Items[ItemNo].Signal->show();
    Zoomer->setZoomBase(Items[ItemNo].Signal->boundingRect());

    delete[] y;
  }
  replot();
}

//
// Reset graph axes to autoscale when fully unzoomed
//
void Edd_Plot::HandleZoom(const QwtDoubleRect &) {

  if(Zoomer->zoomRectIndex() == 0) {
    setAxisAutoScale(QwtPlot::xBottom);
    setAxisAutoScale(QwtPlot::yLeft);
  }
}

//
// Drag and drop methods
//

void Edd_Plot::dragEnterEvent(QDragEnterEvent *Event) {
    
  if (Event->mimeData()->hasFormat("text/plain")) Event->acceptProposedAction();
}

void Edd_Plot::dropEvent(QDropEvent *Event) {

  AddService(Event->mimeData()->text().toAscii().data());
}
    
// Opening context menu
void Edd_Plot::contextMenuEvent(QContextMenuEvent *Event) {

  Menu->exec(Event->globalPos());
}

// Drag&Drop method
void Edd_Plot::LegendClicked(QwtPlotItem *Item) {

  QDrag *Drag = new QDrag(this);
  QMimeData *MimeData = new QMimeData;
  MimeData->setText(Item->title().text().remove(".hist"));
  Drag->setMimeData(MimeData);
  Drag->exec();
}


// Zoom completely out
void Edd_Plot::MenuZoomOut() {

  Zoomer->zoom(0);
  UpdatePlot();
}

// Remove all items except last
void Edd_Plot::MenuSingleTrace() {

  // Lock before accessing Items list
  QMutexLocker Locker(&Mutex);

  while (Items.size() > 1) {  
    delete Items.last().LiveData;
    delete Items.last().Signal;
    Items.takeLast();
  }
  
  Locker.unlock();
  UpdatePlot();
}

// Save data of plot as test 
void Edd_Plot::MenuSaveASCII() {
  QString Filename = QFileDialog::getSaveFileName(this,
     "Filename", ".", "Text files (*.txt *.ascii *.asc);;All files (*)");
  if (Filename.length() <= 0) return;
  
  QFile File(Filename);
  if (!File.open(QFile::WriteOnly | QIODevice::Text | QFile::Truncate)) {
    QMessageBox::warning(this, "Edd Message","Could not open file for writing.",QMessageBox::Ok);
    return;
  }
  
  // Lock before accessing Items list
  QMutexLocker Locker(&Mutex);
  QTextStream Stream(&File);

   // Write x and y data for all signals to file
  for (int ItemNo=0; ItemNo<Items.size(); ItemNo++) {
    Stream << QString("# ")+Items[ItemNo].LiveData->getName()+".hist" << endl;
    for (int i=0; i<Items[ItemNo].Signal->dataSize(); i++) {
      Stream << (int) Items[ItemNo].x.at(i) << " " << Items[ItemNo].Signal->y(i) << endl;
    }
  }
}

// Print plot
void Edd_Plot::MenuPrint() {

  QPrinter *Printer = new QPrinter;
  QPrintDialog *PrintDialog = new QPrintDialog(Printer, this);
  if (PrintDialog->exec() == QDialog::Accepted) {
    QPainter Painter(Printer);
    QPixmap Pixmap = QPixmap::grabWidget(this);
    Painter.drawPixmap(0, 0, Pixmap);
  }
  delete Printer;	delete PrintDialog;
}

// Save plot as image
void Edd_Plot::MenuSave() {

  QString Filename = QFileDialog::getSaveFileName(this,
     "Filename of image", "/home/ogrimm/ddd", "Image files (*.bmp *.jpg *.png *.ppm *.tiff *.xbm *.xpm);;All files (*)");
  if (Filename.length()>0) {
    QPixmap Pixmap = QPixmap::grabWidget(this);
    if(!Pixmap.save(Filename)) {
      QMessageBox::warning(this, "Edd Message","Could not write image file.",QMessageBox::Ok);
      remove(Filename.toAscii().data());
    }
  }
}

// Add new service by pasting name
void Edd_Plot::MenuPasteService() {

  AddService(QApplication::clipboard()->text().toAscii().data());
}


///////////////////////////////////////
// History text box for DIM service //
//////////////////////////////////////

//
// Constructor
//
Edd_TextHist::Edd_TextHist(QString DIMService, QWidget *P): QTextEdit(P), EvidenceHistory() {
  
  // Widget properties
  setReadOnly(true);
  setAttribute(Qt::WA_DeleteOnClose);
  setAutoFillBackground(true);
  document()->setMaximumBlockCount(1000);
  
  // Connect to DIM handler
  if (connect(Handler, SIGNAL(YEP(DimInfo*, int, QByteArray, QString)), SLOT(Update(DimInfo*, int, QByteArray, QString))) == false) {
    printf("Failed connection for %s\n", DIMService.toAscii().data());
  }

  // Get history for this service 
  int Time, Size;
  void *Data;

  if (GetHistory(DIMService.toAscii().data())) {
	while (Next(Time, Size, Data)) {
	  moveCursor (QTextCursor::Start);
	  insertPlainText(QString("(")+QDateTime::fromTime_t(Time).toString()+") ");	  
	  insertPlainText(QString((char *) Data) + "\n");	  
	}
  }

  // DIM client
  Service = new DimStampedInfo(DIMService.toAscii().data(), INT_MAX, NO_LINK, Handler);
}

// Destructor
Edd_TextHist::~Edd_TextHist() {

  delete Service;
}


// Update widget (must happen in GUI thread)
void Edd_TextHist::Update(DimInfo *Info, int Time, QByteArray, QString Text) {

  if (Info != this->Service) return;

  // Check if service available
  if (Time == -1) {
	setStatusTip(QString("%1:  unavailable").arg(Info->getName()));
	return;
  }

  QDateTime Timex = QDateTime::fromTime_t(Time); 

  moveCursor(QTextCursor::Start);
  insertPlainText(QString("(")+Timex.toString()+QString(") "));	  
  insertPlainText(Text + "\n");	  
	
  // Update status tip
  StatusTip = QString("%1:  Last update %2  Format '%3'").arg(Info->getName(), Timex.toString()).arg(Info->getFormat());
}


//////////////////
// Text display //
//////////////////

// Constructor
Edd_Textout::Edd_Textout(QString DIMService, QWidget *P): QTextEdit(P) {

  // Widget properties
  setReadOnly(true);
  setAutoFillBackground(true);
  document()->setMaximumBlockCount(1000);
  Accumulate = true;
  
  // Connect to DIM handler
  if (connect(Handler, SIGNAL(YEP(DimInfo*, int, QByteArray, QString)), SLOT(Update(DimInfo*, int, QByteArray, QString))) == false) {
    printf("Failed connection for %s\n", DIMService.toAscii().data());
  }

  // DIM client
  Data = new DimStampedInfo(DIMService.toAscii().data(), INT_MAX, NO_LINK, Handler);
}

// Destructor
Edd_Textout::~Edd_Textout() {

  delete Data;
}

// Handling of DIM service update
void Edd_Textout::Update(DimInfo *Info, int Time, QByteArray, QString Text) {

  if (Info != this->Data) return;
  
  QPalette Pal = palette();  

  // Check if service available
  if (Time == -1) {
	setStatusTip(QString("%1:  unavailable").arg(Info->getName()));
    Pal.setColor(QPalette::Base, Qt::lightGray);
  }
  else {
    Pal.setColor(QPalette::Base, Qt::white);

	// Clear display in case text should not accumulate
	if (Accumulate == false) clear();
	
    // Add if service contains only a string
    if (strcmp(Info->getFormat(), "C") == 0) insertPlainText(Text);

    // Update status tip
    setStatusTip(QString("%1:  Last update %2  Format '%3'").arg(Info->getName(), QDateTime::fromTime_t(Time).toString()).arg(Info->getFormat()));
  }
  setPalette(Pal);
}


//
// Main GUI (all widgets have ultimately Central as parent)
//
GUI::GUI() {
 
  Handler = this;
  
  // Set features of main window
  Central = new QWidget(this);
  setCentralWidget(Central);
  setStatusBar(new QStatusBar(this));
  setGeometry(100, 100, 800, 650);
  setWindowTitle("Edd - Evidence Data Display");

  Edd_Indicator *Value;
  Edd_Plot *Graph;
  Edd_Textout *Textout; 
  QString Text;

   // TextBox for value
  //Value = new Edd_Indicator((char *) "SQM/NSB", Central);
  //Graph = new Edd_Plot((char *) "SQM/NSB", Central);
  //Graph->AddService("BIAS/VOLT/ID00/00-000");
  

  // Clock (updated every second)
  //Clock = new QwtAnalogClock(Central);
  //Clock->scaleDraw()->setPenWidth(3);
  //Clock->setLineWidth(6);
  //Clock->setFrameShadow(QwtDial::Sunken);
  //Clock->setGeometry(0,0,10,10);
  //Clock->setTime();
  
  //QTimer *Timer = new QTimer(Clock);
  //Timer->connect(Timer, SIGNAL(timeout()), Clock, SLOT(setCurrentTime()));
  //Timer->start(1000);

  MainWidget = new QWidget();
  MainLayout = new QGridLayout(MainWidget);

  Value = new Edd_Indicator("Alarm/Status");
  Value->setMaximumWidth(200);
  MainLayout->addWidget(Value, 0, 0, 1, 2);      

  Value = new Edd_Indicator("Alarm/MasterAlarm");
  MainLayout->addWidget(Value, 0, 1, 1, 1);

  Textout = new Edd_Textout("Alarm/Summary");
  Textout->Accumulate = false;
  Textout->setMaximumWidth(200);
  Textout->setMaximumHeight(150);
  MainLayout->addWidget(Textout, 1, 0, 1, 2);

  QFrame *Val = new QFrame();
  Val->setFrameStyle(QFrame::HLine);
  Val->setLineWidth(10);
  //Value->setMaximumWidth(200);
  MainLayout->addWidget(Val, 2, 0, 2, 1);      

  Value = new Edd_Indicator("DColl/Status");
  Value->setMaximumWidth(200);
  MainLayout->addWidget(Value, 3, 0, 1, 2);      

  Value = new Edd_Indicator("DColl/DataSizekB");
  MainLayout->addWidget(Value, 4, 0, 1, 1);

  Value = new Edd_Indicator("DColl/LogSizekB");
  MainLayout->addWidget(Value, 4, 1, 1, 1);

  Value = new Edd_Indicator("DColl/CurrentFile");
  Value->setMaximumWidth(400);
  MainLayout->addWidget(Value, 5, 0, 1, 3);

  Value = new Edd_Indicator("Config/Status");
  Value->setMaximumWidth(200);
  MainLayout->addWidget(Value, 6, 0, 1, 2);      

  Value = new Edd_Indicator("Config/ModifyTime");
  Value->setMaximumWidth(200);
  Value->ShowAsTime = true;
  MainLayout->addWidget(Value, 7, 0, 1, 1);

  QPushButton *Button = new QPushButton();
  Button->setText("Start DIM browser");
  connect(Button, SIGNAL(released()), SLOT(StartDIMBrowser()));
  MainLayout->addWidget(Button, 7, 1, 1, 1);

  Edd_TextHist *Bla;
  Bla = new Edd_TextHist("DColl/CurrentFile");
  MainLayout->addWidget(Bla, 8, 0, 1, 1);

  // Layout of all widgets
  //MainLayout->addWidget(Value, 0, 0, 1, 1);
  //MainLayout->addWidget(Clock, 0, 1, 1, 1);
  //MainLayout->addWidget(Graph, 1, 0, 1, 2);
  //MainLayout->setColumnStretch(1, 10);
  
  //MainLayout->addWidget(Clock, 0, 1, 1, 1);
  //MainLayout->addWidget(Graph1, 2, 0, 1, 2);

  // Bias voltage page
  BiasWidget = new QWidget();
  BiasLayout = new QGridLayout(BiasWidget);
  Graph = new Edd_Plot();
  for (int i=0; i<18; i++) {
    Text = Text.sprintf("Bias/VOLT/ID00/00-%.3d",i);
    Value = new Edd_Indicator(Text);

    BiasLayout->addWidget(Value, i%9+1, 0+i/9, 1, 1);
    Graph->AddService(Text);

    Text = Text.sprintf("Bias/VOLT/ID00/01-%.3d",i);
    Value = new Edd_Indicator(Text);
    BiasLayout->addWidget(Value, i%9+1, 2+i/9, 1, 1);
    Graph->AddService(Text);
  }

  BiasLayout->addWidget(Graph, 0, 4, 12, 3);
  Value = new Edd_Indicator("Bias/Status");
  Value->setMaximumWidth(200);
  BiasLayout->addWidget(Value, 0, 0, 1, 3);      

  Textout = new Edd_Textout("Bias/Textout");
  Textout->setFixedWidth(400);
  BiasLayout->addWidget(Textout, 10, 0, 4, 4);      

  // Environment page
  EnvironmentWidget = new QWidget();
  EnvironmentLayout = new QGridLayout(EnvironmentWidget);
  Value = new Edd_Indicator("ARDUINO/Status");
  Value->setMaximumWidth(200);
  EnvironmentLayout->addWidget(Value, 0, 0, 1, 3);      

  Graph = new Edd_Plot();
  for (int i=0; i<10; i++) {
    Text = Text.sprintf("ARDUINO/VAL%.2d", i);
    Value = new Edd_Indicator(Text);
    EnvironmentLayout->addWidget(Value, i%5+1, i/5, 1, 1);
    Graph->AddService(Text);
  }
  EnvironmentLayout->addWidget(Graph, 0, 3, 7, 4);      

  Value = new Edd_Indicator("SQM/NSB");
  EnvironmentLayout->addWidget(Value, 6, 0, 1, 1);      

  // Tab widget
  TabWidget = new QTabWidget(Central);
  TabWidget->addTab(BiasWidget, "&Bias");
  TabWidget->addTab(EnvironmentWidget, "&Environment");
  TabWidget->addTab(MainWidget, "Evidence");

  // Menu bar
  QMenu* Menu = menuBar()->addMenu("&Menu");
  Menu->addAction("New history plot", this, SLOT(MenuNewHistory()));
  Menu->addSeparator();
  Menu->addAction("About", this, SLOT(MenuAbout()));
  Menu->addSeparator();
  QAction* QuitAction = Menu->addAction("Quit", qApp, SLOT(quit()));
  QuitAction->setShortcut(Qt::CTRL + Qt::Key_Q);

  // Show main window
  show();
}  
    
GUI::~GUI() {
  delete Central;
}


void GUI::MenuAbout() {
  QString Rev(SVN_REVISION);
  Rev.remove(0,1).chop(2);
  
  QMessageBox::about(this, "About Edd","Evidence Data Display\n\n"
    "Written by Oliver Grimm, IPP, ETH Zurich\n"
    "This version compiled "__DATE__" ("+Rev+")\n\n"
    "Graphical user interface implemented with Qt and Qwt.\n"
    "Evidence control system based on DIM (http://dim.web.cern.ch).\n\n"
    "Comments to oliver.grimm@phys.ethz.ch.");
}

// Open request for new history plot
void GUI::MenuNewHistory() {

  QStringList List;
  char *Name, *Format;
  int Type;
  bool OK;

  // Find all services that are not history services and sort
  getServices("*");
  while ((Type = getNextService(Name, Format)) != 0) {
    if (Type==DimSERVICE && strstr(Name, ".hist")==NULL) List.append(Name);
  }
  List.sort();

  // Open dialog and open history window
  QString Result = QInputDialog::getItem(this, "Edd Request",
    "Enter DIM service name", List, 0, true, &OK);
  if (OK && !Result.isEmpty()) {
    Result = Result.trimmed();
    if (Result.endsWith(".hist")) Result.chop(5);
    QWidget *Hist = OpenHistory(Result.toAscii().data());
    if (Hist != NULL) Hist->show();
  }
}

// Handling of DIM service update
void GUI::infoHandler() {

  // Check if service available
  if (!EvidenceServer::ServiceOK(getInfo())) YEP(getInfo(), -1);
  else {
    char *Txt = EvidenceServer::ToString(getInfo());

    YEP(getInfo(), getInfo()->getTimestamp(), QByteArray((char *) getInfo()->getData(), getInfo()->getSize()), QString(Txt));
	free(Txt);
  }
}

// Start DIM Browser
void GUI::StartDIMBrowser() {

  QProcess::startDetached("did", QStringList(), QString(getenv("DIMDIR"))+"/linux/");
}


//---------------------------------------------------------------------
//************************ All functions ****************************
//-------------------------------------------------------------------

// Quit application when clicking close button on window
void GUI::closeEvent(QCloseEvent *) {
  qApp->quit();
}


//---------------------------------------------------------------------
//**************************** Main program ***************************
//---------------------------------------------------------------------

int main(int argc, char *argv[]) {

  QApplication app(argc, argv); 
  GUI MainWindow;

  return app.exec();
}
