/*
 *
 * FITS.h
 *
 * Global fits header
 *
 * Author: lyard
 *
 */

#ifndef _FITS_H_
#define _FITS_H_

//#include <string>

namespace FITS
{

    //Identifier of the compression schemes processes
    enum CompressionProcess_t
    {
        kFactRaw       = 0x0,
        kFactSmoothing = 0x1,
        kFactHuffman16 = 0x2
    };

    //ordering of the columns / rows
    enum RowOrdering_t
    {
        kOrderByCol = 'C',
        kOrderByRow = 'R'
    };


#ifndef __CINT__

    //Structure helper for tiles headers
    struct TileHeader
    {
      char     id[4];
      uint32_t numRows;
      uint64_t size;

      TileHeader() {}

      TileHeader(uint32_t nRows,
                 uint64_t s) : id({'T', 'I', 'L', 'E'}),
                                 numRows(nRows),
                                 size(s)
      { };
    } __attribute__((__packed__));

    //Structure helper for blocks headers and compresion schemes
    struct BlockHeader
    {
        uint64_t      size;
        char          ordering;
        unsigned char numProcs;
        uint16_t      processings[];

        BlockHeader(uint64_t      s=0,
                    char          o=kOrderByRow,
                    unsigned char n=1) : size(s),
                                         ordering(o),
                                         numProcs(n)
        {}
    } __attribute__((__packed__));

    //Helper structure to simplify the initialization and handling of compressed blocks headers
    struct BlockHeaderWriter
    {
        BlockHeaderWriter(const std::string& comp="") : header(),
                                                        sequence()
        {
            if (comp == "" || comp == "RAW")
            {
                sequence.emplace_back(kFactRaw);
                return;
            }
            if (comp == "SMOOTHMAN")
            {
                header = BlockHeader(0, kOrderByRow, 2);
                sequence.emplace_back(kFactSmoothing);
                sequence.emplace_back(kFactHuffman16);
                return;
            }

            std::ostringstream str;
            str << "Unkown compression requested: " << comp;
#ifdef __EXCEPTIONS
            throw std::runtime_error(str.str());
#else
            gLog << ___err___ << "ERROR - " << str.str();
            return;
#endif
        }

        void Write(char* to)
        {
            memcpy(to, &header, sizeof(BlockHeader));
            memcpy(to+sizeof(BlockHeader), sequence.data(), header.numProcs*sizeof(uint16_t));
        }

        uint32_t SizeOnDisk() { return sizeof(BlockHeader) + sizeof(uint16_t)*header.numProcs;}

        uint16_t Proc(uint32_t i) { return sequence[i]; }

        uint16_t NumProcs() { return header.numProcs; }

        char Ordering() { return header.ordering;}

        void SetBlockSize(uint64_t size) { header.size = size;}

        BlockHeader      header;
        std::vector<uint16_t> sequence;
    };

#endif
};

#endif //_FITS_H_

