/*###########################################################################
  #
  #  This file is part of MTrigLvl2 simulation toolkit for simulating the 
  #  MAGIC Second Level Trigger	
  #                                           e-mail:  stamerra@pi.infn.it 
  #                                                    marcos@gae.ucm.es    #
  #                                             date:  January 2003         #
  #                                          version:  0.1                  #
  ###########################################################################*/

#include "MTrigLvl2.h"
#include <iostream.h>
#include "MGeomCamMagic.h"
#include "MCamDisplay.h"
#include "MMcTrig.hxx"
#include <TCanvas.h>


ClassImp(MTrigLvl2)


//
// Correspondence TABLE between pixel numbering in the trigger celss and
// the standard spiral counting 
// (Notice: Pixels start to count from 1 instead of 0)
//
const Float_t pixels_in_cell[36][19] = {
{26,  91,  66,  71,  76,  81,  86,  269,  224,  233,  242,  251,  260,  391,  336,  347,  358,	369,  380},
{25,  61,  41,  45,  49,  53,  57,  215,  175,  183,  191,  199,  207,  325,  275,  285,  295,  305,  315},
{24,  37,  22,  25,  28,  31,  34,  167,  132,  139,  146,  153,  160,  265,  220,  229,  238,  247,  256},
{23,  19,   9,  11,  13,  15,  17,  125,   95,  101,  107,  113,  119,  211,  171,  179,  187,  195,  203},
{27,  62,  67,  72,  77,  82,  87,  270,  225,  234,  243,  252,  261,  392,  337,  348,  359,  370,  381},
{12,  38,  42,  46,  50,  54,  58,  216,  176,  184,  192,  200,  208,  326,  276,  286,  296,  306,  316},
{11,  20,  23,  26,  29,  32,  35,  168,  133,  140,  147,  154,  161,  266,  221,  230,  239,  248,  257},
{10,   8,  10,  12,  14,  16,  18,  126,   96,  102,  108,  114,  120,  212,  172,  180,  188,  196,  204},
{22,   2,   3,   4,   5,   6,   7,   90,   65,   70,   75,   80,   85,  164,  129,  136,  143,  150,  157},
{28,  93,  99, 105, 111, 117, 123,  271,  226,  235,  244,  253,  262,  393,  338,  349,  360,  371,  382},
{13,  63,  68,  73,  78,  83,  88,  217,  177,  185,  193,  201,  209,  327,  277,  287,  297,  307,  317},
{ 4,  39,  43,  47,  51,  55,  59,  169,  134,  141,  148,  155,  162,  267,  222,  231,  240,  249,  258},
{ 3,  21,  24,  27,  30,  33,  36,  127,   97,  103,  109,  115,  121,  213,  173,  181,  189,  197,  205},
{ 9,   9,  11,  13,  15,  17,  19,   91,   66,   71,   76,   81,   86,  165,  130,  137,  144,  151,  158},
{21,   3,   4,   5,   6,   7,   2,   61,   41,   45,   49,   53,   57,  123,   93,   99,  105,  111,  117},
{29, 130, 137, 144, 151, 158, 165,  218,  227,  236,  245,  254,  263,  394,  339,  350,  361,  372,  383},
{14,  94, 100, 106, 112, 118, 124,  170,  178,  186,  194,  202,  210,  328,  278,  288,  298,  308,  318},
{ 5,  64,  69,  74,  79,  84,  89,  128,  135,  142,  149,  156,  163,  268,  223,  232,  241,  250,  259},
{ 1,  40,  44,  48,  52,  56,  60,   92,   98,  104,  110,  116,  122,  214,  174,  182,  190,  198,  206},
{ 2,  22,  25,  28,  31,  34,  37,   62,   67,   72,   77,   82,   87,  166,  131,  138,  145,  152,  159},
{ 8,  10,  12,  14,  16,  18,   8,   38,   42,   46,   50,   54,   58,  124,   94,  100,  106,  112,  118},
{20,  11,  13,  15,  17,  19,   9,   20,   23,   26,   29,   32,   35,   88,   63,   68,   73,   78,   83},
{30, 131, 138, 145, 152, 159, 166,  219,  228,  237,  246,  255,  264,  329,  279,  289,  299,  309,  319},
{15,  95, 101, 107, 113, 119, 125,  171,  179,  187,  195,  203,  211,  269,  224,  233,  242,  251,  260},
{ 6,  65,  70,  75,  80,  85,  90,  129,  136,  143,  150,  157,  164,  215,  175,  183,  191,  199,  207},
{ 7,  41,  45,  49,  53,  57,  61,   93,   99,  105,  111,  117,  123,  167,  132,  139,  146,  153,  160},
{19,  23,  26,  29,  32,  35,  20,   63,   68,   73,   78,   83,   88,  125,   95,  101,  107,  113,  119},
{37,  24,  27,  30,  33,  36,  21,   39,   43,   47,   51,   55,   59,   89,   64,   69,   74,   79,   84},
{31, 132, 139, 146, 153, 160, 167,  220,  229,  238,  247,  256,  265,  270,  225,  234,  243,  252,  261},
{16,  96, 102, 108, 114, 120, 126,  172,  180,  188,  196,  204,  212,  216,  176,  184,  192,  200,  208},
{17,  66,  71,  76,  81,  86,  91,  130,  137,  144,  151,  158,  165,  168,  133,  140,  147,  154,  161},
{18,  42,  46,  50,  54,  58,  38,   94,  100,  106,  112,  118,  124,  126,   96,  102,  108,  114,  120},
{36,  43,  47,  51,  55,  59,  39,   64,   69,   74,   79,   84,   89,   90,   65,   70,   75,   80,   85},
{32, 133, 140, 147, 154, 161, 168,  221,  230,  239,  248,  257,  266,  217,  177,  185,  193,  201,  209},
{33,  97, 103, 109, 115, 121, 127,  173,  181,  189,  197,  205,  213,  169,  134,  141,  148,  155,  162},
{35,  68,  73,  78,  83,  88,  63,   95,  101,  107,  113,  119,  125,   91,   66,   71,   76,   81,   86}
 };


/******************************************************************************
 * Default constructor
 *
 *****************************************************************************/
MTrigLvl2::MTrigLvl2(const char *name, const char *title)
{
  fName = name ? name : ClassName();
  fTitle = title;

  cout << "created MTrigLvl2" << endl;

  //
  // Initialization of the fPixels array to cero
  //
 for(int i=0; i<36; i++){
    for(int j=0; j<19; j++){
	fPixels[i][j]=0;
    }
  }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Print Functions
//
///////////////////////////////////////////////////////////////////////////////

/******************************************************************************
 * Just print out the pixels containg in a given trigger cell
 *
 *****************************************************************************/
void MTrigLvl2::PrintCell(Int_t cell)
{
   for (int i=0;i<36;i++)
    {
      cout << pixels_in_cell[i][cell]-1 << endl;
    }
}

/******************************************************************************
 * Print a table with the staus (pixel is fired or not after Lvl1) for all
 * the pixels in all the trigger cells
 *
 *****************************************************************************/
void MTrigLvl2::PrintStatus()
{
  for(int i=0; i<36; i++){
    for(int j=0; j<9; j++){
      cout.width(3);
      cout << pixels_in_cell[i][j]-1 << ":" << fPixels[i][j]  << " ";
    } 
    cout << endl;
  }
}






///////////////////////////////////////////////////////////////////////////////
//
//  Set Functions
//
///////////////////////////////////////////////////////////////////////////////

/******************************************************************************
 *
 * Take the information supplied by the Lvl1 (it reads from the object MMcTrig
 * i the status of all pixels after Lvl1) and pass it to the Lvl2 (fill the 
 * array fPixels)
 *
 *****************************************************************************/
void MTrigLvl2::SetLv1(MMcTrig *trig)
{
  fMcTrig = trig;

  //fMcTrig->PrintPixelsFirstLevel();

  for(int i=0; i<36; i++){
    for(int j=0; j<19; j++){
      int pixel = pixels_in_cell[i][j]-1;
      fPixels[i][j] = (fMcTrig->IsPixelFired(pixel,0)) ? 1 : 0;
    }
  }
}


/******************************************************************************
 * Set the trigger status ( pixel fired(=1) or not(=0) ) manually for a given
 * pixel of a given cell
 *
 *****************************************************************************/
void MTrigLvl2::SetPixelFired(Int_t pixel, Int_t fired)
{
  for(int i=0; i<36; i++){
    for(int j=0; j<19; j++){
	if(pixels_in_cell[i][j]-1==pixel) fPixels[i][j]=fired;
    }
  }
}




///////////////////////////////////////////////////////////////////////////////
//
//  Graphical display Functions
//
///////////////////////////////////////////////////////////////////////////////

/******************************************************************************
 * Display the MAGIC Camera a draw the pixels corresponding to a given cell
 *
 *****************************************************************************/
void MTrigLvl2::DrawCell(Int_t cell)
{

  if(cell>18) return;

  //
  // Use MCamDisplay class variable for avoiding to create a MCamDisplay each 
  // time this function is called. Also, now all the hexagons are drawn in 
  // the same camera pad
  //
  if (!fGeomcam)
    fGeomcam = new MGeomCamMagic;  
  if (!fCam)
    {
      fCam = new MCamDisplay(fGeomcam);
      fCam->Draw();
      fCam->DrawPixelNumbers();
    }

  //fCam->Draw();
  fCam->Reset();

  int color=0;

  for(int i=0; i<36; i++){
    color = (fPixels[i][cell]) ? 5 : 3;
    fCam->SetPix( pixels_in_cell[i][cell]-1, color, 1, 5 );
  }

  //
  // Update the display (paint the camera with the new colors)
  //
  gPad->Modified();
  gPad->Update(); 
}



/******************************************************************************
 * Display the MAGIC camera and draw all the pixel fired after Lvl1
 *
 *****************************************************************************/
void MTrigLvl2::DrawLv1()
{
  //
  // Use MCamDisplay class variable for avoiding to create a MCamDisplay each 
  // time this function is called. Also, now all the hexagons are drawn in 
  // the same camera pad
  //
  if (!fGeomcam)
    fGeomcam = new MGeomCamMagic;  
  if (!fCam)
    {
      fCam = new MCamDisplay(fGeomcam);
      fCam->Draw();  
      fCam->DrawPixelNumbers();
    }

  //fCam->Draw(); 
  //fCam->Reset();

  //
  // Set the array of colors for each pixel (that will be painted after 
  // updating the dispaly)
  //
  int color=0;

  for(int i=0; i<577; i++)
    {
      color = (fMcTrig->IsPixelFired(i,0)) ? 5 : 3;
      fCam->SetPix( i, color, 1, 5 );
    }

  //
  // Update the display (paint the camera with the new colors)
  //
  gPad->Modified();
  gPad->Update(); 

}



///////////////////////////////////////////////////////////////////////////////
//
//  Implementation of the Lv2l selection strategies
//
///////////////////////////////////////////////////////////////////////////////

/******************************************************************************
 * For a given cell, just count how many pixels have been fired after Lvl1
 *
 *****************************************************************************/
Int_t MTrigLvl2::GetCellNumberFired(int cell)
{
  int size=0;
 
  for(int i=0; i<36; i++){
    size += fPixels[i][cell];
  }

  return size;
  
}


/******************************************************************************
 * Find the cell which the bigger number of fired pixels
 *
 *****************************************************************************/
Int_t MTrigLvl2::GetBiggerFiredCell()
{
  int size=-1;
  int cell=-1;

  for(int j=0; j<19; j++){
    if (GetCellNumberFired(j) > size) {
      size = GetCellNumberFired(j);
      cell = j;
    }
  }

  cout << "size " <<size <<" in cell " << cell << endl;

  return cell;
  
}



