#include "typedefs.h"
#include "application.h"    // <-- global Registers

#include "spi_master.h"
#include "timer.h"
#include "atmega_adc.h"
#include "ad7719_adc.h"
#include "w5100_spi_interface.h"

#include "macros.h"
#include "muxer_fsc.h"

#include <avr/interrupt.h>
#include <avr/wdt.h>
#include <stdlib.h>


int main(void)
{
    U08 highbyte;
    U08 lowbyte;
    U16 eth_red_bytes;
    U08 rc;

    gReg.ad7719_measured_all = false;
    gReg.adc_measured_all = false;
    for ( U08 i=0; i < (RESISTANCE_REGS); ++i ) {
        gReg.ad7719_enables[i]=0xFF;
        gReg.ad7719_channels_ready[i]=0x00;
    }
    for ( U08 i=0; i < (VOLTAGE_REGS); ++i ) {
        if (i < (VOLTAGE_REGS)-1) {
            gReg.adc_enables[i]=0xFF;
        }
        else {
            gReg.adc_enables[i]=0x0F;
        }
        gReg.adc_channels_ready[i]=0;
    }


    app_init();         // Setup: Watchdog and I/Os
    spi_init();         // Initialize SPI interface as master
    timer_init();       // set up TIMER2: TIMER2_COMP interrupt occurs every 1ms
    atmega_adc_init();

    w5100_reset();
    w5100_init();

    ad7719_init();
    sei();

    //MAIN LOOP
    while (1)
    {
        // checks if socket is okay and resets in case of problem the W5100 
        rc=w5100_caretaker();
        if ( rc != 0x17 && rc != 0x14)  
        {
            // something is not okay with the ethernet ... 
            // will be reset in the next revolution.. but maybe we want to do more...
        }
        // this if should check every 25ms if we have any data on the W5100
        if ( (gVolReg.time_ms % W5100_INPUT_CHECK_TIME == 0) && (w5100_ready) ) {
            if ( (eth_red_bytes = get_S0_RX_RSR()) != 0) {
                for (U08 rep=0; rep<1; ) {
                    if (eth_red_bytes == get_S0_RX_RSR() ) {
                        rep++;
                    }
                    else {
                        eth_red_bytes =get_S0_RX_RSR();
                    }
                }
                //parse_w5300_incoming( w5100_get_RX(ETH_READ_BUFFER_SIZE, true) );
            }
        }
        //----------------------------------------------------------------------------
        if (check_if_measured_all()) {
                write_status_via_eth();
                reset_resistance_done();
                reset_voltage_done();
                reset_voltage_values();
        }

        //IF ATmega internal ADC did finish a conversion --every 200us
        if ( (ADCSRA & (1<<ADIF)) && !gReg.adc_measured_all) {
            lowbyte = ADCL;
            highbyte = ADCH;
            ADCSRA |= 1<<ADIF;
            gReg.adc_current_reading = ((U16)highbyte<<8) | ((U16)lowbyte);
            if (gReg.adc_readings_since_last_muxing == ADC_READINGS_UNTIL_SETTLED + ADC_READINGS_UNTIL_MEAN - 1) {
                gReg.adc_values[gReg.adc_current_channel] += gReg.adc_current_reading;
                gReg.adc_channels_ready[gReg.adc_current_channel/8] |= (1<<(gReg.adc_current_channel%8));
                gReg.adc_current_channel = increase_adc(gReg.adc_current_channel);
                Set_V_Muxer(gReg.adc_current_channel);
                gReg.adc_readings_since_last_muxing = 0;
                _delay_ms(10); // this is a muxer delay
            }
            else if (gReg.adc_readings_since_last_muxing >= ADC_READINGS_UNTIL_SETTLED-1) {
                gReg.adc_values[gReg.adc_current_channel] += gReg.adc_current_reading;
                ++gReg.adc_readings_since_last_muxing;
            }
            else  {
                ++gReg.adc_readings_since_last_muxing;
            }
        }
        //----------------------------------------------------------------------------

        //IF AD7719 ADC just finished a conversion -- every 60ms
        if (AD7719_IS_READY() && !gReg.ad7719_measured_all) {
            gReg.ad7719_current_reading = read_adc(); // --takes at 4MHz SCLK speed about 6us
            // AD7719 is only read out if settled. saves time.	
            if (gReg.ad7719_readings_since_last_muxing == AD7719_READINGS_UNTIL_SETTLED-1) {
                gReg.ad7719_values[gReg.ad7719_current_channel] = gReg.ad7719_current_reading;
                gReg.ad7719_readings_since_last_muxing=0;
                // now prepare the data to be send away via USART.
                // note that this channel is ready, now and 
                // proceed to the next enabled channel.
                gReg.ad7719_channels_ready[gReg.ad7719_current_channel/8] |= (1<<(gReg.ad7719_current_channel%8));
                gReg.ad7719_current_channel = increase_ad7719(gReg.ad7719_current_channel);
                Set_T_Muxer(gReg.ad7719_current_channel);
            } 
            else { // the AD7719 did not settle yet, we discard the reading
                ++gReg.ad7719_readings_since_last_muxing;
            }
        }

    } // end of main while loop
} // end of main function
