'use strict';

var editor1;
var editor2;
var plot;

function debug(txt)
{
    var dbg = document.getElementById("debug");
    dbg.appendChild(document.createTextNode(txt));
    dbg.appendChild(document.createElement("br"));
}

function setupAccordion(accordion, container, inactive)
{
    function onAccordion(event, ui)
    {
        if (ui.oldHeader.length==0)
            $(container).slideDown(400);
        else
            $(container).slideUp(400);
    }

    var config = { collapsible: true };

    if (inactive)
    {
        config.active = false;
        $(container).hide();
    }

    var acc = $(accordion);

    acc.accordion(config);
    acc.on("accordionbeforeactivate", onAccordion);
}

function onResize(event, ui)
{
    if (!ui.size)
        return;

    $(event.target.id).css({marginRight:'2px'});

    var editor = event.target.id=="textcontainer1" ? editor1 : editor2;

    editor.setSize("100%", ui.size.height);
    editor.refresh();
}

function onResizeCenter(event, ui)
{
    var w = document.getElementById("cameracontainer").clientWidth/4;

    var offy = 0;
    var offx = 5;

    var cont = document.getElementById("center").childNodes[0];

    var nn;
    if (cont)
    {
        nn = parseInt(cont.id[cont.id.length-1]);

        document.getElementById("camera"+nn).height=parseInt(w*2);
        document.getElementById("camera"+nn).width=parseInt(w*2);
    }

    if (nn!=1)
    {
        document.getElementById("camera1").height=parseInt(w)-offy;
        document.getElementById("camera1").width=parseInt(w)-offx;
    }

    if (nn!=2)
    {
        document.getElementById("camera2").height=parseInt(w)-offy;
        document.getElementById("camera2").width=parseInt(w)-offx;
    }

    if (nn!=3)
    {
        document.getElementById("camera3").height=parseInt(w)-offy;
        document.getElementById("camera3").width=parseInt(w)-offx;
    }

    if (nn!=4)
    {
        document.getElementById("camera4").height=parseInt(w)-offy;
        document.getElementById("camera4").width=parseInt(w)-offx;
    }

    document.getElementById("center").width=parseInt(w*2);

    document.getElementById("cont1").width=parseInt(w);
    document.getElementById("cont2").width=parseInt(w);
    document.getElementById("cont3").width=parseInt(w);
    document.getElementById("cont4").width=parseInt(w);

    document.getElementById("cont1").height=parseInt(w);
    document.getElementById("cont2").height=parseInt(w);
    document.getElementById("cont3").height=parseInt(w);
    document.getElementById("cont4").height=parseInt(w);

    drawFullCam("camera1");
    drawFullCam("camera2");
    drawFullCam("camera3");
    drawFullCam("camera4");
}

function createEditor(textarea)
{
    var editor;

    var config =
    {
        //value: "function myScript(){return 100;}\n",
        mode:  { name: "text/typescript", globalVars: true },
        lineNumbers: true,
        styleActiveLine: true,
        matchBrackets: true,
        indentUnit: 4,
        foldGutter: true,
        gutters: ["CodeMirror-linenumbers", "CodeMirror-foldgutter"],
        extraKeys: {
            //"Ctrl-D": "duplicateLine",
            //"Alt--": "goToBracket",
            //"Ctrl-H": "findPrev",
            "Ctrl-Down": "autocomplete",
            "Tab": "indentAuto",
            "Ctrl-Y": "deleteLine",
            "Ctrl-.": function(cm) {
                cm.foldCode(cm.getCursor());
            },
            "F11": function(cm) {
                editor.setOption("fullScreen", !editor.getOption("fullScreen"));
            },
            "Ctrl-R": function(cm) {
                editor.execCommand("replace");
            },
            "Esc": function(cm) {
                if (editor.getOption("fullScreen")) editor.setOption("fullScreen", false);
            },
            "Enter": function(cm) {
                editor.execCommand("indentAuto");
                editor.execCommand("newlineAndIndent");
            },
        }

    };

    editor = CodeMirror.fromTextArea(document.getElementById(textarea), config);
    editor.setOption("theme", "blackboard");

    return editor;
}

function colorizeHTML(textarea)
{
    var config =
    {
        //value: "function myScript(){return 100;}\n",
        mode:  { name: "text/typescript", globalVars: true },
        readOnly: true,
    };

    CodeMirror.fromTextArea(document.getElementById(textarea), config);
}

function disableControls(disabled)
{
    $('#event').prop('disabled', disabled);
    $('#pixel').prop('disabled', disabled);
    $('#cbpx').prop('disabled', disabled);
    $('#cbpx-c').prop('disabled', disabled);
    $('#cbpx-b').prop('disabled', disabled);
    $('#cbpx-p').prop('disabled', disabled);
    $('#cbpx-x').prop('disabled', disabled);
    $('#file').prop('disabled', disabled);
}

function onReady()
{
    //$('input,select').keypress(function(event) { return event.keyCode != 13; });

    //colorizeHTML("code0");
    //colorizeHTML("code1");

    $("#accordion").accordion({collapsible:true,active:false,heightStyle:'content'});
    $("#accordion").find('h3').filter(':contains(Runtime)').hide();
    if (location.href.search('debug')==-1)
        $("#accordion").find('h3').filter(':contains(Debug)').hide();

    $("#textcontainer1").resizable({handles:"s",autoHide:true,});
    $("#textcontainer1").on("resize", onResize);

    $("#textcontainer2").resizable({handles:"s",autoHide:true,});
    $("#textcontainer2").on("resize", onResize);

    $("#cameracontainer").on("resize", onResizeCenter);
    onResizeCenter();

    $("#cont1").click(onClickNew);
    $("#cont2").click(onClickNew);
    $("#cont3").click(onClickNew);
    $("#cont4").click(onClickNew);

    $("#camera1").click(onClick);
    $("#camera2").click(onClick);
    $("#camera3").click(onClick);
    $("#camera4").click(onClick);

    editor1 = createEditor("editor1");
    editor2 = createEditor("editor2");

    setupAccordion('#accordion5', '#editorcontainer1', true);
    setupAccordion('#accordion1', '#editorcontainer2');

    $('#accordion5').on("accordionactivate", function() { $('#editorcontainer1fake').hide(); editor1.refresh();  });
    $('#accordion1').on("accordionactivate", function() { $('#editorcontainer2fake').hide(); editor2.refresh();  });

    setupAccordion('#accordion2', '#cameracontainer');
    setupAccordion('#accordion3', '#waveformcontainer');
    setupAccordion('#accordion4', '#helpcontainer', true);

    $("#selectfile1").on('change', onFile);
    $("#selectfile2").on('change', onFile);

    $(document).ajaxStart(function() { disableControls(true) }).ajaxStop(function() { disableControls(false); });


    $.ajax({
        type:    "POST",
        cache:   false,
        url:     "index.php",
        success: onFilelistReceived,
        error:   function(xhr) { if (xhr.status==0) alert("ERROR[0] - Request failed!"); else alert("ERROR[0] - "+xhr.statusText+" ["+xhr.status+"]"); }
    });
}

function onFileSelect(event, ui)
{
    onSubmit(ui.item.value);
}

function onFilelistReceived(result)
{
    var dbg = document.getElementById("debug");

    var pre = document.createElement("pre");
    pre.appendChild(document.createTextNode(rc));
    dbg.appendChild(pre);

    var rc;
    try
    {
        rc = JSON.parse(result);
    }
    catch (e)
    {
        alert("ERROR[0] - Decoding answer:\n"+e);
        return;
    }

    $("#file").autocomplete({source: rc, select: onFileSelect} );
    document.getElementById("file").value = "2014/04/17-181";

    onSubmit("2014/04/17-181");
}

function setZoom(xfrom, xto, yfrom, yto)
{
    var xaxis = plot.getXAxes()[0];
    var yaxis = plot.getYAxes()[0];

    if (xfrom!==undefined)
        xaxis.options.min = xfrom;
    if (xto!==undefined)
        xaxis.options.max = xto;

    if (yfrom!==undefined)
        yaxis.options.min = yfrom;
    if (yto!==undefined)
        yaxis.options.max = yto;

    plot.setupGrid();
    plot.draw();
    plot.clearSelection();
}

function processCameraData(id, data)
{
    var canv = document.getElementById(id);

    canv.dataAbs = [ ];
    for (var i=0; i<1440; i++)
        if (data[i]!==undefined && data[i]!==null)
            canv.dataAbs[i] = data[i];

    canv.min = Math.min.apply(Math, canv.dataAbs.filter(function(e){return !isNaN(e)}));
    canv.max = Math.max.apply(Math, canv.dataAbs.filter(function(e){return !isNaN(e)}));

    canv.dataRel = [ ];
    for (var i=0; i<1440; i++)
        if (data[i]!==undefined && data[i]!==null)
            canv.dataRel[i] = (data[i]-canv.min)/canv.max;

    var n = id[id.length-1];

    if (document.getElementById("cameraminon"+n).checked)
        document.getElementById("cameramin"+n).value = canv.min;
    if (document.getElementById("cameramaxon"+n).checked)
        document.getElementById("cameramax"+n).value = canv.max;

    onCameraMinMax(n);
}


function onDataReceived(rc)
{
    var err = document.getElementById("error");
    var dbg = document.getElementById("debug");
    var con = document.getElementById("console");

    var pre = document.createElement("pre");
    pre.appendChild(document.createTextNode(rc));
    dbg.appendChild(pre);

    try
    {
      rc = JSON.parse(rc);
    }
    catch (e)
    {
        alert("ERROR[1] - Decoding answer:\n"+e);
        return;
    }

    document.getElementById("event").max = rc.event.numEvents;
    var el = document.getElementById("numevents");
    if (el.firstChild)
        el.removeChild(el.firstChild);
    el.appendChild(document.createTextNode(rc.event.numEvents));

    if (rc.ret)
    {
        while (con.lastChild)
            con.removeChild(con.lastChild);
    }

    if (rc.err)
    {
        while (err.lastChild)
            err.removeChild(err.lastChild);

        err.appendChild(document.createTextNode("Javascript runtime exception: "+rc.err.file+":"+rc.err.lineNumber));
        err.appendChild(document.createTextNode("\n"));
        err.appendChild(document.createTextNode(rc.err.sourceLine));
        err.appendChild(document.createTextNode("\n"));
        err.appendChild(document.createTextNode(rc.err.trace));

        var editor = rc.err.file=="main" ? editor2 : editor1;
        editor.setCursor(rc.err.lineNumber-1, 1);

        $("#accordion").find('h3').filter(':contains(Runtime)').show();
        $("#accordion").accordion("option", "active", 0);
    }

    var canv = document.getElementById("camera1");

    if (rc.debug!==undefined)
    {
        con.appendChild(document.createTextNode(rc.debug));

        debug("return "+JSON.stringify(rc.ret));
        debug("");
        debug("Execution times:");
        debug("TimeJS="+(rc.timeJs*1000).toFixed(2)+"ms");
    }

    if (rc.ret!==undefined && Array.isArray(rc.ret))
    {
        if (!Array.isArray(rc.ret[0]))
            processCameraData("camera1", rc.ret);
        else
            processCameraData("camera1", rc.ret[0]);

        if (rc.ret.length>1)
            processCameraData("camera2", rc.ret[1]);

        if (rc.ret.length>2)
            processCameraData("camera3", rc.ret[2]);

        if (rc.ret.length>3)
            processCameraData("camera4", rc.ret[3]);
    }

    if (canv.dataAbs && rc.event)
        document.getElementById("value").value = canv.dataAbs[rc.event.pixel];

    debug("TimePhp="+(rc.timePhp*1000).toFixed(1)+"ms");

    if (Array.isArray(rc.waveform))
    {
        var waveform = document.getElementById("waveform");

        var data = [[],[],[],[]];

        var min = [];
        var max = [];
        if (Array.isArray(rc.waveform) && rc.waveform.length==rc.event.numRoi)
        {
            min.push(Math.min.apply(Math, rc.waveform));
            max.push(Math.max.apply(Math, rc.waveform));

            for (var i=0; i<rc.event.numRoi; i++)
                data[0][i] = [ i, rc.waveform[i] ];
        }

        for (var j=0; j<4; j++)
        {
            var ref = rc.waveform[j];

            if (Array.isArray(ref) && ref.length==rc.event.numRoi)
            {
                min.push(Math.min.apply(Math, ref));
                max.push(Math.max.apply(Math, ref));

                for (var i=0; i<rc.event.numRoi; i++)
                    data[j][i] = [ i, ref[i] ];

            }
        }

        waveform.ymin = Math.min.apply(Math, min);
        waveform.ymax = Math.max.apply(Math, max);
        waveform.xmin = 0;
        waveform.xmax = rc.event.numRoi;

        if (document.getElementById("waveformminon").checked)
            document.getElementById("waveformmin").value = waveform.ymin;
        if (document.getElementById("waveformmaxon").checked)
            document.getElementById("waveformmax").value = waveform.ymax;

        var ymin = document.getElementById("waveformminon").checked ? waveform.ymin : parseInt(document.getElementById("waveformmin").value);
        var ymax = document.getElementById("waveformmaxon").checked ? waveform.ymax : parseInt(document.getElementById("waveformmax").value);

        var opts =
        {
           xaxis: {
               min: waveform.xmin-1,
               max: waveform.xmax+1,
           },
           yaxis: {
               min: ymin-5,
               max: ymax+5,
           },
           series: {
               lines: {
                   show: true
               },
               points: {
                   show: true,
                   symbol: 'cross',
               }
           },
           selection: {
               mode: "xy"
           },
           grid: {
               hoverable: true,
           }
        };

        plot = $.plot("#waveform", data, opts);

        waveform = $('#waveform');
        waveform.bind("plotselected", function (event, ranges)
                      {
                          setZoom(ranges.xaxis.from, ranges.xaxis.to,
                                  ranges.yaxis.from, ranges.yaxis.to);
                      });

        waveform.dblclick(function ()
                          {
                              var waveform = document.getElementById("waveform");
                              setZoom(waveform.xmin-1, waveform.xmax+1, waveform.ymin-5, waveform.ymax+5);
                          });
        waveform.bind("plothover", function (event, pos, item)
                      {
                          if (!item)
                          {
                              $("#tooltip").fadeOut(100);
                              return;
                          }

                          var x = item.datapoint[0].toFixed(2);
                          var y = item.datapoint[1].toFixed(2);
                          //item.series.label

                          var tooltip = $("#tooltip");
                          tooltip.html(parseInt(x) + " / " + y);
                          tooltip.css({top: item.pageY-20, left: item.pageX+5});
                          tooltip.fadeIn(200);
                      });
    }

    //$("#accordion").accordion("refresh");
}

function onSubmit(file, pixelOnly)
{
    if (!file)
        file = document.getElementById("file").value;

    var dbg = document.getElementById("debug");
    while (dbg.lastChild)
        dbg.removeChild(dbg.lastChild);

    var active = $("#accordion").accordion("option", "active");
    if (active==0)
    {
        $("#accordion").accordion("option", "active", false);
        $("#accordion").find('h3').filter(':contains(Runtime)').hide();
    }

    var event   = document.getElementById("event").value;
    var pixel   = document.getElementById("pixel").value;
    var source1 = editor1.getValue();
    var source2 = editor2.getValue();

    var uri = "file="+file+"&event="+event+"&pixel="+map[pixel];
    uri += "&source1="+encodeURIComponent(source1);
    if (!pixelOnly)
        uri += "&source2="+encodeURIComponent(source2);

    $.ajax({
        type:    "POST",
        cache:   false,
        url:     "index.php",
        data:    uri,
        success: onDataReceived,
        error:   function(xhr) { if (xhr.status==0) alert("ERROR[1] - Request failed!"); else alert("ERROR[1] - "+xhr.statusText+" ["+xhr.status+"]"); }
    });
}

function onFile(event, ui)
{
    var f = event.target.files[0];
    if (!f)
        return;

    if (!f.type.match('text/plain') && !f.type.match('application/javascript'))
    {
        alert("ERROR - Unknown file type.");
        return;
    }

    var id     = event.target.id;
    var editor = id[id.length-1]=='1' ? editor1 : editor2;

    var reader = new FileReader();

    // Closure to capture the file information.
    reader.onload = (function(theFile) { return function(e) { editor.setValue(e.target.result); }; })(f);
    // onloadstart
    // onloadend
    // onprogress

    // Read in the text file
    reader.readAsText(f);
}

function onEvent()
{
    onSubmit();
}

function onPixel()
{
    var p = parseInt(document.getElementById("pixel").value);

    var cbpx = map[p];

    document.getElementById("cbpx-c").value = parseInt((cbpx/360));
    document.getElementById("cbpx-b").value = parseInt((cbpx/36)%10);
    document.getElementById("cbpx-p").value = parseInt((cbpx/9)%4);
    document.getElementById("cbpx-x").value = parseInt((cbpx%9));
    document.getElementById("cbpx").value = parseInt(cbpx);

    onSubmit("", true);
}

function onCBPX()
{
    var c = parseInt(document.getElementById("cbpx-c").value);
    var b = parseInt(document.getElementById("cbpx-b").value);
    var p = parseInt(document.getElementById("cbpx-p").value);
    var x = parseInt(document.getElementById("cbpx-x").value);

    var cbpx = c*360 + b*36 + p*9 + x;

    document.getElementById("cbpx").value = parseInt(cbpx);
    document.getElementById("pixel").value = map.indexOf(cbpx);

    onSubmit("", true);
}

function onHW()
{
    var cbpx = parseInt(document.getElementById("cbpx").value);;

    document.getElementById("cbpx-c").value = parseInt((cbpx/360));
    document.getElementById("cbpx-b").value = parseInt((cbpx/36)%10);
    document.getElementById("cbpx-p").value = parseInt((cbpx/9)%4);
    document.getElementById("cbpx-x").value = parseInt((cbpx%9));

    document.getElementById("pixel").value = map.indexOf(cbpx);

    onSubmit("", true);
}

function isInside(x, y, mouse)
{
    var dist = Math.sqrt((mouse.x-x)*(mouse.x-x)+(mouse.y-y)*(mouse.y-y));
    return dist<0.5;

    /*
    ctx.translate(x, y);
    ctx.scale(1/2, 1/3);

    ctx.beginPath();
    ctx.moveTo( 1,  1);
    ctx.lineTo( 0,  2);
    ctx.lineTo(-1,  1);
    ctx.lineTo(-1, -1);
    ctx.lineTo( 0, -2);
    ctx.lineTo( 1, -1);
    ctx.fill();

    ctx.restore();
    */
}

function getIndex(id, mouse)
{
    var canv = document.getElementById(id);

    var scale = 83;

    var w = Math.min(canv.width/scale, canv.height/scale);

    //ctx.translate(canv.width/2, canv.height/2);
    //ctx.scale(w*2, w*2);
    //ctx.scale(1, Math.sqrt(3)/2);
    //ctx.translate(-0.5, 0);

    mouse.x -= canv.width/2;
    mouse.y -= canv.height/2;
    mouse.x /= w*2;
    mouse.y /= w*2;
    mouse.y /= Math.sqrt(3)/2;
    mouse.x -= -0.5;

    if (isInside(0, 0, mouse))
        return 0;

    var cnt = 1;
    for (var ring=1; ring<24; ring++)
    {
        for (var s=0; s<6; s++)
        {
            for (var i=1; i<=ring; i++)
            {
                var pos = new Position(s, ring, i);
                if (pos.d() - pos.x > 395.75)
                    continue;

                if (isInside(pos.x, pos.y, mouse))
                    return cnt;
                cnt++;
            }
        }
    }

    if (isInside(7, -22, mouse))
        return 1438;
    if (isInside(7, 22, mouse))
        return 1439;

    return -1;
}

var inprogress = 0;
function moveAnimate(n, target)
{
    if (inprogress==n || inprogress<0)
        return;

    inprogress = target ? -n : n;

    var element   = $("#camera"+n); //Allow passing in either a JQuery object or selector
    var newParent = $(target ? target : "#cont"+n); //Allow passing in either a JQuery object or selector

    var oldOffset = element.offset();

    var newOffset = newParent.offset();

    var w = newParent.width();
    var h = newParent.height();

    var temp = element.appendTo('body');
    temp.css('position', 'absolute')
        .css('left', oldOffset.left)
        .css('top',  oldOffset.top)
        .css('zIndex', 400);

    temp.animate( {'top': newOffset.top, 'left':newOffset.left, 'width':w, 'height': h},
    'slow', function()
    {
        temp = temp.appendTo(newParent);
        temp.css('position', 'default');
        temp.css('width', 'default');
        temp.css('height', 'default');
        temp.css('left', '0');
        temp.css('top', '0');

        var canv = document.getElementById("camera"+n);

        canv.width = w;
        canv.height = h;

        drawFullCam("camera"+n);

        inprogress = 0;
    });
}

function onClickNew(event)
{
    var id = event.target.id;
    var n = parseInt(id[id.length-1]);

    if (id.substr(0, 3)=="cam")
    {
        var cont = document.getElementById("center").childNodes[0];
        if (cont)
        {
            var nn = parseInt(cont.id[cont.id.length-1]);
            moveAnimate(nn);
        }
        moveAnimate(n, "#center");

    }
    else
        moveAnimate(n);
}

function onClick(event)
{
    var cont = document.getElementById("center").childNodes[0];
    if (!cont)
        return;

    if (cont.id!=event.target.id)
        return;

    var mouse = getClickPosition();
    var index = getIndex(event.target.id, mouse);

    if (index<0)
        return;

    document.getElementById("pixel").value = index;

    onPixel();
}

function getClickPosition(event)
{
    var event = window.event;

    if (event.offsetX !== undefined && event.offsetY !== undefined)
        return { x: event.offsetX, y: event.offsetY };

    return { x: event.layerX, y: event.layerY };
}

function onCameraMinMax(n)
{
    var canv = document.getElementById("camera"+n);

    canv.zmin = document.getElementById("cameramin"+n).value;
    canv.zmax = document.getElementById("cameramax"+n).value;

    drawFullCam("camera"+n);
}

function onCameraMinMaxOn(n)
{
    var canv = document.getElementById("camera"+n);

    var redraw;
    if (document.getElementById("cameraminon"+n).checked)
    {   
        document.getElementById("cameramin"+n).setAttribute("disabled", "true");
        document.getElementById("cameramin"+n).value = canv.min;
        redraw = true;
    }
    else
        document.getElementById("cameramin"+n).removeAttribute("disabled");

    if (document.getElementById("cameramaxon"+n).checked)
    {
        document.getElementById("cameramax"+n).setAttribute("disabled", "true");
        document.getElementById("cameramax"+n).value = canv.max;
        redraw = true;
    }
    else
        document.getElementById("cameramax"+n).removeAttribute("disabled");

    if (redraw)
        onCameraMinMax(n);
}

function onWaveformMinMax()
{
    var wf = document.getElementById("waveform");

    var xmin, xmax, ymin, ymax;

    var redraw;
    if (!document.getElementById("waveformminon").checked)
        ymin = document.getElementById("waveformmin").value;
    if (!document.getElementById("waveformmaxon").checked)
        ymax = document.getElementById("waveformmax").value;

    setZoom(xmin, xmax, ymin, ymax);

}

function onWaveformMinMaxOn()
{
    var wf = document.getElementById("waveform");

    var xmin, xmax, ymin, ymax;

    var redraw;
    if (document.getElementById("waveformminon").checked)
    {
        document.getElementById("waveformmin").setAttribute("disabled", "true");
        document.getElementById("waveformmin").value = wf.ymin;
        ymin = wf.ymin-5;
    }
    else
        document.getElementById("waveformmin").removeAttribute("disabled");

    if (document.getElementById("waveformmaxon").checked)
    {
        document.getElementById("waveformmax").setAttribute("disabled", "true");
        document.getElementById("waveformmax").value = wf.ymax;
        ymax = wf.ymax+5;
    }
    else
        document.getElementById("waveformmax").removeAttribute("disabled");

    setZoom(xmin, xmax, ymin, ymax);
}

//document.addEventListener("click", getClickPosition, false);

$(document).ready(onReady);

// ================================== Pixel mapping =================================================

var codedMap = "966676:6:A;68656364626Y?\\?;A=A<AGADAN4K4i5g5h5o506W?Z?]?_?>A@A?AJAIAFACAM4J4H4f5d5e5l5m5n516X?[?^?N?P?AA1ABAVAUAKAHAEAO4L4I4G4E4c5a5b5M6j5k5V6Y6\\6_6G?J?O?Q?S?2A4A3AYAXAWAbA_AnAkAhA3404F4D4B4`5^5_5J6K6L6S6T6W6Z6]6E?H?K?M?R?T?V?5A7A6A\\A[AZAeAdAaA^AmAjAgA24o3m3C4A4?4]5[5\\5G6H6I6P6Q6R6U6X6[6^6F?I?L?<?>?U?;@=@8Ah@9AMALA]ABCACfAcA`AoAlAiA4414n3l3<4@4>425Z5X5Y5D6E6F698N6O608E8H8K8H>K>N>?>B>=???A?<@>@@@i@k@j@PAOANAECDCCC<C9CZCWCTC=3:3734313=4;4943515o4W5U5V5A6B6C6687888m7n7C8F8I8F>I>L>=>@>C>E>@?B?D??@A@C@l@n@m@SARAQAHCGCFC?C>C;C8CYCVCSC<393633303n2:4846405n4l4T5R5S5>6?6@6384858j7k7l7o7D8G8J8G>J>M>>>A>D>4>6>C?3?5?B@2@4@o@_@0ALBKBTA0CoBIC8D7D@C=C:C[CXCUC>3;3835323o2m2k27454j3m4k4i4Q5O5P5C7<6=6g71828o8h7i7S9<8?8B8Q>T>W>@=C=F=e<h<5>7>9>4?6?8?3@5@7@`@b@a@OBNBMB3C2C1C;D:D9D_D\\DQCNCKCF3C3@3>2;282Z1W1l2j2h2k3i3g3j4h4f4N5L5M5@7A7B7d7e7f7l8m8n8P9Q9:8=8@8O>R>U>>=A=D=c<f<i<k<8>:><>7?9?;?6@8@:@c@e@d@RBQBPB6C5C4C>D=D<DbDaD^D[DPCMCJCE3B3?3=2:272Y1V1T1i2g2e2h3f3d3g4e4c4K5I5J5=7>7?7a7b7c7i8j8k8M9N9O9R9;8>8A8P>S>V>?=B=E=d<g<j<Z<\\<;>k=m=:?j>l>9@i?k?f@V@g@CBBBSBgBfB7CoCnC?DSDRDcD`D]DRCOCLCG3D3A3?2<292[1X1U1S1Q1f2d2b2e3c3a3d4b4`4H5F5G5:7;7<7^7_7`7f8g8h8J9K9L97:::=:@:C:F:I:I=L=O=7=:===n<1=[<]<_<l=n=0>k>m>o>j?l?n?W@Y@X@FBEBDBjBiBhB2D1D0DVDUDTDCE@EOELEIEXEUERE5222o1l1i1f1c1`1R1P1N1c2a2_2b3`3^3a4_4]4<5:5;5778797[7\\7]7c8d8e8G9H9I94:5:8:;:>:A:D:G:G=J=M=5=8=;=l<o<2=4=^<`<b<o=1>3>n>0?2?m?o?1@Z@\\@[@IBHBGBmBlBkB5D4D3DYDXDWDFEEEBE?ENEKEHEWETEQE4212n1k1h1e1b1_1]1O1M1K1`2^2\\2_3]3[3^4\\4Z4957585475767X7Y7Z7`8a8b8D9E9F91:2:3:6:9:<:?:B:E:H:H=K=N=6=9=<=m<0=3=d;f;a<H<J<2>b=d=1?a>c>0@`?b?]@D@^@:B9BJB^B]BnBfCeC6DJDIDZDnDmDGEDEAEPEMEJEYEVESE623202m1j1g1d1a1^1\\1[0L1J1?1]2[2Y2\\3Z3X3[4Y4W4654555172737U7V7W7]8^8_8A9B9C9e9o90:n9g:j:m:L:O:R:U:X:[:];`;c;T;W;Z;7<9<e;g;i;I<K<M<c=e=g=b>d>f>a?c?e?E@G@F@=B<B;BaB`B_BiChCgCMDLDKD1E0EoD9E7E<F9F6FaE^E[EjEgEdER0O0L0I0F0C0n0l0\\0Z0X0@1>1<1Z2X2V2Y3W3U3X4V4T4E5C5D5n6o607R7S7T7Z8[8\\8>9?9@9b9c9d9l9m9e:h:k:J:M:P:S:V:Y:[;^;a;R;U;X;6<8<:<;<h;j;l;L<N<P<f=h=j=e>g>i>d?f?h?N@Q@H@@B?B>BdBcBbBlCkCjCPDODND4E3E2E;E:E8E6E;F8F5F`E]EZEiEfEcEQ0N0K0H0E0B0m0k0j0Y0W0U0=1;191W2U2S2V3T3R3U4S4Q4B5@5A5k6l6m6O7P7Q7W8X8Y8;9<9=9_9`9a9j9k9\\:_:f:i:l:K:N:Q:T:W:Z:\\;_;b;S;V;Y;C<E<G<<<m;k;1<2<O<Q<S<i=[=P=h>X>Z>g?M@O@R@U@S@J@I@AB1B0BeBTB\\CmCAD@DQDiDhD5EdD<E4F2F0F=F:F7FbE_E\\EkEhEeES0P0M0J0G0D011o0h0g0e0V0T0`0:181Q2T2R2H2S3Q3P3R4P4K3?5=5>5c6i6j6h6M7N7L7U8V8T899:9W9]9^9\\9g9h9i9]:`:c:5;2;o:>;;;8;P;M;J;G;D;A;?<A<D<F<=<><n;o;3<5<R<T<Y=Z=Q=R=Y>[>\\>^>P@T@L@K@5B4B3B2BVBUB^C]CCDBDkDjDfDeD>E=E3F1FnElE@FCFFFIFLFOF;0>0A0205080513101i0f0d0c0b0_0I1H1D1P2O2G2F2D2O3N3M3J3H3a6b6e6f6g6I7J7K7R8S8397989V9Y9Z9[9f9^:a:d:4;1;n:=;:;7;O;L;I;F;C;@;@<B<0<4<U<V<X<`=]=\\=S=U=W=]>_>`>8B7B6BZBXBWB_CaC`CFDEDDDlDgDoEmE>FAFDFGFJFMF90<0?0003060714121a0^0]0G1C1B1N2L2E2C2B2@2L3I3`6d6E7F7G7H7O8Q8192969T9U9X96;3;0;?;<;9;Q;N;K;H;E;B;W<Y<^=_=a=T=V=X=\\B[BYBdCcCbCHDGD?FBFEFHFKFNF:0=0@0104070F1E1A1M2K2J2I2A2D7L8M8N8P809495961b:";
var map = new Array(1440);

// first: decode the pixel mapping!
var sum = 1036080;
for (var i=0; i<1440; i++)
{
    var d0 = codedMap.charCodeAt(i*2)  -48;
    var d1 = codedMap.charCodeAt(i*2+1)-48;

    map[i] = d0 | (d1<<6);
    sum -= map[i];
}
if (sum!=0)
    alert("Pixel mapping table corrupted!");

// ================================== Camera Display ================================================

function hueToRGB(hue)
{
    hue /= 3;
    hue %= 6;

    if (hue<1) return parseInt(255*hue,     10);
    if (hue<3) return parseInt(255,         10);
    if (hue<4) return parseInt(255*(4-hue), 10);

    return 0.
}

function hueToHex(flt)
{
    var s = hueToRGB(flt).toString(16);
    return s.length==2 ? s : "0"+s;
}

function HLStoRGB(hue)
{
    if (isNaN(hue))
        return "fff";

    if (hue<0)
        return "eef"; // 555

    if (hue>1)
        return "700"; // 666

    hue *= 14;

    var sr = hueToHex(20-hue);
    var sg = hueToHex(14-hue);
    var sb = hueToHex(26-hue);

    return sr+sg+sb;
}

function drawHex(ctx, x, y, col, min, max)
{
    if (col===undefined || col===null)
        return;

    var lvl = (col-min)/(max-min);

    ctx.fillStyle = "#"+HLStoRGB(lvl);

    ctx.save();

    ctx.translate(x, y);
    ctx.scale(1/2, 1/3);

    ctx.beginPath();
    ctx.moveTo( 1,  1);
    ctx.lineTo( 0,  2);
    ctx.lineTo(-1,  1);
    ctx.lineTo(-1, -1);
    ctx.lineTo( 0, -2);
    ctx.lineTo( 1, -1);
    ctx.fill();

    ctx.restore();
}

function Position(s, ring, i)
{
    switch (s)
    {
    case 1: this.x =  ring     - i*0.5;  this.y =       + i; break;
    case 2: this.x =  ring*0.5 - i;      this.y =  ring    ; break;
    case 3: this.x = -ring*0.5 - i*0.5;  this.y =  ring - i; break;
    case 4: this.x = -ring     + i*0.5;  this.y =       - i; break;
    case 5: this.x = -ring*0.5 + i;      this.y = -ring    ; break;
    case 0: this.x =  ring*0.5 + i*0.5;  this.y = -ring + i; break;
    }
    this.d = (function () { return this.x*this.x + this.y*this.y*3/4; });
}

function drawFullCam(id)
{
    var canv = document.getElementById(id);
    if (!canv)
        return;

    var ctx = canv.getContext("2d");

    ctx.clearRect(0, 0, canv.width, canv.height);

    // ======================= Draw Graphics ======================

    var data = canv.dataRel;
    if (!data)
        return;

    var min = (canv.zmin-canv.min)/canv.max;
    var max = (canv.zmax-canv.min)/canv.max;

    var scale = 83;

    var w = Math.min(canv.width/scale, canv.height/scale);

    ctx.save();
    ctx.translate(canv.width/2, canv.height/2);
    ctx.scale(w*2, w*2);
    // ctx.rotate(Math.PI/3);

    ctx.scale(1, Math.sqrt(3)/2);
    ctx.translate(-0.5, 0);

    drawHex(ctx, 0, 0, data[map[0]], min, max);

    var cnt = 1;
    for (var ring=1; ring<24; ring++)
    {
        for (var s=0; s<6; s++)
        {
            for (var i=1; i<=ring; i++)
            {
                var pos = new Position(s, ring, i);
                if (pos.d() - pos.x > 395.75)
                    continue;

                drawHex(ctx, pos.x, pos.y, data[map[cnt]], min, max);
                cnt++;
            }
        }
    }

    drawHex(ctx, 7, -22, data[map[1438]], min, max);
    drawHex(ctx, 7,  22, data[map[1439]], min, max);

    ctx.restore();

    // ======================= Draw Legend ======================

    var lmin = parseFloat(canv.min).toFixed(1);
    var lmax = parseFloat(canv.max).toFixed(1);

    var v0 = parseFloat(canv.zmin);
    var v1 = parseFloat(canv.zmax);

    var diff = v1-v0;

    var cw = canv.width;
    //var ch = canv.height;

    ctx.font         = "8pt Arial";
    ctx.textAlign    = "right";
    ctx.textBaseline = "top";

    for (var i=0; i<11; i++)
    {
        ctx.strokeStyle = "#"+HLStoRGB(i/10);
        ctx.strokeText((v0+diff*i/10).toPrecision(3), cw-5, 125-i*12);
    }

    var mw = Math.max(ctx.measureText(lmin).width,
                      ctx.measureText(lmax).width);

    ctx.textBaseline = "top";
    ctx.strokeStyle  = "#000";

    //ctx.strokeText(min, 5+mw, 5+24);
    ctx.strokeText(lmin, 5+mw, 5+12);
    ctx.strokeText(lmax, 5+mw, 5);
}

