#!/bin/bash

# new version of the script to run on newdaq

# option whether to fill all row or only those where information is missing
# $doupdate might be given as environment variable
if [ "$doupdate" = "" ]
then
   doupdate="yes" # update all entries (needed when new fields have been added)
   doupdate="no" # fill only entries which are not yet existing (default)
fi

source `dirname $0`/../Sourcefile.sh
printprocesslog "INFO starting $0 with option doupdate="$doupdate

logfile=$runlogpath"/FillAuxData-"$datetime".log"
date >> $logfile

# check if software is available
if ! ls $factpath/fitsdump >/dev/null 2>&1
then 
   printprocesslog "ERROR "$factpath"/fitsdump is not available."
   finish
fi

# get dates
if [ "$certaindate" != "" ]
then
   checkstring=`echo $certaindate | grep -E -o '^20[0-9][0-9]\/[01][0-9]\/[0-3][0-9]$'`
   if [ "$checkstring" = "" ]
   then
      echo "Please give the variable certaindate in the correct format (YYYY/MM/DD)"
      finish
   fi
   getdates $certaindate
else
   # get all night
   #getdates "all"
   # get last 3 nights if hour between 7 and 19h, else only current night
   getdates 3 7 19
fi


printprocesslog "INFO processing the following night(s): "${dates[@]}
echo  `date`": processing the following night(s): "${dates[@]} >> $logfile 2>&1

# mjd of 1970-01-01
#   needed in this script as for 1 day the mjd in the aux files are inconsistent
# mjdref=40587

# known: 
# 2011/11/22 MJDREF in DRIVE empty, Time > 55000
# 2011/11/23 MJDREF in DRIVE not empty, Time > 55000
# 2011/11/24 MJDREF in DRIVE not empty, Time > 15000
# raw files
# 2011/11/21 no MJDREF
# 2011/11/22 MJDREF
# further things: https://www.fact-project.org/logbook/showthread.php?tid=67

# trigger rate has as first value -1, but with using the median it should be fine


function evaluatestatistics()
{
   # $1 variable name
   # $@ statistics
   if [ "$2" = "" ]
   then
      printprocesslog "WARN couldn't get statistics from file $1 for run "$date" "$file
      continue
   fi
   min=
   mean=
   med=
   max=
   rms=
   evaluation=`echo $@ | grep -E -o '\['${1}':0[:]?[0-9]*\]\ Min:\ [-]?[0-9]+[.]?[0-9]*\ Max:\ [-]?[0-9]+[.]?[0-9]*\ Med:\ [-]?[0-9]+[.]?[0-9]*\ Avg:\ [-]?[0-9]+[.]?[0-9]*\ Rms:\ [-]?[0-9]+[.]?[0-9]*[e]?[-]?[0-9]*'`
   if [ "$evaluation" = "" ]
   then
      printprocesslog "WARN empty evaluation of statistic ("$@") for run "$date" "$file
      #echo "   ---> "$@
   fi
   #echo "ev: "$evaluation
   min=`echo $evaluation | grep -E -o 'Min:\ [-]?[0-9]+[.]?[0-9]*' | sed -e 's/Min:\ //'`
   max=`echo $evaluation | grep -E -o 'Max:\ [-]?[0-9]+[.]?[0-9]*' | sed -e 's/Max:\ //'`
   med=`echo $evaluation | grep -E -o 'Med:\ [-]?[0-9]+[.]?[0-9]*' | sed -e 's/Med:\ //'`
   mean=`echo $evaluation | grep -E -o 'Avg:\ [-]?[0-9]+[.]?[0-9]*' | sed -e 's/Avg:\ //'`
   rms=`echo $evaluation | grep -E -o 'Rms:\ [-]?[0-9]+[.]?[0-9]*[e]?[-]?[0-9]+' | sed -e 's/Rms:\ //'`
   #echo "eval: "$min" "$max" "$med" "$mean" "$rms
}

# do filling of aux data 
for date in ${dates[@]}
do 
   auxdir=$auxdata/$date
   rawdir=$rawdata/$date
   runnumber=`echo $date | sed -e 's/\///g'`

   # check if aux files are available from that night
   if ! [ -d $auxdir ]
   then
      printprocesslog "INFO no data available in "$auxdir
      continue
   else
      printprocesslog "INFO processing files in "$auxdir
      #echo "INFO processing files in "$auxdir
   fi

   # check if raw files are available from that night
   #   only needed to get start/stop time
   #   might be removed once the start/stop time comes from the DB
   if ! [ -d $rawdir ]
   then
      printprocesslog "INFO no data available in "$rawdir" -> continue"
      continue
   fi

   # get file numbers from DB
   #   but only for not-corrupted files
   # as aux files are written only once a minute, select only files which are older than 1.5 minutes
   query="SELECT fRunID from RunInfo WHERE fNight="$runnumber" AND fFitsFileErrors=0 AND NOT ISNULL(fRunStop) AND fRunStop < SUBTIME(UTC_TIMESTAMP(), \"00:01:30\")"
   # only runs which are not yet filled 
   if [ "$doupdate" = "no" ]
   then
      query=$query" AND ISNULL(fRightAscension) "
   fi
   printprocesslog "DEBUG get filenumbers from DB: QUERY: "$query
   filenumbers=( `sendquery $query` )
   if [ ${#filenumbers} -eq 0 ]
   then
      printprocesslog "INFO No files found in the DB for night "$date
      continue
   fi

   # get daily fits files 
   trackingfile=$auxdir/$runnumber.DRIVE_CONTROL_TRACKING_POSITION.fits
   if ! [ -e $trackingfile ]
   then 
      printprocesslog "WARN "$trackingfile" not found."
   fi
   
   sourceposfile=$auxdir/$runnumber.DRIVE_CONTROL_SOURCE_POSITION.fits
   if ! [ -e $sourceposfile ]
   then 
      printprocesslog "WARN "$sourceposfile" not found."
   else
      sourceposfiletstarti=`$factpath/fitsdump -h $sourceposfile  2>/dev/null | grep 'TSTARTI' | grep -E -o '[0-9]{5}'`
      sourceposfiletstartf=`$factpath/fitsdump -h $sourceposfile  2>/dev/null | grep 'TSTARTF' | grep -E -o '0[.][0-9]+'`
      if [ $sourceposfiletstarti -gt 30000 ]
      then 
         sourceposfiletstart=`echo " $sourceposfiletstarti + $sourceposfiletstartf - 40587 " | bc -l`
      else
         sourceposfiletstart=`echo " $sourceposfiletstarti + $sourceposfiletstartf " | bc -l`
      fi
   fi
   
   triggerratefile=$auxdir/$runnumber.FTM_CONTROL_TRIGGER_RATES.fits
   if ! [ -e $triggerratefile ]
   then 
      printprocesslog "WARN "$triggerratefile" not found."
   fi
   
   thresholdfile=$auxdir/$runnumber.FTM_CONTROL_STATIC_DATA.fits
   if ! [ -e $thresholdfile ]
   then 
      printprocesslog "WARN "$thresholdfile" not found."
   fi
   
   biasvoltagefile=$auxdir/$runnumber.BIAS_CONTROL_VOLTAGE.fits
   if ! [ -e $biasvoltagefile ]
   then 
      printprocesslog "WARN "$biasvoltagefile" not found."
   fi

   # fill auxiliary information for files
   for filenum in ${filenumbers[@]}
   do
      printprocesslog "INFO processing file number "$runnumber"_"`printf %03d $filenum`
      echo  `date`": processing file number "$runnumber"_"`printf %03d $filenum` >> $logfile 2>&1
      #echo  `date`": processing file number "$runnumber"_"`printf %03d $filenum`
      
      # get information from rawfile
      rawfile=`ls $rawdir/$runnumber"_"\`printf %03d $filenum\`.fits*`
      if ! [ -e $rawfile ]
      then 
         printprocesslog "ERROR: "$rawfile" not found."
         continue
      fi

      #runtype=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep RUNTYPE | grep -E -o "['][a-z-]+[']" | sed -e "s/'//g"`
      #mjdrefraw=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'MJDREF' | grep -E -o '[0-9]{5}'`
      tstarti=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTARTI' | grep -E -o '[0-9]{5}'`
      tstartf=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTARTF' | grep -E -o '0[.][0-9]+'`
      tstopi=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTOPI' | grep -E -o '[0-9]{5}'`
      tstopf=`$factpath/fitsdump -h $rawfile  2>/dev/null | grep 'TSTOPF' | grep -E -o '0[.][0-9]+'`
      if [ "$tstarti" == "" ] || [ "$tstopi" == "" ] || [ "$tstartf" == "" ] || [ "$tstopf" == "" ]
      then 
         printprocesslog "WARN: "$rawfile": one of the following keywords is empty or 0: TSTARTI TSTARTF TSTOPI TSTOPF "
         continue
      fi
      # assuming that at least TSTARTI and TSTOPI are consistent
      #echo $rawfile
      #echo $tstarti
      #echo $tstopi
      #echo $tstartf
      #echo $tstopf
      if [ $tstarti -gt 30000 ]
      then 
         tstart=`echo " $tstarti + $tstartf - 40587 " | bc -l`
         tstart2=`echo " $tstarti + $tstartf - 40587 - 0.00011574 " | bc -l`  # 10 sec
         #tstart2=`echo " $tstarti + $tstartf - 40587 - 0.000023148 " | bc -l` # 2 sec
         tstop=`echo " $tstopi + $tstopf - 40587 " | bc -l`
      else
         tstart=`echo " $tstarti + $tstartf " | bc -l`
         tstart2=`echo " $tstarti + $tstartf - 0.00011574 " | bc -l`  # 10 sec
         #tstart2=`echo " $tstarti + $tstartf - 0.000023148 " | bc -l` # 2 sec
         tstop=`echo " $tstopi + $tstopf " | bc -l`
      fi
      #echo $tstart
      #echo $tstop
      #if [ $runnumber -eq 20111123 ] 
      #then 
      #   # add mjdref for days were aux files were inconsistent
      #   tstart=`echo " $tstart + $mjdref " | bc -l`
      #   tstart2=`echo " $tstart2 + $mjdref " | bc -l`
      #   tstop=`echo " $tstop + $mjdref " | bc -l`
      #fi

      # get information from source_pos file
      if [ -e $sourceposfile ] 
      then 
         sourcename=`$factpath/fitsdump ${sourceposfile} -c Time -c Name --filter='[1]<'${tstop}' && [1]>'${sourceposfiletstart}  2>/dev/null | tail -1 2>&1 | grep -o -E "['][a-zA-Z0-9\ \.\+\-]+[']" | sed -e "s/'//g"`
         if [ "$sourcename" == "" ]
         then
            printprocesslog "INFO couldn't get sourcename ("$sourcename") from "$sourceposfile" for "$runnumber"_"$filenum
         else 
            query="SELECT fSourceKey FROM Source WHERE fSourceName='"$sourcename"'"
            sourcekey=`sendquery`
            if [ "$sourcename" == "" ]
            then
               printprocesslog "WARN couldn't get sourcekey for source "$sourcename" from DB for "$runnumber"_"$filenum
            fi 
         fi
      fi
      
      # build query to update runinfo in DB
      query="UPDATE RunInfo SET "
      
      # fill source key only if available
      if ! [ "$sourcekey" = "" ]
      then
         query=$query" fSourceKey="$sourcekey", "
      else
         query=$query" fSourceKey=NULL, "
      fi
      
      # get information from tracking
      if [ -e $trackingfile ] 
      then 
         # get statistics
         trackingstats=`$factpath/fitsdump $trackingfile -s -c Time -c Ra -c Dec -c Zd -c Az --filter='[1]<'${tstop}' && [1]>'${tstart}  2>/dev/null`
         # RA
         evaluatestatistics "Ra" $trackingstats
         #echo $min"_"$max"_"$
         if [ "$evaluation" != "" ]
         then
            if [ "$min" == "$max" ] 
            then
               query=$query" fRightAscension="$mean
            else
               query=$query" fRightAscension=NULL"
               printprocesslog "WARN for $rawfile RA changes within run (min: "$min", max: "$max")."
            fi
            # Declination
            evaluatestatistics "Dec" $trackingstats
            if [ "$decmin" == "$decmax" ]
            then
               query=$query", fDeclination="$mean
            else
               query=$query", fDeclination=NULL"
               printprocesslog "WARN for $rawfile declination changes within run (min: "$min", max: "$max")."
            fi
         else
            query=$query" fRightAscension=NULL"
            query=$query", fDeclination=NULL"
         fi
         # Zd
         evaluatestatistics "Zd" $trackingstats
         if [ "$evaluation" != "" ]
         then
            query=$query", fZenithDistanceMin="$min
            query=$query", fZenithDistanceMean="$mean
            query=$query", fZenithDistanceMax="$max
         else
            query=$query", fZenithDistanceMin=NULL"
            query=$query", fZenithDistanceMean=NULL"
            query=$query", fZenithDistanceMax=NULL"
         fi
         # Az
         evaluatestatistics "Az" $trackingstats
         if [ "$evaluation" != "" ]
         then
            query=$query", fAzimuthMin="$min
            query=$query", fAzimuthMean="$mean
            query=$query", fAzimuthMax="$max
         else
            query=$query", fAzimuthMin=NULL"
            query=$query", fAzimuthMean=NULL"
            query=$query", fAzimuthMax=NULL"
         fi
      else
         query=$query" fRightAscension=NULL"
         query=$query", fDeclination=NULL"
         query=$query", fZenithDistanceMin=NULL"
         query=$query", fZenithDistanceMean=NULL"
         query=$query", fZenithDistanceMax=NULL"
         query=$query", fAzimuthMin=NULL"
         query=$query", fAzimuthMean=NULL"
         query=$query", fAzimuthMax=NULL"
      fi
   
      # get information from trigger
      if [ -e $triggerratefile ] 
      then 
         # get statistics
         triggerstats=`$factpath/fitsdump $triggerratefile -s -c Time -c TriggerRate --filter='[1]<'${tstop}' && [1]>'${tstart}  2>/dev/null`
         evaluatestatistics "TriggerRate" $triggerstats
         if [ "$evaluation" != "" ]
         then
            query=$query", fTriggerRateMedian="$med
         else
            query=$query", fTriggerRateMedian=NULL"
         fi
      else
         query=$query", fTriggerRateMedian=NULL"
      fi
      
      # get information from thresholds
      if [ -e $thresholdfile ] 
      then 
         # get statistics
         thresholdstats=`$factpath/fitsdump $thresholdfile -s -c Time -c PatchThresh --filter='[1]<'${tstop}' && [1]>'${tstart}  2>/dev/null`
         evaluatestatistics "PatchThresh" $thresholdstats
         if [ "$evaluation" = "" ]
         then
            thresholdstats=`$factpath/fitsdump $thresholdfile -s -c Time -c PatchThresh --filter='[1]<'${tstop}' && [1]>'${tstart2}  2>/dev/null`
            #echo "$factpath/fitsdump $thresholdfile -s -c Time -c PatchThresh --filter='[1]<'${tstop}' && [1]>'${tstart2}  2>/dev/null"
            evaluatestatistics "PatchThresh" $thresholdstats
         fi
         if [ "$evaluation" != "" ]
         then
            query=$query", fThresholdMedian="$med
         else
            query=$query", fThresholdMedian=NULL"
         fi
      else
         query=$query", fThresholdMedian=NULL"
      fi

      # get information from bias: U
      if [ -e $biasvoltagefile ] 
      then 
         if [ $runnumber -gt 20120324 ]
         then
            biasstats=`$factpath/fitsdump $biasvoltagefile -s -c Time -c Uout --filter='[1]<'${tstop}' && [1]>'${tstart}  2>/dev/null`
            evaluatestatistics "Uout" $biasstats
            if [ "$evaluation" = "" ]
            then
               biasstats=`$factpath/fitsdump $biasvoltagefile -s -c Time -c Uout --filter='[1]<'${tstop}' && [1]>'${tstart2}  2>/dev/null`
               evaluatestatistics "Uout" $biasstats
            fi
         else
            biasstats=`$factpath/fitsdump $biasvoltagefile -s -c Time -c U --filter='[1]<'${tstop}' && [1]>'${tstart}  2>/dev/null`
            evaluatestatistics "U" $biasstats
            if [ "$evaluation" = "" ]
            then
               biasstats=`$factpath/fitsdump $biasvoltagefile -s -c Time -c U --filter='[1]<'${tstop}' && [1]>'${tstart2}  2>/dev/null`
               evaluatestatistics "U" $biasstats
            fi
         fi
         if [ "$evaluation" != "" ]
         then
            query=$query", fBiasVoltageMedian="$med
         else
            query=$query", fBiasVoltageMedian=NULL"
         fi
      else
         query=$query", fBiasVoltageMedian=NULL"
      fi
      
      # add where condition
      query=$query" WHERE fNight="$runnumber" AND fRunID="$filenum
      #echo $query

      # send query to DB
      sendquery >/dev/null
   done
done

finish


