/********************************************************************\

  Alarm handler of the Evidence Control System

  - Checks periodically if all required servers are up
    (later it should try to start them if not)
  - Listens to the 'Status' service of each server.
  - A text describing the state of all servers is published as DIM service.
    The states are described in StateString[].
  - A master alarm (indicating most severe of individual alarms) is published. 
    
  Oliver Grimm, January 2010

\********************************************************************/

#define SERVER_NAME "Alarm"
#include "Evidence.h"

#define SUMMARYSIZE 10000	// Bytes for alarm summary text

const char* StateString[] = {"OK", "WARN", "ERROR", "FATAL", "UNAVAILABLE"};

//
// Data handling class
//
class AlarmHandler : public DimClient, public EvidenceServer {
    
    DimStampedInfo **StatusService;

    void infoHandler();

  public:
    AlarmHandler();
    ~AlarmHandler();

	DimService *Summary, *Master;
	
	char *AlarmSummary;
	int MasterAlarm;
	int *State;    
    char **Server;
    unsigned int NumServers;
    char *ServerList; 
	
	void UpdateAlarmSummary();
}; 

// Constructor
AlarmHandler::AlarmHandler(): EvidenceServer(SERVER_NAME) {

  AlarmSummary = new char [SUMMARYSIZE];
  MasterAlarm = 0;

  // Create DIM services
  Summary = new DimService(SERVER_NAME"/Summary", AlarmSummary);
  Master = new DimService(SERVER_NAME"/MasterAlarm", MasterAlarm);

  // Copy original list of servers to observe
  char *ServerNames = GetConfig("servers");
  ServerList = new char [strlen(ServerNames)+1];
  strcpy(ServerList, ServerNames);
  
  // Extract DIM servers to observe
  Server = new char* [strlen(ServerNames)];
  NumServers = 0;
  char *NextToken = strtok(ServerNames, " \t");
  while (NextToken != NULL) {
    Server[NumServers++] = NextToken; // Subscribe with handler
    NextToken = strtok(NULL, " \t");     
  }

  // Subscribe with handler to 'Status' service of all servers
  StatusService = new DimStampedInfo* [NumServers];
  State = new int [NumServers];
  
  for (int i=0; i<NumServers; i++) {
    char *Buffer = new char [strlen(Server[i])+10];
    strcpy(Buffer, Server[i]);
    strcat(Buffer, "/Status");
    StatusService[i] = new DimStampedInfo(Buffer, NO_LINK, this);
    delete[] Buffer;
	
	State[i] = 0;
  }
}

// Destructor
AlarmHandler::~AlarmHandler() {

  for (int i=0; i<NumServers; i++) delete StatusService[i];
  delete[] StatusService;
  delete Master;
  delete Summary;
  delete[] State;
  delete[] Server;
  delete[] ServerList;
  delete[] AlarmSummary;
}

// Print messages of status changes to screen and update status string
void AlarmHandler::infoHandler() {

  // Identify status service
  for (int i=0; i<NumServers; i++) if (getInfo() == StatusService[i]) {

	// Ignore DIS_DNS (has no status service)
	if (strcmp(getInfo()->getName(),"DIS_DNS/Status") == 0) return;
	
	// Update State: unavailable or current severity of status  
	if (!ServiceOK(getInfo())) State[i] = 4;
	else {
	  State[i] = *(getInfo()->getString()+getInfo()->getSize()-1);

	  // Print message
	  time_t RawTime = getInfo()->getTimestamp();
	  struct tm *TM = localtime(&RawTime);
	  printf("%s (%02d:%02d:%02d): %s\n", getInfo()->getName(), TM->tm_hour,
		TM->tm_min, TM->tm_sec, getInfo()->getString());	  
	}
	UpdateAlarmSummary();
  }  
}


// Update alarm status summary
void AlarmHandler::UpdateAlarmSummary() {
  
  int Offset = 0;
  MasterAlarm = 0;
   
  for (int i=0; i<NumServers; i++) {
    snprintf(AlarmSummary+Offset, SUMMARYSIZE-Offset, "%s: %s (%d)\n", Server[i], State[i]<=4 ? StateString[State[i]] : "unknown", State[i]);
	Offset += strlen(AlarmSummary+Offset);
	if (State[i] > MasterAlarm) MasterAlarm = State[i];
  }
  Summary->updateService();
  Master->updateService();
}

//	    
// Main program
//
int main() {
    
  DimBrowser Browser;
  char *ServerName, *Node;
  bool Exists;
  
  // Static declaration ensures calling of destructor by exit()
  static AlarmHandler Alarm; 
  
  // Check periodically if servers are up
  while(!Alarm.ExitRequest) {
    for (int i=0; i<Alarm.NumServers; i++) {
      Exists = false;
      Browser.getServers();
      while (Browser.getNextServer(ServerName, Node) == 1) {
        if (strcmp(ServerName, Alarm.Server[i]) == 0) Exists = true;
      }
      if (!Exists) Alarm.State[i] = 4;
    }
    
    Alarm.UpdateAlarmSummary();
    sleep(atoi(Alarm.GetConfig("period")));
  }
}
