#! /usr/bin/env python2
# coding: utf-8
"""plot_ratescan

Creates a plot of a FACT ratescan, by evaluating the file YYYYMMDD.RATE_SCAN_DATA.fits
of "yesterday night" by default. The resulting PNG is stored in a hardcoded location by default.
The location currently is:
  /loc_data/analysis/ratescan_plots/{y}/{m}/{d}/{y}{m}{d}.png

This script is intended to be run on daq, once every morning, to create the ratescan plot of 
yesterday night. But in can of course also be called manually. (You might want to change the paths in that case).
This script is planned be called by the cron daemon running for the user 'fact' on 'daq' at the moment.


Usage:
  plot_ratescan.py [-d DATE]
  plot_ratescan.py (-h | --help) 
  plot_ratescan.py --version

Options:
  -h --help     Show this screen.
  -d --date DATE  date in typical fact format: YYYYMMDD
"""
from docopt import docopt
import os, os.path
import datetime
from astropy.io import fits
import numpy as np

# import matplotlib, without a running X server.
import matplotlib as mpl
mpl.use('Agg')
import matplotlib.pyplot as plt

outpath = '/loc_data/analysis/ratescan_plots/{y:04d}/{m:02d}/{d:02d}/{y:04d}{m:02d}{d:02d}.png'
infile_path = '/loc_data/aux/{y:04d}/{m:02d}/{d:02d}/{y:04d}{m:02d}{d:02d}.RATE_SCAN_DATA.fits'

def make_outpath(file_path):
  directory = os.path.dirname(file_path)
  if not os.path.exists(directory):
    os.makedirs(directory)

def plot_ratescan(datetime_instance):
    dt = datetime_instance
    year, month, day = dt.year, dt.month, dt.day

    file_path = infile_path.format(y=year, m=month, d=day)
    #print "opening file_path:", file_path
    try:
        f = fits.open(file_path)
    
        d = f[1].data

        plt.figure(figsize=(8, 6), dpi=80, facecolor='w', edgecolor='k')
        plt.semilogy( d['Threshold'], d['TriggerRate'], 'o:')
        plt.grid()

        plt.xlabel('Threshold [DAC]')
        plt.ylabel('camera rate [Hz]')
        plt.ylim(1,1e9)
        plt.xlim(0,1000)
    
        file_path = outpath.format(y=year, m=month, d=day)
        make_outpath(file_path)
        plt.savefig(file_path, dpi=80)
        print file_path, "has been written"

    except IOError as e: 
        print e

if __name__ == '__main__':
    arguments = docopt(__doc__, version='plot_ratescan 0.9')
    
    date = arguments['--date']
    now = datetime.datetime.utcnow()
    if date is None:
        # We assume it is the morning of the next day, and we 
        # need to get y,m,d of yesterday.
        yesterday = now - datetime.timedelta(hours=24)
        plot_ratescan(yesterday)
    else:
        # The user supplied a date-string, which we need to parse
        try:
            ymd = map(int, date.split('/'))
            date = datetime.datetime(*ymd)
          
            if date > now:
                print date, "is in the future, now is:", now
                print "Please choose a date in the past"
            if date.year <= 2011:
                print "At the date you chose, there were no ratescans taken. Your choice:", date
                print "Please choose a better date"
        
            plot_ratescan(date)
        except ValueError as e:
            print e
        
