#! /usr/bin/env python2
# coding: utf-8
"""plot_ratescan

Creates a plot of a FACT ratescan, by evaluating the file YYYYMMDD.RATE_SCAN_DATA.fits
of "yesterday night" by default. The resulting PNG is stored in a hardcoded location by default.
The location currently is:
  /loc_data/analysis/ratescan_plots/{y}/{m}/{d}/{y}{m}{d}.png

This script is intended to be run on daq, once every morning, to create the ratescan plot of 
yesterday night. But in can of course also be called manually. (You might want to change the paths in that case).
This script is planned be called by the cron daemon running for the user 'fact' on 'daq' at the moment.


Usage:
  plot_ratescan.py [-d DATE] [--cronjob | --verbose]
  plot_ratescan.py (-h | --help) 
  plot_ratescan.py --version
  plot_ratescan.py all [--verbose]

Options:
  -h --help       Show this screen.
  -d --date DATE  date in typical fact format: YYYYMMDD
  --cronjob       Suppress most output for humans, only create output in case of real malfunctions.
  --verbose       Create even more output
"""
from docopt import docopt
import os, os.path
import datetime
from astropy.io import fits
import numpy as np
import glob

# import matplotlib, without a running X server.
import matplotlib as mpl
mpl.use('Agg')
import matplotlib.pyplot as plt

outpath = '/loc_data/analysis/ratescan_plots/{y:04d}/{m:02d}/{d:02d}/{y:04d}{m:02d}{d:02d}.png'
infile_path = '/loc_data/aux/{y:04d}/{m:02d}/{d:02d}/{y:04d}{m:02d}{d:02d}.RATE_SCAN_DATA.fits'
pointing_file_path_template = '/loc_data/aux/{y:04d}/{m:02d}/{d:02d}/{y:04d}{m:02d}{d:02d}.DRIVE_CONTROL_POINTING_POSITION.fits'

def make_outpath(file_path):
  directory = os.path.dirname(file_path)
  if not os.path.exists(directory):
    os.makedirs(directory)

def plot_ratescan(datetime_instance, arguments):
    dt = datetime_instance
    year, month, day = dt.year, dt.month, dt.day

    file_path = infile_path.format(y=year, m=month, d=day)
    pointing_file_path = pointing_file_path_template.format(y=year, m=month, d=day)
    if arguments['--verbose']:
        print "opening file_path:", file_path

    try:
        f = fits.open(file_path)
        p_file = fits.open(pointing_file_path)
        d = f[1].data
        p_data = p_file[1].data
        
        ids = d['Id']
        threshold = d['Threshold']
        trigger_rate = d['TriggerRate']
        id_set = sorted(list(set(ids)))
        
        plt.figure(figsize=(8, 6), dpi=80, facecolor='w', edgecolor='k')
        for _id in id_set:
            starttime = _id / (24. * 3600)
            stoptime = (_id+30) / (24. * 3600)
            idx = np.where((p_data['Time'] >= starttime) * (p_data['Time'] < stoptime))[0]
            zd = p_data['Zd'][idx].mean()
            az = p_data['Az'][idx].mean()
            # _id is starttime of ratescan in unix timestamp.
            dt = datetime.datetime.utcfromtimestamp(_id)
            try:
                tresh = threshold[ids==_id]
                rate = trigger_rate[ids==_id]
                plt.semilogy(
                    tresh[rate > 0.], 
                    rate[rate > 0.], 
                    'o:', 
                    label=dt.strftime("%H:%M:%S")+" Zd:{0:.1f}".format(zd)
                    )
            except Exception as e:
                print e
                print 'trigger_rate', trigger_rate[ids==_id], (trigger_rate[ids==_id] < 0.).any()
        plt.grid()
        plt.legend()

        plt.xlabel('Threshold [DAC]')
        plt.ylabel('camera rate [Hz]')
        plt.ylim(1,1e9)
        plt.xlim(0,1000)
    
        file_path = outpath.format(y=year, m=month, d=day)
        make_outpath(file_path)
        plt.savefig(file_path, dpi=80)
        if not arguments['--cronjob']:
            # the cronjob does not like too much output
            print file_path, "has been written"

    except IOError as e: 
        if not arguments['--cronjob']:
            # The cronjob might regularly try to open ratescans
            # which have not yet been taken. So ... well, we don't care.
            print e

if __name__ == '__main__':
    arguments = docopt(__doc__, version='plot_ratescan 0.9.1')
    if arguments['--verbose']:
        print arguments

    if arguments['all']:
        my_list = sorted(glob.glob('/loc_data/aux/*/*/*/*.RATE_SCAN_DATA.fits'))
        file_names = []
        for x in my_list:
            if 'bad' in x:
                continue
            if '_' in x.split('/')[-1].split('.')[0]:
                continue
            file_names.append(x)
        
            ymd = map(int, x.split('/')[3:6])
            date = datetime.datetime(*ymd)
            try:
                plot_ratescan(date, arguments)
            except Exception as e:
                print e

    date = arguments['--date']
    now = datetime.datetime.utcnow()
    if date is None:
        # We assume it is the morning of the next day, and we 
        # need to get y,m,d of yesterday.
        yesterday = now - datetime.timedelta(hours=24)
        plot_ratescan(yesterday, arguments)
    else:
        # The user supplied a date-string, which we need to parse
        try:
            ymd = map(int, date.split('/'))
            date = datetime.datetime(*ymd)
          
            if date > now:
                print date, "is in the future, now is:", now
                print "Please choose a date in the past"
            if date.year <= 2011:
                print "At the date you chose, there were no ratescans taken. Your choice:", date
                print "Please choose a better date"
        
            plot_ratescan(date, arguments)
        except ValueError as e:
            print e
        
