/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Harald Kornmayer 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MHMcCT1CollectionAreaCalc
//
//
//////////////////////////////////////////////////////////////////////////////

#include "MMcCT1CollectionAreaCalc.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MMcEvt.hxx"

#include "MHMcCT1CollectionArea.h"

ClassImp(MMcCT1CollectionAreaCalc);

MMcCT1CollectionAreaCalc::MMcCT1CollectionAreaCalc(const char *input,
                                             const char *name, const char *title)
{
    fName  = name  ? name  : "MMcCT1CollectionAreaCalc";
    fTitle = title ? title : "Task to calculate the collection area";

    AddToBranchList("MMcEvt.fEnergy");
    AddToBranchList("MMcEvt.fImpact");
    AddToBranchList("MMcEvt.fTelescopeTheta");
} 

Bool_t MMcCT1CollectionAreaCalc::PreProcess (MParList *pList)
{
    // connect the raw data with this task

    fMcEvt = (MMcEvt*)pList->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << err << dbginf << "MMcEvt not found... exit." << endl;
        return kFALSE;
    }

    fCollArea = (MHMcCT1CollectionArea*)pList->FindCreateObj("MHMcCT1CollectionArea");
    if (!fCollArea)
        return kFALSE;

    fTotalNumSimulatedShowers =  0;
    fAllEvtsTriggered         = kTRUE;
    fAllEvtsAtSameTheta       = kTRUE;
    fTelescopeTheta           = -1.;

    return kTRUE;
}

Bool_t MMcCT1CollectionAreaCalc::Process()
{
    const Double_t energy = fMcEvt->GetEnergy();
    const Double_t impact = fMcEvt->GetImpact()/100.;

    if (fTelescopeTheta < 0.)
      fTelescopeTheta = fMcEvt->GetTelescopeTheta();

    if (fTelescopeTheta != fMcEvt->GetTelescopeTheta())
      {
	fAllEvtsAtSameTheta = kFALSE;
      }

    fCollArea->FillSel(energy, impact);

    return kTRUE;
}

Bool_t MMcCT1CollectionAreaCalc::PostProcess()
{ 
  if ( ! fAllEvtsAtSameTheta )
    {
        *fLog << err << dbginf << "ERROR: input data contain events at different zenith angles (not supported)... exiting." << endl;
	return kFALSE;
    }
  //
  //   do the calculation of the effective area
  //
  *fLog << inf << "Calculation Collection Area..." << endl;
  
  fCollArea->CalcEfficiency(fTelescopeTheta);

  return kTRUE;
}

