import sys
import numpy as np
import matplotlib as mpl
mpl.use('Agg')
import matplotlib.pyplot as plt
import pandas as pd
from astropy.io import fits
from calculate_burst_ratio_file import get_trigger_rates, fjd


night_string = sys.argv[1]
night_int = int(night_string)
OFFSET=719163

runs = pd.DataFrame.from_csv(
    'foo.bar', 
    infer_datetime_format=True, 
    parse_dates=['fRunStart', 'fRunStop']
    ).dropna()

runs = runs.loc[runs.fNight==night_int]

trigger_rates = get_trigger_rates(night_int, 
    base_path='/fact/aux/')

fig, ax1 = plt.subplots(figsize=(16, 9))
ax1.set_xlabel("Time")
ax1.set_yscale('log')
ax1.set_ylabel("Rate [Hz]")
ax1.grid()

plt.title("basic Ratescan information from: " + night_string)

plot_0 = ax1.plot_date(
    trigger_rates['Time']+OFFSET, 
    np.median(trigger_rates['BoardRate'], axis=1), 
    'r.', ms=4.,
    label="median BoardRate"
    )

plot_1 = ax1.plot_date(
    trigger_rates['Time']+OFFSET, 
    trigger_rates['TriggerRate'], 
    'g.', ms=4.,
    label="Camera TriggerRate"
    )

for runs_index in range(len(runs)):
    this_run = runs.iloc[runs_index]
    run_number = this_run['fRunID']

    a = fjd(this_run['fRunStart'])+OFFSET
    e = fjd(this_run['fRunStop'])+OFFSET
    
    ax1.axvline(a, color='g', linestyle='dashed')
    ax1.axvline(e, color='r', linestyle='dashed')
    ax1.text(a + (e-a)/2., 900, 
        "{0}".format(run_number),
        horizontalalignment='center',
        verticalalignment='center',
        fontsize=7,
        )

    med = this_run['median_of_board_rates']
    std_dev = this_run['std_dev_of_board_rates']
    
    ax1.plot_date(
        [a, e], 
        [med+3*std_dev, med+3*std_dev], 
        'm-')
    


plt.xlim((
    fjd(runs.iloc[0]['fRunStart'])+OFFSET,
    fjd(runs.iloc[-1]['fRunStop'])+OFFSET 
    ))
plt.ylim(1e-1, 1e6)

plt.legend(
    handles=[plot_0[0], plot_1[0]],
    loc='lower left')

plt.tight_layout()
plt.savefig("board_rate_overview_"+night_string+'.png', 
    dpi=300)

plt.figure()
plt.title("fHighBoardTriggerRateRatio distribution from {0}".format(night_int))
plt.hist(runs.fBoardTriggerRateRatioAboveThreshold.values, 
    bins=np.arange(0,0.6,1./60.), 
    log=True )
plt.grid()
plt.xlabel("fHighBoardTriggerRateRatio")
plt.savefig("ratio_dist_"+night_string+'.png')
