/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek  03/2003 <wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MPointingCorr                                                          //
//                                                                         //
//  This is a task to do the pointing correction                           //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MPointingCorr.h"

#include "MParList.h"

#include "MMcEvt.hxx"
#include "MSrcPosCam.h"
#include "MGeomCam.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MPointingCorr);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MPointingCorr::MPointingCorr(const char *name, const char *title,
                             const char *srcname)
{
    fName  = name  ? name  : "MPointingCorr";
    fTitle = title ? title : "Task to do the pointing correction";

    fSrcName = srcname;
}

// --------------------------------------------------------------------------
//
// 
// 
// 
//
Bool_t MPointingCorr::PreProcess(MParList *pList)
{
    MGeomCam *geom = (MGeomCam*)pList->FindObject("MGeomCam");
    if (!geom)
        *fLog << warn << GetDescriptor() << ": No Camera Geometry available. Using mm-scale for histograms." << endl;
    else
    {
        fMm2Deg = geom->GetConvMm2Deg();
    }


    fMcEvt = (MMcEvt*)pList->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << dbginf << "MMcEvt not found... aborting." << endl;
        return kFALSE;
    }


    fSrcPos = (MSrcPosCam*)pList->FindObject(fSrcName, "MSrcPosCam");
    if (!fSrcPos)
    {
        *fLog << err << dbginf << fSrcName << " [MSrcPosCam] not found... aborting." << endl;
        return kFALSE;
    }


    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Do the pointing correction
//
// the parametrization is for Mkn421 2001 data (Daniel Kranich)
// 
Bool_t MPointingCorr::Process()
{
   // fhourangle is the hour angle [degrees]
   // (cx, cy) is the source position in the camera [mm]
   //
   Float_t fhourangle = fMcEvt->GetOtherCphFraction();
   Float_t cx = -0.05132 - 0.001064 * fhourangle 
                         - 3.530e-6 * fhourangle * fhourangle;
   cx /= fMm2Deg;

   Float_t cy = -0.04883 - 0.0003175* fhourangle
                         - 2.165e-5 * fhourangle * fhourangle;
   cy /= fMm2Deg;

   fSrcPos->SetXY(cx, cy);

   //*fLog << "MPointingCorr::Process; fhourangle, cx, cy, fMm2Deg = "
   //      << fhourangle << ",  " << cx << ",  " << cy << ",  " 
   //      << fMm2Deg << endl;

   return kTRUE;
}
// --------------------------------------------------------------------------
//
//  
//
Bool_t MPointingCorr::PostProcess()
{
    return kTRUE;
}
//============================================================================






