#ifndef MARS_MParContainer
#define MARS_MParContainer

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// MParContainer                                                        //
//                                                                      //
// The basis for all parameter containers                               //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
#ifndef MARS_MAGIC
#include "MAGIC.h"
#endif

#ifndef ROOT_TObject
#include <TObject.h>
#endif
#ifndef ROOT_TString
#include <TString.h>
#endif

class ofstream;
class ifstream;

class TEnv;
class TDataMember;
class TMethodCall;

class MParContainer : public TObject
{
protected:
    TString fName;        // parameter container identifier (name)
    TString fTitle;       // parameter container title

    MLog   *fLog;         // The general log facility for this object, initialized with the global object

private:
    enum {
        kIsSavedAsPrimitive    = BIT(15),
    };

    Bool_t  fReadyToSave; // should be set to true if the contents of the container is changed somehow

    virtual void StreamPrimitive(ofstream &out) const;

public:
    enum {
        kEnableGraphicalOutput = BIT(16)
    };

    MParContainer(const char *name="", const char *title="") : fName(name), fTitle(title), fLog(&gLog), fReadyToSave(kFALSE) {  }
    MParContainer(const TString &name, const TString &title) : fName(name), fTitle(title), fLog(&gLog), fReadyToSave(kFALSE) {  }
    MParContainer(const MParContainer &named);
    MParContainer& operator=(const MParContainer& rhs);
    virtual ~MParContainer() {}

    virtual TObject    *Clone(const char *newname="") const;
    virtual Int_t       Compare(const TObject *obj) const;
    virtual void        Copy(TObject &named)
#if ROOT_VERSION_CODE > ROOT_VERSION(3,04,01)
const
#endif
        ;
    virtual void        FillBuffer(char *&buffer);

    virtual const char   *GetDescriptor() const { return Form("%s [%s]", fName.Data(), ClassName()); }
    virtual const TString GetUniqueName() const;
    virtual const char   *GetName() const       { return fName.Data(); }
    virtual const char   *GetTitle() const      { return fTitle.Data(); }
    virtual ULong_t       Hash() const          { return fName.Hash(); }
    virtual Bool_t        IsSortable() const    { return kTRUE; }

    virtual void        SetName(const char *name); // *MENU*
    virtual void        SetObject(const char *name, const char *title);
    virtual void        SetTitle(const char *title=""); // *MENU*
    virtual void        ls(Option_t *option="") const;
    virtual void        Print(Option_t *option="") const;
    virtual Int_t       Sizeof() const;
    virtual void        SavePrimitive(ofstream &out, Option_t *o="");

    virtual MParContainer *New() const;
    virtual void   SetLogStream(MLog *lg) { fLog = lg; }
    virtual void   Reset() { }
    virtual Bool_t IsReadyToSave() const             { return fReadyToSave; }
    virtual void   SetReadyToSave(Bool_t flag=kTRUE) { fReadyToSave=flag; }
    virtual Bool_t IsSavedAsPrimitive() const        { return TestBit(kIsSavedAsPrimitive); }
    virtual void   EnableGraphicalOutput(Bool_t flag=kTRUE) { flag ? SetBit(kEnableGraphicalOutput) : ResetBit(kEnableGraphicalOutput);}
    virtual Bool_t IsGraphicalOutputEnabled() const  { return TestBit(kEnableGraphicalOutput); }

    TMethodCall *GetterMethod(const char *name) const;

    Bool_t WriteDataMember(ostream &out, const char *member, Double_t scale=1) const;
    Bool_t WriteDataMember(ostream &out, const TDataMember *member, Double_t scale=1) const;
    Bool_t WriteDataMember(ostream &out, const TList *list) const;

    virtual void AsciiRead(ifstream &fin);
    virtual Bool_t AsciiWrite(ostream &out) const;

    virtual void GetNames(TObjArray &arr) const;
    virtual void SetNames(TObjArray &arr);

    virtual Bool_t ReadEnv(const TEnv &env, TString prefix, Bool_t print=kFALSE);
    virtual Bool_t WriteEnv(TEnv &env, TString prefix, Bool_t print=kFALSE) const;

    Bool_t ReadEnv(const TEnv &env, Bool_t print=kFALSE) { return ReadEnv(env, "", print); }
    Bool_t WriteEnv(TEnv &env, Bool_t print=kFALSE) const { return WriteEnv(env, "", print); }

    Bool_t IsEnvDefined(const TEnv &env, TString prefix, TString postfix, Bool_t print) const;
    Bool_t IsEnvDefined(const TEnv &env, TString name, Bool_t print) const;

    Int_t       GetEnvValue(const TEnv &env, TString prefix, TString postfix, Int_t dflt) const;
    Double_t    GetEnvValue(const TEnv &env, TString prefix, TString postfix, Double_t dflt) const;
    const char *GetEnvValue(const TEnv &env, TString prefix, TString postfix, const char *dflt) const;

    ClassDef(MParContainer, 0)  //The basis for all parameter containers
};

/*
class MParContainer : public TNamed
{
protected:
    MLog    *fLog;         //! The general log facility for this object, initialized with the global object

private:
    Bool_t   fReadyToSave; //! should be set to true if the contents of the container is changed somehow

public:
    MParContainer(const char *name="", const char *title="") : TNamed(name, title), fLog(&gLog), fReadyToSave(kFALSE) { }
    MParContainer(const TString &name, const TString &title) : TNamed(name, title), fLog(&gLog), fReadyToSave(kFALSE) { }

    void SetLogStream(MLog *lg)                      { fLog = lg; }

    virtual void   Reset()                           {  }

    virtual Bool_t IsReadyToSave()                   { return fReadyToSave; }
    virtual void   SetReadyToSave(Bool_t flag=kTRUE) { fReadyToSave=flag; }

    virtual void AsciiRead(ifstream &fin);
    virtual void AsciiWrite(ofstream &fout) const;

    ClassDef(MParContainer, 0)  //The basis for all parameter containers
};
*/
#endif


