/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek 03/2003 <wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MNewImagePar
//
// Storage Container for new image parameters
//
/////////////////////////////////////////////////////////////////////////////
#include "MNewImagePar.h"

#include <fstream.h>
#include <TArrayF.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"
#include "MGeomPix.h"
#include "MCerPhotEvt.h"
#include "MCerPhotPix.h"
#include "MSrcPosCam.h"

ClassImp(MNewImagePar);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MNewImagePar::MNewImagePar(const char *name, const char *title)
{
    fName  = name  ? name  : "MNewImagePar";
    fTitle = title ? title : "New image parameters";
}

// --------------------------------------------------------------------------
//
void MNewImagePar::Reset()
{
    fLeakage1 = 0;
    fLeakage2 = 0;
}

// --------------------------------------------------------------------------
//
//  Calculation of new image parameters
//
//
Bool_t MNewImagePar::Calc(const MGeomCam &geom, const MCerPhotEvt &evt,
                          const MHillas *hillas)
{
    //fHillas = (MHillas*)hillas;

    const UInt_t npixevt = evt.GetNumPixels();

    Double_t edgepix1 = 0.0;
    Double_t edgepix2 = 0.0;

    for (UInt_t i=0; i<npixevt; i++)
    {
        const MCerPhotPix &pix = evt[i];
        if (!pix.IsPixelUsed())
            continue;

        const MGeomPix &gpix = geom[pix.GetPixId()];

        Double_t nphot = pix.GetNumPhotons();                        

        // count photons in outer rings of camera
        if (gpix.IsInOutermostRing())
           edgepix1 += nphot;
        if (gpix.IsInOuterRing())
           edgepix2 += nphot;
    }

    fLeakage1 = edgepix1 / hillas->GetSize();
    fLeakage2 = edgepix2 / hillas->GetSize();

    SetReadyToSave();

    return kTRUE;
} 

// --------------------------------------------------------------------------
//
void MNewImagePar::Print(Option_t *) const
{
    *fLog << all;
    *fLog << "New Image Parameters (" << GetName() << ")" << endl;
    *fLog << " - Leakage1            = " << fLeakage1     << endl;
    *fLog << " - Leakage2            = " << fLeakage2     << endl;
}
