/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek  02/2003 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MSelFinal                                                              //
//                                                                         //
//  This is a task to evaluate the Final Cuts                              //
//  (these cuts define the final sample of gammas;                         //
//   relevant for the calculation of the effective collection areas)       //
//                                                                         //
//  to be called after the calculation of the hadroness                    //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "math.h"
#include "MSelFinal.h"

#include "MParList.h"

#include "MHillas.h"
#include "MHillasExt.h"
#include "MHillasSrc.h"
#include "MCerPhotEvt.h"
#include "MMcEvt.hxx"
#include "MGeomCam.h"
#include "MGeomPix.h"
#include "MHadronness.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MSelFinal);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MSelFinal::MSelFinal(const char *HilName, const char *HilSrcName,
                     const char *name, const char *title)
{
    fName  = name  ? name  : "MSelFinal";
    fTitle = title ? title : "Task to evaluate the Final Cuts";

    fHilName    = HilName;
    fHilSrcName = HilSrcName;

    // default values of cuts
    fHadronnessMax =   1.0;
    fAlphaMax      = 100.0; //degrees
}

// --------------------------------------------------------------------------
//
// 
// 
// 
//
Bool_t MSelFinal::PreProcess(MParList *pList)
{
    fHil    = (MHillasExt*)pList->FindObject(fHilName, "MHillasExt");
    if (!fHil)
    {
      *fLog << dbginf << "MHillasExt object " << fHilName << " not found... aborting." << endl;
      return kFALSE;
    }

    fHilSrc = (MHillasSrc*)pList->FindObject(fHilSrcName, "MHillasSrc");
    if (!fHilSrc)
    {
      *fLog << dbginf << "MHillasSrc object " << fHilSrcName << " not found... aborting." << endl;
      return kFALSE;
    }


    fHadronness = (MHadronness*)pList->FindObject("MHadronness");
    if (!fHadronness)
    {
      *fLog << dbginf << "MHadronness not found... aborting." << endl;
      return kFALSE;
    }


    fMcEvt = (MMcEvt*)pList->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << dbginf << "MMcEvt not found... aborting." << endl;
        return kFALSE;
    }

    memset(fCut, 0, sizeof(fCut));

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Evaluate final cuts
// 
// if cuts are fulfilled      : return 0
// if they are not fullfilled : skip the remaining tasks for this event
//
Bool_t MSelFinal::Process()
{
  //*fLog << "Entry MSelFinal; fHilSrc = " << fHilSrc << endl;

    

    Int_t rc = 0;

    Double_t modalpha = fabs( fHilSrc->GetAlpha() ); 

    Double_t h = fHadronness->GetHadronness();

    if ( h>fHadronnessMax )
    {
      //*fLog << "MSelFinal::Process; h, alpha = " << h << ",  " 
      //      << fHilSrc->GetAlpha() << endl;
      rc = 1;
    }    

    else if ( modalpha > fAlphaMax )
    {
      //*fLog << "MSelFinal::Process; h, alpha = " << h << ",  " 
      //      << fHilSrc->GetAlpha() << endl;
      rc = 2;
    }    

    fCut[rc]++;

    return rc==0 ? kTRUE : kCONTINUE;
}

// --------------------------------------------------------------------------
//
//  Prints some statistics about the Final selections.
//
Bool_t MSelFinal::PostProcess()
{
    if (GetNumExecutions()==0)
        return kTRUE;

    *fLog << inf << endl;
    *fLog << GetDescriptor() << " execution statistics:" << endl;
    *fLog << dec << setfill(' ');
    *fLog << " " << setw(7) << fCut[1] << " (" << setw(3) 
          << (int)(fCut[1]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: hadroness > "<< fHadronnessMax << endl;

    *fLog << " " << setw(7) << fCut[2] << " (" << setw(3) 
          << (int)(fCut[2]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: |ALPHA| > " << fAlphaMax
          << " [degrees]" << endl;

    *fLog << " " << fCut[0] << " (" 
          << (int)(fCut[0]*100/GetNumExecutions()) 
          << "%) Evts survived Final selections!" << endl;
    *fLog << endl;

    return kTRUE;
}



