
/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek  02/2003 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MSelStandard                                                           //
//                                                                         //
//  This is a task to evaluate the Standard Cuts                           //
//                                                                         //
//  to be called after the calculation of the image parameters             //
//               before the g/h separation                                 //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MSelStandard.h"

#include "MParList.h"

#include "MHillas.h"
#include "MHillasExt.h"
#include "MHillasSrc.h"
#include "MCerPhotEvt.h"
#include "MMcEvt.hxx"
#include "MGeomCam.h"
#include "MGeomPix.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MSelStandard);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MSelStandard::MSelStandard(const char *HilName, const char *HilSrcName,
                           const char *name, const char *title)
{
    fName  = name  ? name  : "MSelStandard";
    fTitle = title ? title : "Task to evaluate the Standard Cuts";

    fHilName    = HilName;
    fHilSrcName = HilSrcName;

    // default values of cuts
    fUsedPixelsMax =   92;
    fCorePixelsMin =    4;
    fSizeMin       =   60;
    fDistMin       =  0.4;
    fDistMax       = 1.05;
    fLengthMin     =  0.0;
    fWidthMin      =  0.0;
}

// --------------------------------------------------------------------------
//
// 
// 
// 
//
Bool_t MSelStandard::PreProcess(MParList *pList)
{
    fHil    = (MHillasExt*)pList->FindObject(fHilName, "MHillasExt");
    if (!fHil)
    {
      *fLog << dbginf << "MHillasExt object " << fHilName << " not found... aborting." << endl;
      return kFALSE;
    }

    fHilSrc = (MHillasSrc*)pList->FindObject(fHilSrcName, "MHillasSrc");
    if (!fHilSrc)
    {
      *fLog << dbginf << "MHillasSrc object " << fHilSrcName << " not found... aborting." << endl;
      return kFALSE;
    }


    fMcEvt = (MMcEvt*)pList->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << dbginf << "MMcEvt not found... aborting." << endl;
        return kFALSE;
    }

    fEvt = (MCerPhotEvt*)pList->FindObject("MCerPhotEvt");
    if (!fEvt)
    {
        *fLog << dbginf << "MCerPhotEvt not found... aborting." << endl;
        return kFALSE;
    }


    fCam = (MGeomCam*)pList->FindObject("MGeomCam");
    if (!fCam)
    {
        *fLog << dbginf << "MGeomCam (Camera Geometry) missing in Parameter List... aborting." << endl;
        return kFALSE;
    }
    fMm2Deg = fCam->GetConvMm2Deg();

    //*fLog << "fMm2Deg = " << fMm2Deg << endl;

    memset(fCut, 0, sizeof(fCut));

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Evaluate standard cuts
// 
// if cuts are fulfilled      : return 0
// if they are not fullfilled : skip the remaining tasks for this event
// 
//
Bool_t MSelStandard::Process()
{
    Int_t rc = 0;

    Double_t length       = fHil->GetLength() * fMm2Deg;
    Double_t width        = fHil->GetWidth()  * fMm2Deg;
    Double_t dist         = fHilSrc->GetDist()* fMm2Deg;
    //Double_t delta        = fHil->GetDelta()  * kRad2Deg;
    Double_t size         = fHil->GetSize();
    Int_t numusedpixels   = fHil->GetNumUsedPixels();
    Int_t numcorepixels   = fHil->GetNumCorePixels();

    if ( numusedpixels >= fUsedPixelsMax  ||  numcorepixels <= fCorePixelsMin )
    {
      rc = 1;
    }    

    else if ( size <= fSizeMin )
    {
      rc = 2;
    }    

    else if ( dist< fDistMin   ||  dist > fDistMax )
    {
      rc = 3;
    }    

    else if ( length <= fLengthMin   ||  width <= fWidthMin )
    {
      rc = 4;
    }    


    fCut[rc]++;

    return rc==0 ? kTRUE : kCONTINUE;
}

// --------------------------------------------------------------------------
//
//  Prints some statistics about the Standard selections.
//
Bool_t MSelStandard::PostProcess()
{
    if (GetNumExecutions()==0)
        return kTRUE;

    *fLog << inf << endl;
    *fLog << GetDescriptor() << " execution statistics:" << endl;
    *fLog << dec << setfill(' ');
    *fLog << " " << setw(7) << fCut[1] << " (" << setw(3) 
          << (int)(fCut[1]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: Used pixels >= " << fUsedPixelsMax 
          << " or Core pixels <= " << fCorePixelsMin << endl;

    *fLog << " " << setw(7) << fCut[2] << " (" << setw(3) 
          << (int)(fCut[2]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: SIZE <= " << fSizeMin << endl;

    *fLog << " " << setw(7) << fCut[3] << " (" << setw(3) 
          << (int)(fCut[3]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: DIST < " << fDistMin 
          << " or DIST > " << fDistMax << endl;

    *fLog << " " << setw(7) << fCut[4] << " (" << setw(3) 
          << (int)(fCut[4]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: LENGTH <= " << fLengthMin 
          << " or WIDTH <= " << fWidthMin << endl;

    *fLog << " " << fCut[0] << " (" 
          << (int)(fCut[0]*100/GetNumExecutions()) 
          << "%) Evts survived Standard selections!" << endl;
    *fLog << endl;

    return kTRUE;
}












