/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek  04/2003 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MFCT1SelFinal                                                          //
//                                                                         //
//  This is a class to evaluate the Final Cuts                             //
//  (these cuts define the final sample of gammas;                         //
//   relevant for the calculation of the effective collection areas)       //
//                                                                         //
//  to be called after the calculation of the hadroness                    //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "math.h"
#include "MFCT1SelFinal.h"

#include "MParList.h"

#include "MHillas.h"
#include "MHillasExt.h"
#include "MHillasSrc.h"
#include "MCerPhotEvt.h"
#include "MMcEvt.hxx"
#include "MGeomCam.h"
#include "MGeomPix.h"
#include "MHadronness.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MFCT1SelFinal);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MFCT1SelFinal::MFCT1SelFinal(const char *HilName, const char *HilSrcName,
                     const char *name, const char *title)
{
    fName  = name  ? name  : "MFCT1SelFinal";
    fTitle = title ? title : "Class to evaluate the Final Cuts";

    fHilName        = HilName;
    fHilSrcName     = HilSrcName;
    fHadronnessName = "MHadronness";

    // default values of cuts
    SetCuts(1.0, 100.0);
}

// --------------------------------------------------------------------------
//
// Set cut values
// 
//
void MFCT1SelFinal::SetCuts(Float_t hadmax, Float_t alphamax) 
{ 
  fHadronnessMax = hadmax; 
  fAlphaMax      = alphamax;  

  *fLog << inf << "MFCT1SelFinal cut values : fHadronnessMax, fAlphaMax = "
        << fHadronnessMax << ",  " << fAlphaMax << endl;
}

// --------------------------------------------------------------------------
//
// Set the pointers
// 
//
Bool_t MFCT1SelFinal::PreProcess(MParList *pList)
{
    fHil    = (MHillasExt*)pList->FindObject(fHilName, "MHillasExt");
    if (!fHil)
    {
      *fLog << dbginf << "MHillasExt object " << fHilName << " not found... aborting." << endl;
      return kFALSE;
    }

    fHilSrc = (MHillasSrc*)pList->FindObject(fHilSrcName, "MHillasSrc");
    if (!fHilSrc)
    {
      *fLog << dbginf << "MHillasSrc object " << fHilSrcName << " not found... aborting." << endl;
      return kFALSE;
    }

    fHadronness = (MHadronness*)pList->FindObject(fHadronnessName, "MHadronness");
    if (!fHadronness)
    {
      *fLog << dbginf << "MHadronness not found... aborting." << endl;
      return kFALSE;
    }

    fMcEvt = (MMcEvt*)pList->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << dbginf << "MMcEvt not found... aborting." << endl;
        return kFALSE;
    }

    memset(fCut, 0, sizeof(fCut));

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Evaluate final cuts
// 
// if cuts are fulfilled set fResult = kTRUE
//
Bool_t MFCT1SelFinal::Process()
{
    Int_t rc = 0;
    fResult = kFALSE;

    Double_t modalpha = fabs( fHilSrc->GetAlpha() ); 

    Double_t h = fHadronness->GetHadronness();

    if ( h>fHadronnessMax )
    {
      rc = 1;
      fResult = kTRUE;
    }    

    else if ( modalpha > fAlphaMax )
    {
      rc = 2;
      fResult = kTRUE;
    }    

    fCut[rc]++;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
//  Prints some statistics about the Final selections.
//
Bool_t MFCT1SelFinal::PostProcess()
{
    if (GetNumExecutions()==0)
        return kTRUE;

    *fLog << inf << endl;
    *fLog << GetDescriptor() << " execution statistics:" << endl;
    *fLog << dec << setfill(' ');
    *fLog << " " << setw(7) << fCut[1] << " (" << setw(3) 
          << (int)(fCut[1]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: hadronness > "<< fHadronnessMax 
          << " (hadronness from '" << fHadronnessName << "')" << endl;

    *fLog << " " << setw(7) << fCut[2] << " (" << setw(3) 
          << (int)(fCut[2]*100/GetNumExecutions()) 
          << "%) Evts skipped due to: |ALPHA| > " << fAlphaMax
          << " [degrees]" << endl;

    *fLog << " " << fCut[0] << " (" 
          << (int)(fCut[0]*100/GetNumExecutions()) 
          << "%) Evts survived Final selections!" << endl;
    *fLog << endl;

    return kTRUE;
}
