#include <vector>
#include <iostream>

#include <TH1.h>
#include <TMarker.h>
#include <TArrow.h>
#include <TCanvas.h>
#include <TRandom.h>

#include "MFresnelLens.h"

/*****************************************************************

 fresnellens_traceray.C - 2D Example of ray-tracing the fresnel lens

 The macro has to be compilesd to work

 To run the macro from the command line (assuming you are in a directory
 Mars/build where you have built your Mars environment) you have to do

    root ../hawc/fresnellens_traceray.C++

 or from within root

    [0] .x ../hawc/fresnellens_traceray.C++

******************************************************************/
void fresnellens_traceray()
{
    // ------------------- setup lens -----------------------

    const double D = 54.92;  // [cm] Diameter of the refractive surface
    const double F = 50.21;  // [cm] Nominal focal length of the lens
    const double Z = F+H/2;  // [cm] camera position w.r.t. lens exit (see also MGeomCamFAMOUS!)

    const double R = D/2;    // [cm] radius of lens
    const double w = 0.01;   // [cm] Width of a single groove
    const double H = 0.25;   // [cm] Width of a single groove
    //const double w = 1;        // [cm] Width of a single groove
    //const double H = 2.5;    // [cm] Thickness of lens

    const double lambda = 546; // [nm] Wavelength for lens surface definition

    double angle = 0;  // Check ix=0 at angle=-55 and N=10

    double Z0 = 3;

    const double n0 = MFresnelLens::RefractiveIndex(lambda);

    MFresnelLens lens;
    lens.DefineLens(F, D, w, H, lambda);
    //lens.SetPSF(psf);
    //lens.DefineLens(F, D, w, H, lambda);
    //lens.EnableSlopeAbsorption();
    //lens.EnableDraftAbsorption();
    //lens.DisableBottomReflection();
    //lens.DisableMultiEntry();
    //lens.DisableFresnelReflection();
    lens.ReadTransmission("resmc/hawcseye/transmission-pmma-3mm.txt", 0.3, true);

    // ------------------ Setup display ---------------------

    TH1C h("", "", 10000, -R*1.1, R*1.1);
    h.SetStats(kFALSE);
    h.SetMaximum( 3);
    h.SetMinimum(-1.1*F);
    h.DrawCopy();

    TLine line;
    line.SetLineWidth(2);

    for (unsigned int i=0; i<lens.GetNumGrooves(); i++)
    {
        const MFresnelLens::Groove &g = lens[i];

        line.SetLineColor(kBlack);
        line.DrawLine( w*i, 0,  g.r, g.slope.h);
        line.DrawLine(-w*i, 0, -g.r, g.slope.h);;

        line.SetLineColor(kRed);
        line.DrawLine( g.r, g.slope.h,  w*i+w, 0);
        line.DrawLine(-g.r, g.slope.h, -w*i-w, 0);
    }

    line.SetLineColor(kBlack);
    line.SetLineWidth(1);
    line.DrawLine(-30, -H,   30, -H);
    line.DrawLine(-30, -Z, 30, -Z);
    line.DrawLine( R,  -Z,  R,  0);
    line.DrawLine(-R,  -Z, -R,  0);
    line.SetLineWidth(3);
    line.DrawLine(-4.5*1.5, -Z, 4.5*1.5, -Z);
    line.SetLineWidth(1);

    // ------------------------------------------------------

    TArrow arrow;
    TMarker marker;

    marker.SetMarkerColor(kBlue);
    marker.SetMarkerStyle(kFullDotMedium);

    arrow.SetArrowSize(0.01);
    arrow.SetLineColor(kBlue);

    double vc = 1./(TMath::C()*100/1e9); // cm/ns

    for (int i=0; i<100; i++)
    {
        double X = gRandom->Uniform(-R, R);

        TVector3 pos(X, 0,  0);
        TVector3 dir(0, 0, -1);

        dir.SetMagThetaPhi(1, (180-angle)*TMath::DegToRad(), 0);

        MQuaternion p(pos, 0);
        MQuaternion u(dir, vc);

        // Photon must be at the lens surface before it can enter the lens
        p.PropagateZ(u, Z0);

        vector<MQuaternion> vec;
        vec.push_back(p);

        const int cnt = lens.TraceRay(vec, p, u, lambda, true);

        // Particle sucdessfully reached the focal plane
        if (cnt>=0)
        {
            p.PropagateZ(u, -Z);
            vec.push_back(p);

            p.PropagateZ(u, u.Z()<0 ? -2.5*F : Z0);
            vec.push_back(p);
        }

        // Particle is upgoing and has hit no surface
        if (cnt==-1291 || cnt==-1391)
        {
            p.PropagateZ(u, 3);
            vec.push_back(p);
        }

        arrow.SetLineColor(kBlue);
        for (unsigned int j=0; j<vec.size()-1; j++)
        {
            arrow.DrawArrow(vec[j].X(), vec[j].Z(), vec[j+1].X(), vec[j+1].Z());
            marker.DrawMarker(vec[j+1].X(), vec[j+1].Z());
        }
    }
}
