/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek  03/2003   <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

///////////////////////////////////////////////////////////////////////
//
// MHNewImagePar
//
// This class contains histograms for every Hillas parameter
//
///////////////////////////////////////////////////////////////////////
#include "MHNewImagePar.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"

#include "MParList.h"

#include "MHillas.h"
#include "MNewImagePar.h"

ClassImp(MHNewImagePar);

// --------------------------------------------------------------------------
//
// Setup histograms 
//
MHNewImagePar::MHNewImagePar(const char *name, const char *title)
    : fUseMmScale(kTRUE)
{
    fName  = name  ? name  : "MHNewImagePar";
    fTitle = title ? title : "Container for histograms of new image parameters";

    fLeakage1 = new TH1F("Leakage1", "Leakage1", 100, 0.0, 1.0);
    fLeakage1->SetDirectory(NULL);
    fLeakage1->SetXTitle("Leakage1");
    fLeakage1->SetYTitle("Counts");

    fLeakage2 = new TH1F("Leakage2", "Leakage2", 100, 0.0, 1.0);
    fLeakage2->SetDirectory(NULL);
    fLeakage2->SetXTitle("Leakage2");
    fLeakage2->SetYTitle("Counts");
}

// --------------------------------------------------------------------------
//
// Delete the four histograms
//
MHNewImagePar::~MHNewImagePar()
{
    delete fLeakage1;
    delete fLeakage2;
}

// --------------------------------------------------------------------------
//
// Setup the Binning for the histograms automatically if the correct
// instances of MBinning (with the names 'BinningAlpha' and 'BinningDist')
// are found in the parameter list
// Use this function if you want to set the conversion factor which
// is used to convert the mm-scale in the camera plain into the deg-scale
// used for histogram presentations. The conversion factor is part of
// the camera geometry. Please create a corresponding MGeomCam container.
//
Bool_t MHNewImagePar::SetupFill(const MParList *plist)
{
    const MGeomCam *geom = (MGeomCam*)plist->FindObject("MGeomCam");
    if (!geom)
        *fLog << warn << dbginf << "No Camera Geometry available. Using mm-scale for histograms." << endl;
    else
    {
        fMm2Deg = geom->GetConvMm2Deg();
        SetMmScale(kFALSE);
    }

    //ApplyBinning(*plist, "Alpha",    fAlpha);
    //ApplyBinning(*plist, "Dist",     fDist);
    //ApplyBinning(*plist, "HeadTail", fHeadTail);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the histograms with data from a MNewImagePar container.
//
Bool_t MHNewImagePar::Fill(const MParContainer *par)
{
    const MNewImagePar &h = *(MNewImagePar*)par;

    fLeakage1->Fill(h.GetLeakage1());
    fLeakage2->Fill(h.GetLeakage2());

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Use this function to setup your own conversion factor between degrees
// and millimeters. The conversion factor should be the one calculated in
// MGeomCam. Use this function with Caution: You could create wrong values
// by setting up your own scale factor.
//
void MHNewImagePar::SetMm2Deg(Float_t mmdeg)
{
    if (mmdeg<=0)
    {
        *fLog << warn << dbginf << "Warning - Conversion factor <= 0 - nonsense. Ignored." << endl;
        return;
    }

    if (fMm2Deg>0)
        *fLog << warn << dbginf << "Warning - Conversion factor already set. Overwriting" << endl;

    fMm2Deg = mmdeg;
}

// --------------------------------------------------------------------------
//
// With this function you can convert the histogram ('on the fly') between
// degrees and millimeters.
//
void MHNewImagePar::SetMmScale(Bool_t mmscale)
{
    if (fUseMmScale == mmscale)
        return;

    if (fMm2Deg<0)
    {
        *fLog << warn << GetDescriptor() << ": Warning - Sorry, no conversion factor for conversion available." << endl;
        return;
    }

    //const Double_t scale = mmscale ? 1./fMm2Deg : fMm2Deg;
    //MH::ScaleAxis(fDist,     scale);
    //MH::ScaleAxis(fHeadTail, scale);

    if (mmscale)
    {
      //    fDist->SetXTitle("Dist [mm]");
      //    fHeadTail->SetXTitle("Head-Tail [mm]");
    }
    else
    {
      //    fDist->SetXTitle("Dist [\\circ]");
      //    fHeadTail->SetXTitle("Head-Tail [\\circ]");
    }

    fUseMmScale = mmscale;
}

// --------------------------------------------------------------------------
//
// Draw clones of all two histograms. So that the object can be deleted
// and the histograms are still visible in the canvas.
// The cloned object are deleted together with the canvas if the canvas is
// destroyed. If you want to handle dostroying the canvas you can get a
// pointer to it from this function
//
TObject *MHNewImagePar::DrawClone(Option_t *opt) const
{
    return MH::DrawClone(opt, 300, 600);
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the two histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHNewImagePar::Draw(Option_t *)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this, 300, 600);
    pad->SetBorderMode(0);

    AppendPad("");

    pad->Divide(1, 2);

    pad->cd(1);
    pad->SetBorderMode(0);
    fLeakage1->Draw();

    pad->cd(2);
    pad->SetBorderMode(0);
    fLeakage2->Draw();

    pad->Modified();
    pad->Update();
}

TH1 *MHNewImagePar::GetHistByName(const TString name)
{
    if (name.Contains("Leakage1", TString::kIgnoreCase))
        return fLeakage1;

    if (name.Contains("Leakage2", TString::kIgnoreCase))
        return fLeakage2;

    return NULL;
}

















