/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz    1/2002 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Wolfgang Wittek 1/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHTimeDiffTheta                                                         //
//                                                                          //
//  calculates the 2D-histogram   time-difference vs. Theta                 //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHTimeDiffTheta.h"

#include <TCanvas.h>

#include "MTime.h"
#include "MMcEvt.hxx"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHTimeDiffTheta);


// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title of the histogram.
//
MHTimeDiffTheta::MHTimeDiffTheta(const char *name, const char *title)
    : fLastTime(0), fHist()
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHTimeDiffTheta";
    fTitle = title ? title : "2-D histogram in Theta and time difference";

    fHist.SetDirectory(NULL);

    fHist.SetXTitle("\\Delta t [s]");
    fHist.SetYTitle("\\Theta [\\circ]");
}

// --------------------------------------------------------------------------
//
// Set the binnings and prepare the filling of the histogram
//
Bool_t MHTimeDiffTheta::SetupFill(const MParList *plist)
{
   fTime = (MTime*)plist->FindObject("MTime");
   if (!fTime)
   {
       *fLog << err << dbginf << "MTime not found... aborting." << endl;
       return kFALSE;
   }

   fMcEvt = (MMcEvt*)plist->FindObject("MMcEvt");
   if (!fMcEvt)
   {
       *fLog << err << dbginf << "MMcEvt not found... aborting." << endl;
       return kFALSE;
   }

   const MBinning* binsdtime = (MBinning*)plist->FindObject("BinningTimeDiff");
   const MBinning* binstheta = (MBinning*)plist->FindObject("BinningTheta");
   if (!binstheta || !binsdtime)
   {
       *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
       return kFALSE;      
   }

   SetBinning(&fHist, binsdtime, binstheta);

   return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw the histogram
//
void MHTimeDiffTheta::Draw(Option_t *opt)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    TH1 *h;

    pad->Divide(2,2);

    pad->cd(1);
    gPad->SetLogy();
    h = fHist.ProjectionX("ProjX_Theta", -1, 9999, "E");
    h->SetTitle("Distribution of \\Delta t [s]");
    h->SetXTitle("\\Delta t [s]");
    h->SetYTitle("Counts");
    h->Draw(opt);
    h->SetBit(kCanDelete);;

    pad->cd(2);
    h = fHist.ProjectionY("ProjY_timediff", -1, 9999, "E");
    h->SetTitle("Distribution of \\Theta [\\circ]");
    h->SetXTitle("\\Theta [\\circ]");
    h->SetYTitle("Counts");
    h->Draw(opt);
    h->SetBit(kCanDelete);;

    pad->cd(3);
    fHist.Draw(opt);

    pad->Modified();
    pad->Update();
}

// --------------------------------------------------------------------------
//
//  Fill the histogram
//
Bool_t MHTimeDiffTheta::Fill(const MParContainer *par)
{
    const Double_t time = 200e-9*fTime->GetTimeLo() + fTime->GetTimeHi();

    fHist.Fill(time-fLastTime, fMcEvt->GetTelescopeTheta()*kRad2Deg);
    fLastTime = time;

    return kTRUE;
}

