/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek  03/2003   <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

///////////////////////////////////////////////////////////////////////
//
// MHNewImagePar
//
// This class contains histograms for every Hillas parameter
//
///////////////////////////////////////////////////////////////////////
#include "MHNewImagePar.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"

#include "MParList.h"

#include "MHillas.h"
#include "MNewImagePar.h"

ClassImp(MHNewImagePar);

// --------------------------------------------------------------------------
//
// Setup histograms 
//
MHNewImagePar::MHNewImagePar(const char *name, const char *title)
{
    fName  = name  ? name  : "MHNewImagePar";
    fTitle = title ? title : "Histograms of new image parameters";

    fLeakage1 = new TH1F("Leakage1", "Leakage_{1}", 100, 0.0, 1.0);
    fLeakage1->SetDirectory(NULL);
    fLeakage1->SetXTitle("Leakage");
    fLeakage1->SetYTitle("Counts");

    fLeakage2 = new TH1F("Leakage2", "Leakage_{2}", 100, 0.0, 1.0);
    fLeakage2->SetDirectory(NULL);
    fLeakage2->SetXTitle("Leakage");
    fLeakage2->SetYTitle("Counts");
    fLeakage2->SetLineColor(kBlue);
}

// --------------------------------------------------------------------------
//
// Delete the four histograms
//
MHNewImagePar::~MHNewImagePar()
{
    delete fLeakage1;
    delete fLeakage2;
}

// --------------------------------------------------------------------------
//
// Fill the histograms with data from a MNewImagePar container.
//
Bool_t MHNewImagePar::Fill(const MParContainer *par)
{
    const MNewImagePar &h = *(MNewImagePar*)par;

    fLeakage1->Fill(h.GetLeakage1());
    fLeakage2->Fill(h.GetLeakage2());

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the two histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHNewImagePar::Draw(Option_t *)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    MH::Draw(*fLeakage1, *fLeakage2, "Leakage1 and Leakage2");

    pad->Modified();
    pad->Update();
}

TH1 *MHNewImagePar::GetHistByName(const TString name)
{
    if (name.Contains("Leakage1", TString::kIgnoreCase))
        return fLeakage1;

    if (name.Contains("Leakage2", TString::kIgnoreCase))
        return fLeakage2;

    return NULL;
}
