#!/bin/bash

# The data is expected to be found in /data/raw and data/callisto
DATA="/data"
CALLISTO=callisto
# output is written to data/star
STAR=star
# Directory to Mars environment (usually Mars/build)
MARS=~/Mars/build
# Relative path to macros
MACROS=../hawc

# Get all runs that can (and should) be calibrated
# Pipe the output to mysql and read the
# mysql output line-by-libe
echo \
"\
SELECT 
    NIGHT, RUNID
FROM 
    Calibration_new
ORDER BY
    NIGHT, RUNID\
"\
 |  mysql \
       --defaults-file=/home/tbretz/data/.password.cnf \
       --user=iceactwrite \
       --host=ihp-pc45.ethz.ch \
       --database=iceactdata \
       --skip-column-names \
       --batch --raw \
       --compress \
 | \
while IFS= read -r LINE
do
  # Get NIGHT and RUNID of all files
  NIGHT=`echo "$LINE" | awk -F"\t" '{print $1}'`
  RUNID=`echo "$LINE" | awk -F"\t" '{print $2}'`

  # Split into year, month, day
  YEAR=`echo ${NIGHT}  | cut -c1-4`
  MONTH=`echo ${NIGHT} | cut -c5-6`
  DAY=`echo ${NIGHT}   | cut -c7-8`

  # Formatting of the file paths and names
  DATPATH="${YEAR}/${MONTH}/${DAY}"

  PREFIX=`printf ${NIGHT}_%03d ${RUNID}`

  DATNAME=`printf ${PREFIX}_Y.root ${RUNID}`

  OUT=${DATA}/${STAR}/${DATPATH}
  IN=${DATA}/${CALLISTO}/${DATPATH}

  echo ""
  echo ${DATNAME}
  echo ${IN} " => " ${OUT}

  # Check if not yet successfully processed but sucessfully calibrated
  if [[ -f "${IN}/.${PREFIX}.success" && ! -f "${IN}/.${PREFIX}.running" && ! -f "${OUT}/.${PREFIX}.success" ]]
  then

     cd ${MARS}

     mkdir -p ${OUT}

     # Flag that a process is running
     rm -f ${OUT}/.${PREFIX}.done
     rm -f ${OUT}/.${PREFIX}.success

     touch ${OUT}/.${PREFIX}.running

     # Execute image parameter calculation and write log-file
     root -b -q -l ${MACROS}/star.C'("'${IN}/${DATNAME}'","'${OUT}'")' \
        2>&1 | tee ${OUT}/${PREFIX}.log

     # Remember exit status of callisto
     RC=${PIPESTATUS[0]}

     echo RC=${RC} | tee -a ${OUT}/${PREFIX}.log
     echo ${RC} > ${OUT}/.${PREFIX}.done

     # Flag that processing is finished
     rm -f ${OUT}/.${PREFIX}.running

     # If processing was successfull write coresponding flag     
     if [ "${RC}" == "0" ]
     then
         touch ${OUT}/.${PREFIX}.success
     fi

     cd -

  else
     echo Skipped.
  fi

done
