#!/bin/bash
#
# Runs callisto macro on all files in directory DIR for a given telescope TEL
# and night NIGHT and writes output to DIR/../callisto
#
# Example:
# 	bash run-callisto2.sh cred-file 1 /data/HE01/raw 20201111

set -o errexit
set -o errtrace
set -o nounset
set -o pipefail

function ErrExit()
{
	echo "ERROR: Line `caller`: ${BASH_COMMAND}" >&2
	exit 1
}

function StrgCExit()
{
	echo " "
	echo "$0 was forcefully terminated" >&2
	exit 1
}

trap ErrExit ERR
trap StrgCExit INT

readonly CALLISTO=callisto
# The .success file in /data/star will be resetted
readonly STAR=star
# Directory to Mars environment (usually Mars/build)
readonly MARS="/home/hawc/Desktop/Mars/"
# Absolut path to macros
readonly MACROS="/home/hawc/Desktop/Mars-src/hawc"
readonly PROGRAM=$0
REPLACE=false

usage()
{
	echo "usage: $PROGRAM [-hr] [Credentials Telescope Dir Night]" 
	echo "	-h 		display help"
	echo "	-r 		replace data"
	exit 1;
}

# Check for flags
while getopts 'hr' flag
do
	case "${flag}" in
		h) usage ;;
		r) REPLACE=true ;;
		*) usage ;;
	esac
done
shift $(($OPTIND-1))

# Check if Mars exists
if [ ! -d "${MARS}" ]
then
	echo "Mars does not exists at ${MARS}. Please change in \\
		script ${PROGRAM}."
	exit 1
fi

# Check if MACROS exists
if [ ! -d "${MACROS}" ]
then
	echo "Macros directorey does not exists at ${MACROS}. Please change in \\
		script ${PROGRAM}."
	exit 1
fi

# Check if at least two arguments are provided
if [ -z "${1}" ] || [ -z "${2}" ] || [ -z "${3}" ] || [ -z "${4}" ]
then
	echo "Not enough arguments. Check -h for help!"
	exit 1
fi

# File containing the access credentials for the database
CRED="${1}"
# Specify the telescope for which to run the script.
TEL="${2}"
# The data is expected to be found in /data/raw and data/callisto
DATA="${3}"
NIGHT="${4}"

function UpdateStatus()
{
	local tel="${1}"
	local night="${2}"
	local runid="${3}"
	local stat="${4}"

	mysql --defaults-file=${CRED} \
		--compress \
		-e "UPDATE DataOnDisk \
		SET DataOnDisk.callisto = ${stat} \
		WHERE DataOnDisk.Telescope = ${tel} \
		AND DataOnDisk.NIGHT = ${night} \
		AND DataOnDisk.RUNID = ${runid};"
}

# Get all runs that can (and should) be calibrated
# Pipe the output to mysql and read the
# mysql output line-by-libe
echo \
"\
SELECT 
    	Calibration.NIGHT,
	Calibration.RUNID,
	Calibration.DrsNight,
	Calibration.DrsRunID
FROM 
	Calibration
INNER JOIN
	DataOnDisk
ON
	DataOnDisk.Telescope = Calibration.Telescope
AND
	DataOnDisk.NIGHT = Calibration.NIGHT
AND
	DataOnDisk.RUNID = Calibration.RUNID
WHERE
    	Calibration.Telescope = ${TEL}
AND	
	Calibration.NIGHT = ${NIGHT}
AND
	DataOnDisk.calibration = 0
ORDER BY
    	Calibration.NIGHT, Calibration.RUNID\
"\
|  mysql \
       --defaults-file=${CRED} \
       --skip-column-names \
       --batch --raw \
       --compress \
       | \
	while read -r -a LINE
	do
		# Extract night/runid for data file and calibration file
	       	DATNIGHT=${LINE[0]}
		DATRUNID=${LINE[1]}
		DRSNIGHT=${LINE[2]}
		DRSRUNID=${LINE[3]}

	       	# Formatting of the file paths and names
		DATPATH=${DATNIGHT:0:4}/${DATNIGHT:4:2}/${DATNIGHT:6:2}
		DRSPATH=${DRSNIGHT:0:4}/${DRSNIGHT:4:2}/${DRSNIGHT:6:2}

		PREFIX=`printf ${DATNIGHT}_%03d ${DATRUNID}`

		DATNAME=`printf ${DATNIGHT}_%03d.fits.fz ${DATRUNID}`
		DRSNAME=`printf ${DRSNIGHT}_%03d.drs.fits ${DRSRUNID}`

		OUT="${DATA}"/../${CALLISTO}/${DATPATH}

		echo DAT=${DATNAME} [${DATPATH}]
		echo DRS=${DRSNAME} [${DRSPATH}]


		if [ ${REPLACE} = true ]
		then
			echo "deleting ${OUT}/.${PREFIX}.succsess !!!"
		       	rm -f "${OUT}"/.${PREFIX}.success
		fi

		# Check if not yet successfully processed
		# Removing this file can be used to trigger a re-processing
		# the next time this script is executed
		if [ ! -f "${OUT}/.${PREFIX}.success" ]
		then

	       		cd ${MARS}

			mkdir -p "${OUT}"

			# Trigger reprocessing of the process-fils in the star directory
			rm -f "${DATA}/${STAR}/${DATPATH}/.${PREFIX}.*"

			# Flag that a process is running
			rm -f "${OUT}"/.${PREFIX}.done
			rm -f "${OUT}"/.${PREFIX}.success

			touch "${OUT}"/.${PREFIX}.running

			echo ${DATA}

			RC=0
			# Execute the calibration and write output to log-file
			root -b -q -l ${MACROS}/callisto.C'("'"${DATA}"/${DATPATH}/${DATNAME}'","'"${DATA}"/${DRSPATH}/${DRSNAME}'", 0 ,"'"${OUT}"'")' \
			       2>&1 | tee "${OUT}"/${PREFIX}.log || RC=1

		       	# Remember exit status of callisto
		       	echo RC=${RC} | tee -a "${OUT}"/${PREFIX}.log
			echo ${RC} > "${OUT}"/.${PREFIX}.done
		       	# Processing is finished
			rm -f "${OUT}"/.${PREFIX}.running

		       	# If processing was successfull write corresponding flag
			if [ "${RC}" == "0" ]
			then
				touch "${OUT}"/.${PREFIX}.success
		       	fi

			cd -

			echo "-----End of first iteration-----"
		else
			echo Skipped.
		fi

	done
