/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Harald Kornmayer 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

///////////////////////////////////////////////////////////////////////
//
// MGeomCam
//
// This is the base class of different camera geometries. It creates
// a pixel object for a given number of pixels and defines the
// interface of how to acccess the geometry information.
//
///////////////////////////////////////////////////////////////////////
#include "MGeomCam.h"

#include <TClass.h>     // IsA()->New()

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomPix.h"

ClassImp(MGeomCam);

using namespace std;

// --------------------------------------------------------------------------
//
// Initializes a Camera Geometry with npix pixels. All pixels
// are deleted when the corresponding array is deleted.
//
MGeomCam::MGeomCam(UInt_t npix, Float_t dist, const char *name, const char *title)
    : fNumPixels(npix), fCamDist(dist), fConvMm2Deg(kRad2Deg/(dist*1000))
{
    fName  = name  ? name  : "MGeomCam";
    fTitle = title ? title : "Storage container for  a camera geometry";

    fPixels = new TObjArray(npix);

    //
    // make sure that the destructor delete all contained objects
    //
    fPixels->SetOwner();

    for (UInt_t i=0; i<npix; i++)
        (*fPixels)[i] = new MGeomPix;

    SetReadyToSave();
}

// --------------------------------------------------------------------------
//
//  Destructor delete the pixel array with all pixels
//
MGeomCam::~MGeomCam()
{
    delete fPixels;
}

// --------------------------------------------------------------------------
//
//  Set the kIsOuterRing flag for all pixels which have a outermost pixel
//  as Next Neighbor and don't have the kIsOutermostRing flag itself.
//
void MGeomCam::InitOuterRing()
{
    for (UInt_t i=0; i<fNumPixels; i++)
        (*this)[i].CheckOuterRing(*this);
}

// --------------------------------------------------------------------------
//
// Calculate the maximum radius of the camera. This is ment for GUI layout.
//
void MGeomCam::CalcMaxRadius()
{
    fMaxRadius = 0;

    for (UInt_t i=0; i<fNumPixels; i++)
    {
        const MGeomPix &pix = (*this)[i];

        const Float_t x = pix.GetX();
        const Float_t y = pix.GetY();
        const Float_t d = pix.GetD();

        const Float_t maxr = sqrt(x*x+y*y) + d;

        if (maxr>fMaxRadius)
            fMaxRadius = maxr;
    }
}

// --------------------------------------------------------------------------
//
//  returns the ratio of the area of the given pixel to the pixel with
//  the id 0 to scale variables with the pixel size.
//
Float_t MGeomCam::GetPixRatio(UInt_t i) const
{
    return i<fNumPixels ? (*this)[0].GetA()/(*this)[i].GetA() : 0;
}

// --------------------------------------------------------------------------
//
//  Prints the Geometry information of all pixels in the camera
//
void MGeomCam::Print(Option_t *) const
{
    //
    //   Print Information about the Geometry of the camera
    //
    *fLog << all << " Number of Pixels (" << GetTitle() << "): " << fNumPixels << endl;

    for (UInt_t i=0; i<fNumPixels; i++)
    {
        *fLog << " Pixel: " << i << "  ";
        (*this)[i].Print();
    }
} 

TObject *MGeomCam::Clone(const char *newname) const
{
    return (TObject*)IsA()->New();
}
