/* $ZEL: sis1100_ioctl32_linux.c,v 1.2 2005/11/18 21:12:44 wuestner Exp $ */

/*
 * Copyright (c) 2005
 * 	Peter Wuestner.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#if !defined(__linux__)
#error Invalid Operating System
#endif

#include <linux/config.h>

#ifdef CONFIG_COMPAT

#include <linux/compat.h>
#include <linux/ioctl.h>
#include <linux/ioctl32.h>
#include "sis1100_sc.h"
#include "sis1100_ioctl32_linux.h"

static int
ioctl32_ident(struct sis1100_softc* sc, struct sis1100_fdata* fd,
    struct sis1100_32_ident* d)
{
    d->local.hw_type=sc->local_ident&0xff;
    d->local.hw_version=(sc->local_ident>>8)&0xff;
    d->local.fw_type=(sc->local_ident>>16)&0xff;
    d->local.fw_version=(sc->local_ident>>24)&0xff;

    d->remote.hw_type=sc->remote_ident&0xff;
    d->remote.hw_version=(sc->remote_ident>>8)&0xff;
    d->remote.fw_type=(sc->remote_ident>>16)&0xff;
    d->remote.fw_version=(sc->remote_ident>>24)&0xff;

    d->remote_ok=sc->remote_hw!=sis1100_hw_invalid;
    d->remote_online=(sis1100readreg(sc, sr)&sr_synch)==sr_synch;
    return 0;
}

static int
ioctl32_setvmespace(struct sis1100_softc* sc, struct sis1100_fdata* fd,
    struct vmespace_32* d)
{
    if ((d->datasize!=1) && (d->datasize!=2) && (d->datasize!=4))
        return EINVAL;
    fd->vmespace_am=d->am;
    fd->vmespace_datasize=d->datasize;
    if (d->swap>=0) {
        sc->user_wants_swap=d->swap;
        sis1100_update_swapping(sc, "ioctl_setvmespace");
    }
    if (d->mindmalen>=0) {
        fd->mindmalen_r=d->mindmalen;
        fd->mindmalen_w=d->mindmalen;
    }
    return 0;
}

static int
ioctl32_vme_block_read(struct sis1100_softc* sc, struct sis1100_fdata* fd,
    struct sis1100_32_vme_block_req* d)
{
    int res;
    size_t num;

    if (sc->remote_hw==sis1100_hw_invalid)
        return ENXIO;

    res=sis1100_read_block(sc, fd, d->size, d->fifo, d->num, &num,
        1/*space*/, d->am, d->addr, compat_ptr(d->data), &d->error);
    d->num=num;

    return res;
}

static int
ioctl32_vme_block_write(struct sis1100_softc* sc, struct sis1100_fdata* fd,
    struct sis1100_32_vme_block_req* d)
{
    int res;
    size_t num;
 
    if (sc->remote_hw==sis1100_hw_invalid)
        return ENXIO;

    res=sis1100_write_block(sc, fd, d->size, d->fifo, d->num, &num,
        1/*space*/, d->am, d->addr, compat_ptr(d->data), &d->error);
    d->num=num;

    return res;
}

static int
ioctl32_pipe(struct sis1100_softc* sc, struct sis1100_fdata* fd,
    struct sis1100_32_pipe* d32)
{
    int res;
    struct sis1100_pipe d;

    if (sc->remote_hw==sis1100_hw_invalid)
        return ENXIO;

    d.num=d32->num;
    d.list=compat_ptr(d32->list);
    d.data=compat_ptr(d32->data);
    res=sis1100_read_pipe(sc, &d);
    d32->error=d.error;
    return res;
}

static int
ioctl32_write_pipe(struct sis1100_softc* sc, struct sis1100_fdata* fd,
    struct sis1100_32_writepipe* d)
{
    u_int32_t* list;
    int res;

    if (sc->remote_hw==sis1100_hw_invalid)
        return ENXIO;

    list=kmalloc(d->num*2*sizeof(u_int32_t), GFP_KERNEL);
    if (!list)
        return ENOMEM;

    if (copy_from_user(list, compat_ptr((u32)d->data),
            d->num*2*sizeof(u_int32_t))) {
	res=EFAULT;
        goto raus;
    }

    res=0;
    d->error=sis1100_write_pipe(sc, d->am, 1/*space*/, d->num, list);

raus:
    kfree(list);
    return res;
}

static int
ioctl32_dma_alloc(struct sis1100_softc* sc, struct sis1100_fdata* fd,
    struct sis1100_32_dma_alloc* d)
{
    return ENOTTY;
}

static int
ioctl32_dma_free(struct sis1100_softc* sc, struct sis1100_fdata* fd,
    struct sis1100_32_dma_alloc* d)
{
    /*return sis1100_dma_free(sc, fd, d);*/
    return ENOTTY;
}

static int
ioctl32_dsp_load(struct sis1100_softc* sc, struct sis1100_fdata* fd,
    struct sis1100_32_dsp_code* d32)
{
    struct sis1100_dsp_code d;
    int res;

    d.src=compat_ptr(d32->src);
    d.dst=d32->dst;
    d.size=d32->size;
    res=sis1100_dsp_load(sc, fd, &d);
    return res;
}

static int
ioctl32_ddma_map(struct sis1100_softc* sc, struct sis1100_fdata* fd,
    struct sis1100_32_ddma_map* d32)
{
    struct sis1100_ddma_map d;
    d.num=d32->num;
    d.addr=compat_ptr(d32->addr);
    d.size=d32->size;
    return sis1100_ddma_map(sc, fd, &d);
}

static int
ioctl32_read_eeprom(struct sis1100_softc* sc, struct sis1100_fdata* fd,
        struct sis1100_32_eeprom_req* d)
{
    return sis1100_read_eeprom(sc, d->num, d->addr, compat_ptr(d->data));
}

static int
ioctl32_write_eeprom(struct sis1100_softc* sc, struct sis1100_fdata* fd, 
        struct sis1100_32_eeprom_req* d)
{
    /*if (test_super(sc, fd)) return EPERM;*/
    return sis1100_write_eeprom(sc, d->num, d->addr, compat_ptr(d->data));
}

static int
_sis1100_ioctl32(struct sis1100_softc* sc, struct sis1100_fdata* fd,
    unsigned int cmd, void* data)
{
    int res=0;

    switch (cmd) {
    case SIS1100_32_SETVMESPACE:
        res=ioctl32_setvmespace(sc, fd, (struct vmespace_32*)data); break;
    case SIS3100_VME_BLOCK_READ:
        res=ioctl32_vme_block_read(sc, fd, (struct sis1100_32_vme_block_req*)data);
        break;
    case SIS3100_VME_BLOCK_WRITE:
        res=ioctl32_vme_block_write(sc, fd, (struct sis1100_32_vme_block_req*)data);
        break;
    case SIS1100_32_PIPE:
        res=ioctl32_pipe(sc, fd, (struct sis1100_32_pipe*)data); break;
    case SIS1100_32_IDENT:
        res=ioctl32_ident(sc, fd, (struct sis1100_32_ident*)data); break;
/*
    case SIS3100_VME_SUPER_BLOCK_READ:
        res=ioctl32_vme_super_block_read(sc, fd,
                (struct sis1100_32_vme_super_block_req*)data);
        break;
*/
    case SIS1100_32_WRITE_PIPE:
        res=ioctl32_write_pipe(sc, fd, (struct sis1100_32_writepipe*)data); break;

    case SIS1100_32_DMA_ALLOC:
        res=ioctl32_dma_alloc(sc, fd, (struct sis1100_32_dma_alloc*)data); break;
    case SIS1100_32_DMA_FREE:
        res=ioctl32_dma_free(sc, fd, (struct sis1100_32_dma_alloc*)data); break;

    case SIS1100_32_DSP_LOAD:
        res=ioctl32_dsp_load(sc, fd, (struct sis1100_32_dsp_code*)data); break;

    case SIS1100_32_DMA_MAP:
        res=ioctl32_ddma_map(sc, fd, (struct sis1100_32_ddma_map*)data); break;

    case SIS1100_32_READ_EEPROM:
        res=ioctl32_read_eeprom(sc, fd, (struct sis1100_32_eeprom_req*)data); break;
    case SIS1100_32_WRITE_EEPROM:
        res=ioctl32_write_eeprom(sc, fd, (struct sis1100_32_eeprom_req*)data); break;

    default:
#ifdef HAVE_COMPAT_IOCTL
        return _sis1100_ioctl(sc, fd, cmd, data);
#else
        pINFO(sc, "ioctl32: cmd=0x%x DIR=%d TYPE=%d NR=%d SIZE=%d", cmd,
            _IOC_DIR(cmd), _IOC_TYPE(cmd), _IOC_NR(cmd), _IOC_SIZE(cmd));
        res=ENOTTY; break;
#endif
    }
    return res;
}

union alles {
    struct vmespace_32                               vmespace;
    struct sis1100_32_vme_req                         vme_req;
    struct sis1100_32_vme_block_req             vme_block_req;
    struct sis1100_32_ctrl_reg                       ctrl_reg;
    struct sis1100_32_pipe                               pipe;
    struct sis1100_32_ident                             ident;
    struct sis1100_32_irq_ctl                         irq_ctl;
    struct sis1100_32_irq_get                         irq_get;
    struct sis1100_32_vme_super_block_req vme_super_block_req;
    struct sis1100_32_writepipe                     writepipe;
    struct sis1100_32_dma_alloc                     dma_alloc;
    struct sis1100_32_dsp_code                       dsp_code;
    struct sis1100_32_eeprom_req                   eeprom_req;
};

#define MAX_DATA (sizeof(union alles))

#ifdef HAVE_COMPAT_IOCTL
long
sis1100_ioctl32(struct file * file, unsigned int cmd, unsigned long arg)
#else
static int
sis1100_ioctl32(unsigned int fd_, unsigned int cmd,
    unsigned long arg, struct file * file)
#endif
{
    struct sis1100_softc* sc=SIS1100SC(file);
    struct sis1100_fdata* fd=SIS1100FD(file);
    u_int8_t data[MAX_DATA];
    int res;

    if ((cmd&IOC_INOUT) && (_IOC_SIZE(cmd)>MAX_DATA)) {
        pINFO(sc, "sis1100_ioctl32: cmd=0x%08x _IOC_SIZE(cmd)=%d",
            cmd, _IOC_SIZE(cmd));
        return -EINVAL;
    }

    if (cmd&IOC_IN) {
        if (copy_from_user(&data, (void *)arg, _IOC_SIZE(cmd)))
                return -EFAULT;
    }
    if ((res=_sis1100_ioctl32(sc, fd, cmd, &data)))
            return -res;

    if (cmd&IOC_OUT) {
        if (copy_to_user((void *)arg, &data, _IOC_SIZE(cmd)))
                return -EFAULT;
    }
    return 0;
}

#ifndef HAVE_COMPAT_IOCTL
static struct ioctl_trans sis1100_ioctl32_trans[] = {
    {SIS1100_32_SETVMESPACE, sis1100_ioctl32, 0},
    {SIS3100_32_VME_PROBE, 0, 0},
    {SIS3100_32_VME_READ, 0, 0},
    {SIS3100_32_VME_WRITE, 0, 0},
    {SIS3100_32_VME_BLOCK_READ, sis1100_ioctl32, 0},
    {SIS3100_32_VME_BLOCK_WRITE, sis1100_ioctl32, 0},
    {SIS1100_32_CTRL_READ, 0, 0},
    {SIS1100_32_CTRL_WRITE, 0, 0},
    {SIS1100_32_PIPE, sis1100_ioctl32, 0},
    {SIS1100_32_MAPSIZE, 0, 0},
    {SIS1100_32_LAST_ERROR, 0, 0},
    {SIS1100_32_IDENT, sis1100_ioctl32, 0},
    {SIS1100_32_FIFOMODE, 0, 0},
    {SIS1100_32_IRQ_CTL, 0, 0},
    {SIS1100_32_IRQ_GET, 0, 0},
    {SIS1100_32_IRQ_ACK, 0, 0},
    {SIS1100_32_IRQ_WAIT, 0, 0},
    {SIS1100_32_MINDMALEN, 0, 0},
    {SIS1100_32_FRONT_IO, 0, 0},
    {SIS1100_32_FRONT_PULSE, 0, 0},
    {SIS1100_32_FRONT_LATCH, 0, 0},
 /* {SIS3100_32_VME_SUPER_BLOCK_READ, sis1100_ioctl32, 0}, */
    {SIS1100_32_WRITE_PIPE, sis1100_ioctl32, 0},
    {SIS1100_32_DMA_ALLOC, sis1100_ioctl32, 0},
    {SIS1100_32_DMA_FREE, sis1100_ioctl32, 0},
    {SIS5100_32_CCCZ, 0, 0},
    {SIS5100_32_CCCC, 0, 0},
    {SIS5100_32_CCCI, 0, 0},
    {SIS5100_32_CNAF, 0, 0},
    {SIS1100_32_SWAP, 0, 0},
    {SIS3100_32_TIMEOUTS, 0, 0},
    {SIS1100_32_DSP_LOAD, sis1100_ioctl32, 0},
    {SIS1100_32_DSP_RESET, 0, 0},
    {SIS1100_32_DSP_START, 0, 0},
    {SIS1100_32_DMA_MAP, sis1100_ioctl32, 0},
    {SIS1100_32_DMA_START, 0, 0},
    {SIS1100_32_DMA_STOP, 0, 0},
    {SIS1100_32_DMA_WAIT, 0, 0},
    {SIS1100_32_RESET, 0, 0},
    {SIS1100_32_REMOTE_RESET, 0, 0},
    {SIS1100_32_DEVTYPE, 0, 0},
    {SIS1100_32_DRIVERVERSION, 0, 0},
    {SIS1100_32_READ_EEPROM, sis1100_ioctl32, 0},
    {SIS1100_32_WRITE_EEPROM, sis1100_ioctl32, 0},
    {SIS1100_32_JTAG_ENABLE, 0, 0},
    {SIS1100_32_JTAG_CTRL, 0, 0},
    {SIS1100_32_JTAG_DATA, 0, 0},
    {SIS1100_32_JTAG_PUT, 0, 0},
    {SIS1100_32_JTAG_GET, 0, 0},
    {SIS1100_32_BELL_READ, 0, 0},
    {SIS1100_32_BELL_WRITE, 0, 0},
    {0, 0, },
};

void __init
sis1100_ioctl32_init(void)
{
    int i;

printk(KERN_WARNING "sis1100: sis1100_ioctl32_init called\n");
    for (i=0; sis1100_ioctl32_trans[i].cmd!=0; i++) {
        register_ioctl32_conversion(sis1100_ioctl32_trans[i].cmd,
                sis1100_ioctl32_trans[i].handler);
    }
}

void
sis1100_ioctl32_exit(void)
{
    int i;

printk(KERN_WARNING "sis1100: sis1100_ioctl32_exit called\n");
    for (i=0; sis1100_ioctl32_trans[i].cmd!=0; i++)
        unregister_ioctl32_conversion(sis1100_ioctl32_trans[i].cmd);
}
#endif /* HAVE_COMPAT_IOCTL */

#endif /* CONFIG_COMPAT */
