/* $ZEL: sis1100_init_remote.c,v 1.11.2.3 2003/08/07 11:51:37 wuestner Exp $ */

/*
 * Copyright (c) 2001-2003
 * 	Matthias Drochner, Peter Wuestner.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/pci.h>

#include "sis1100_sc.h"

void
sis1100_synch_s_handler(void* data)
{
    struct SIS1100_softc* sc=(struct SIS1100_softc*)data;
    u_int32_t status;

/* reenable IRQs, but not sis1100irq_prot_end */
    sis1100_enable_irq(sc, 0, irq_synch_chg|irq_reset_req|irq_prot_l_err);
    status=sis1100readreg(sc, sr);

    printk(KERN_INFO "SIS1100[%d]: synch_s_handler: status=0x%08x\n",
        sc->unit, status);

    if ((status&sr_synch)==sr_synch) sis1100_init_remote(sc);
}

void
sis1100_synch_handler(unsigned long data)
{
    struct SIS1100_softc* sc=(struct SIS1100_softc*)data;
    int res;
/*
    printk(KERN_NOTICE "SIS1100[%d]: synch_handler called\n");
*/
    res=schedule_task(&sc->link_up_task);
    if (!res) printk(KERN_NOTICE
        "SIS1100[%d]: sis1100_synch_s_handler not scheduled\n", sc->unit);
}

static int sharc_present(struct SIS1100_softc* sc)
{
    u_int32_t dsp_sc;
    int res;

    res=sis3100readreg(sc, dsp_sc, &dsp_sc, 0);
    if (res) {
        printk(KERN_INFO "SIS1100[%d]: read dsp_sc: res=%d\n", sc->unit, res);
        return 0;
    }
    return !!(dsp_sc&dsp_available);
}

static void
sis3100_dump_timeouts(struct SIS1100_softc* sc)
{
    u_int32_t stat;
    int berr_timer, long_timer, berr_time, long_time;

    down(&sc->sem_hw);
    stat=plxreadlocal0(sc, 0x800+0x100);
    up(&sc->sem_hw);

    berr_timer=(stat>>14)&3;
    long_timer=(stat>>12)&3;
    berr_time=0;
    long_time=0;
    switch (berr_timer) {
        case 0: berr_time=1250; break;
        case 1: berr_time=6250; break;
        case 2: berr_time=12500; break;
        case 3: berr_time=100000; break;
    }
    switch (long_timer) {
        case 0: long_time=1; break;
        case 1: long_time=10; break;
        case 2: long_time=50; break;
        case 3: long_time=100; break;
    }
    printk("SIS1100[%d]: berr_time=%d ns\n", sc->unit, berr_time);
    printk("SIS1100[%d]: long_time=%d ms\n", sc->unit, long_time);
}

static void
sis3100_set_timeouts(struct SIS1100_softc* sc, int berr, int arb)
/* berr in terms of 10**-9 s */
/* arb  in terms of 10**-3 s */
{
    int berr_timer, long_timer;
    u_int32_t bits;

    if (berr>12500)
        berr_timer=3;
    else if (berr>6250)
        berr_timer=2;
    else if (berr>1250)
        berr_timer=1;
    else 
        berr_timer=0;

    if (arb>50)
        long_timer=3;
    else if (arb>10)
        long_timer=2;
    else if (arb>1)
        long_timer=1;
    else 
        long_timer=0;

    bits=(long_timer|(berr_timer<<2))<<12;
    bits|=(~bits<<16)&0xf0000000;
    sis3100writereg(sc, vme_master_sc, bits, 0);
}

char* rnames[]={"PCI", "VME", "CAMAC"};

void
sis1100_init_remote(struct SIS1100_softc* sc)
{
#define MIN_FV 3
#define MAX_FV 5
    u_int32_t error, balance, typ, hv, fk, fv, stat;

    down(&sc->sem_hw);

    /*sis1100writereg(sc, cr, cr_rem_reset);*/ /* reset remote */
    flush_fifo(sc, "init_remote" , 0); /* clear local fifo */
    sis1100writereg(sc, p_balance, 0);
    sis1100readreg(sc, prot_error);

    sc->remote_ident=plxreadlocal0(sc, 0x800);
    error=sis1100readreg(sc, prot_error);
    balance=sis1100readreg(sc, p_balance);
    up(&sc->sem_hw);

    if (error || balance) {
        printk(KERN_ERR "SIS1100[%d]: error reading remote ident\n", sc->unit);
        printk(KERN_ERR "error=0x%x balance=%d\n", error, balance);
        flush_fifo(sc, "after reading ident" , 0); /* clear local fifo */
        /*dump_glink_status(sc, "init remote");*/
        return;
    }

    typ=sc->remote_ident&0xff;
    hv=(sc->remote_ident>>8)&0xff;
    fk=(sc->remote_ident>>16)&0xff;
    fv=(sc->remote_ident>>24)&0xff;
    printk(KERN_INFO "SIS1100[%d]:%s: remote ident: 0x%08x\n",
            sc->unit, sc->pcidev->slot_name, sc->remote_ident);
    if ((typ>0) && (typ<4))
        printk(KERN_INFO "SIS1100[%d]:%s: remote is %s\n",
                sc->unit, sc->pcidev->slot_name, rnames[typ-1]);
    else
        printk(KERN_ERR "SIS1100[%d]:%s: unknown remote type %d\n",
                sc->unit, sc->pcidev->slot_name, typ);
    printk(KERN_INFO "SIS1100[%d]: remote HW_ver %d FW_code %d FW_ver %d\n",
                sc->unit, hv, fk, fv);

    switch (sc->remote_ident&0xffffff) { /* type, HW version and FW code */
        case 0x010102: {
            if (fv<MIN_FV) {
                printk(KERN_ERR "SIS1100[%d]: remote firmware version too old;"
                        " at least version %d is required.\n",
                        sc->unit, MIN_FV);
                return;
            }
            if (fv>MAX_FV) {
                printk(KERN_WARNING "SIS1100[%d]: Driver not tested with"
                        " remote firmware versions greater than %d.\n",
                        sc->unit, MAX_FV);
            }
        } break;
        case 0x010103: {
            printk(KERN_ERR "SIS CAMAC Controller is not supported yet.\n"
                "  please ask for a driver version 2.+\n");
            return;
        } break;
        default:
            printk(KERN_ERR "SIS1100[%d]: remote: unknown remote device\n",
                    sc->unit);
            return;
    }

    if (init_sdram(sc)<0)
        return;
    printk(KERN_INFO "SIS1100[%d]: size of SDRAM: 0x%Lx (%Ld MByte)\n",
        sc->unit, sc->sdram_size, sc->sdram_size>>20);

    sc->sharc_present=sharc_present(sc);
    sc->sharc_size=sc->sharc_present?0x400000:0;
    printk(KERN_INFO "SIS1100[%d]: SHARC is %spresent\n",
        sc->unit, sc->sharc_present?"":"not ");

    down(&sc->sem_hw);
    plxwritelocal0(sc, 0x800+0x104, 0x00fe0001);
    stat=plxreadlocal0(sc, 0x800+0x100);
    up(&sc->sem_hw);
    if (!(stat&0x10000)) {
        printk(KERN_WARNING "SIS1100[%d]: System Controller NOT enabled!\n",
            sc->unit);
    }
    sis3100_set_timeouts(sc, 5000, 10);
    sis3100_dump_timeouts(sc);

    sc->old_remote_ok=sc->remote_ok;
    sc->remote_ok=1;
    schedule_task(&sc->vme_irq_task);
#undef MIN_FV
#undef MAX_FV
}
