/* $ZEL: sis1100_linux_init.c,v 1.16.2.3 2003/08/07 11:51:38 wuestner Exp $ */

/*
 * Copyright (c) 2001
 * 	Matthias Drochner.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/iobuf.h>
#include <linux/wrapper.h>

#include "sis1100_sc.h"

int SIS1100_major = -1;
int SIS3100sdram_major = -1;
int SIS3100sharc_major = -1;

struct pci_device_id SIS1100_table[]={
    {
    PCI_VENDOR_FZJZEL, PCI_PRODUCT_FZJZEL_GIGALINK,
    PCI_ANY_ID, PCI_ANY_ID,
    0, 0,
    0
    },
    { 0 }
};

MODULE_AUTHOR("Peter Wuestner <P.Wuestner@fz-juelich.de>");
MODULE_DESCRIPTION("SIS1100 PCI-VME link/interface (http://zelweb.zel.kfa-juelich.de/projects/gigalink/)");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif
MODULE_SUPPORTED_DEVICE("sis1100/sis3100; http://www.struck.de/pcivme.htm");

struct SIS1100_softc *SIS1100_devdata[MAX_SIS1100_DEVICES];

struct file_operations SIS1100_fops = {
	owner:		THIS_MODULE,
	open:		sis1100_open,
	release:	sis1100_release,
	ioctl:		sis1100_ioctl,
	llseek:		sis1100_llseek,
	read:		sis1100_read,
	write:		sis1100_write,
	mmap:		sis1100_mmap,
};

struct file_operations SIS3100sdram_fops = {
	owner:		THIS_MODULE,
	open:		sis1100sdram_open,
	release:	sis1100_release,
	ioctl:		sis1100_ioctl,
	llseek:		sis1100_llseek,
	read:		sis1100_read,
	write:		sis1100_write,
};

struct file_operations SIS3100sharc_fops = {
	owner:		THIS_MODULE,
	open:		sis1100sharc_open,
	release:	sis1100_release,
	ioctl:		sis1100_ioctl,
	llseek:		sis3100sharc_llseek,
	read:		sis3100sharc_read,
	write:		sis3100sharc_write,
};

static void
SIS1100_linux_intr(int irq, void *vsc, struct pt_regs *regs)
{
    /*printk(KERN_INFO "SIS1100 got irq %d\n", irq);*/
    SIS1100_intr((struct SIS1100_softc *)vsc);
}

void __init SIS1100_print_info(void)
{
    printk(KERN_INFO "SIS1100 driver V1.3 (c) 31.07.2003 FZ Juelich\n");
}

int __init
SIS1100_linux_init(struct pci_dev *dev)
{
	struct SIS1100_softc *sc;
	int idx, res;
	u_long pmembase;
#ifdef USE_DEVFS
        char devname[100];
#endif

	printk(KERN_INFO "SIS1100: found %s at %s\n", dev->name, dev->slot_name);
/*
        printk(KERN_INFO "vendor=0x%04x\n", dev->vendor);
        printk(KERN_INFO "device=0x%04x\n", dev->device);
        printk(KERN_INFO "sub_vendor=0x%04x\n", dev->subsystem_vendor);
        printk(KERN_INFO "sub_device=0x%04x\n", dev->subsystem_device);
        printk(KERN_INFO "class=%d\n", dev->class);
        printk(KERN_INFO "name=>%s<\n", dev->name);
        printk(KERN_INFO "slot_name=>%s<\n", dev->slot_name);
*/
	for (idx = 0; idx < MAX_SIS1100_DEVICES; idx++) {
		if (!SIS1100_devdata[idx]) break;
	}
	if (idx == MAX_SIS1100_DEVICES) return -ENOMEM;

	sc = kmalloc(sizeof(struct SIS1100_softc), GFP_KERNEL);
	if (!sc) return -ENOMEM;
	SIS1100_devdata[idx] = sc;
	sc->unit = idx;

	init_waitqueue_head(&sc->sis1100_wait);
	init_waitqueue_head(&sc->irq_wait);
	init_timer(&sc->link_up_timer);
        sc->link_up_timer.function=sis1100_synch_handler;
        sc->link_up_timer.data=(unsigned long)sc;
        INIT_TQUEUE(&sc->link_up_task, sis1100_synch_s_handler, sc);
        INIT_TQUEUE(&sc->vme_irq_task, sis1100_vme_irq_handler, sc);
        init_MUTEX (&sc->sem_hw);
        init_MUTEX (&sc->sem_fdata_list);
        spin_lock_init(&sc->lock_intcsr);
        INIT_LIST_HEAD(&sc->fdata_list_head);

        sc->plxmembase=0;
        sc->plxlocallen0=0;
        sc->plxlocallen1=0;

	pmembase = pci_resource_start(dev, 0);
	sc->plxmemlen = pci_resource_len(dev, 0);
	sc->plxmembase = ioremap(pmembase, sc->plxmemlen);
        /*printk(KERN_INFO "SIS1100: plxmembase=0x%08lx\n", pmembase);*/
	if (!sc->plxmembase) {
            printk(KERN_ERR "SIS1100: can't map plx space\n");
            res=-ENOMEM;
	    goto fehler_ioremap;
        }

	pmembase = pci_resource_start(dev, 2);
	sc->plxlocallen0 = pci_resource_len(dev, 2);
	sc->plxlocalbase0 = ioremap(pmembase, sc->plxlocallen0);
        /*printk(KERN_INFO "SIS1100: plxlocalbase0=0x%08lx\n", pmembase);*/
	if (!sc->plxlocalbase0) {
            printk(KERN_ERR "SIS1100: can't map register space\n");
            res=-ENOMEM;
	    goto fehler_ioremap;
        }
	pmembase = pci_resource_start(dev, 3);
	sc->plxlocallen1 = pci_resource_len(dev, 3);
	if (!sc->plxlocallen1) {
            printk(KERN_WARNING "SIS1100: remote space (VME) not mapped."
                                " BIOS too old?\n");
	    printk(KERN_WARNING "SIS1100: mmap not available\n");
        } else {
	    sc->plxlocalbase1 = ioremap(pmembase, sc->plxlocallen1);
            /*printk(KERN_INFO "SIS1100: plxlocalbase1=0x%08lx\n", pmembase);*/
	    if (!sc->plxlocalbase1) {
                printk(KERN_WARNING "SIS1100: can't map remote space\n");
	        printk(KERN_WARNING "SIS1100: mmap not available\n");
                sc->plxlocallen1=0;
            }
        }
	res = pci_request_regions(dev, "SIS1100");
	if (res)
		goto fehler_ioremap;

	res = request_irq(dev->irq, SIS1100_linux_intr, SA_SHIRQ, "SIS1100", sc);
	if (res)
		goto fehler_request_regions;
        printk(KERN_INFO "SIS1100: irq %d installed\n", dev->irq);

	sc->pcidev = dev;
	pci_set_drvdata(dev, sc);

	pci_set_master(dev);

        if ((res=alloc_kiovec(1, &sc->iobuf))<0) {
                sc->iobuf=0;
                goto fehler_request_irq;
        }

        sc->descbuf.size=(((MAXSIZE_KIO-1)>>PAGE_SHIFT)+1)
                *sizeof(struct plx9054_dmadesc);
        sc->descbuf.cpu_addr=pci_alloc_consistent(sc->pcidev,
    	        sc->descbuf.size, &sc->descbuf.dma_handle);
        if (!sc->descbuf.cpu_addr) {
                printk(KERN_ERR "SIS1100: pci_alloc_consistent failed\n");
                res=-ENOMEM;
                goto fehler_alloc_kiovec;
        }
        mem_map_reserve(virt_to_page(sc->descbuf.cpu_addr));
/*
        printk(KERN_INFO "SIS1100: descbuf.dma_handle=0x%08x\n",
                sc->descbuf.dma_handle);
*/
    	res=SIS1100_init(sc);
	if (res) {
	    goto fehler_alloc_descbuf;
	}

#ifdef USE_DEVFS
        sc->devfs_dev=0;
        sc->devfs_dev_sd=0;
        sc->devfs_dev_sh=0;
        sprintf(devname, "sis1100/%d", sc->unit);
        sc->devfs_dev=devfs_register(NULL, devname, DEVFS_FL_AUTO_DEVNUM,
                0, 0, S_IFCHR|S_IRUGO|S_IWUGO, &SIS1100_fops,
                (void*)(0xffff0000|(0<<8)|sc->unit));
        sprintf(devname, "sis1100/%dsd", sc->unit);
        sc->devfs_dev_sd=devfs_register(NULL, devname, DEVFS_FL_AUTO_DEVNUM,
                0, 0, S_IFCHR|S_IRUGO|S_IWUGO, &SIS3100sdram_fops,
                (void*)(0xffff0000|(1<<8)|sc->unit));
        sprintf(devname, "sis1100/%dsh", sc->unit);
        sc->devfs_dev_sh=devfs_register(NULL, devname, DEVFS_FL_AUTO_DEVNUM,
                0, 0, S_IFCHR|S_IRUGO|S_IWUGO, &SIS3100sharc_fops,
                (void*)(0xffff0000|(2<<8)|sc->unit));
#endif

    	return 0;

fehler_alloc_descbuf:
        mem_map_unreserve(virt_to_page(sc->descbuf.cpu_addr));
        pci_free_consistent(sc->pcidev, sc->descbuf.size,
                sc->descbuf.cpu_addr, sc->descbuf.dma_handle);
fehler_alloc_kiovec:
        free_kiovec(1, &sc->iobuf);
fehler_request_irq:
	free_irq(dev->irq, sc);
fehler_request_regions:
	pci_release_regions(dev);
fehler_ioremap:
	del_timer_sync(&sc->link_up_timer);
	if (sc->plxmembase) iounmap((void *)sc->plxmembase);
	if (sc->plxlocalbase0) iounmap((void *)sc->plxlocalbase0);
	if (sc->plxlocalbase1) iounmap((void *)sc->plxlocalbase1);

	SIS1100_devdata[sc->unit] = 0;
	kfree(sc);
	pci_set_drvdata(dev, NULL);
    	return res;
}

void __exit
SIS1100_linux_done(struct pci_dev *dev)
{
	struct SIS1100_softc *sc;

	sc = pci_get_drvdata(dev);
        free_kiovec(1, &sc->iobuf);
        mem_map_unreserve(virt_to_page(sc->descbuf.cpu_addr));
        pci_free_consistent(sc->pcidev, sc->descbuf.size,
                sc->descbuf.cpu_addr, sc->descbuf.dma_handle);

#ifdef USE_DEVFS
        devfs_unregister(sc->devfs_dev);
        devfs_unregister(sc->devfs_dev_sd);
        devfs_unregister(sc->devfs_dev_sh);
#endif
       
	SIS1100_done(sc);
	free_irq(dev->irq, sc);
	del_timer_sync(&sc->link_up_timer);

	iounmap((void *)sc->plxmembase);
	iounmap((void *)sc->plxlocalbase0);
	if (sc->plxlocalbase1) iounmap((void *)sc->plxlocalbase1);
	SIS1100_devdata[sc->unit] = 0;

	kfree(sc);
	pci_release_regions(dev);
	pci_set_drvdata(dev, NULL);
}

#ifdef USE_DEVFS
int __init
SIS1100_linux_drvinit(void)
{
    return (0);
}
#else
int __init
SIS1100_linux_drvinit(void)
{
    SIS1100_major = register_chrdev(0, "SIS1100", &SIS1100_fops);
    SIS3100sdram_major = register_chrdev(0, "SIS3100sdram", &SIS3100sdram_fops);
    SIS3100sharc_major = register_chrdev(0, "SIS3100sharc", &SIS3100sharc_fops);
    printk(KERN_INFO "SIS1100 major=%d\n", SIS1100_major);
    printk(KERN_INFO "SIS3100 sdram=%d\n", SIS3100sdram_major);
    printk(KERN_INFO "SIS3100 sharc=%d\n", SIS3100sharc_major);
    if ((SIS1100_major<0) || (SIS3100sdram_major<0) || (SIS3100sharc_major<0)) {
        printk(KERN_INFO "SIS1100: unable to register devices\n");
        if (SIS1100_major>0) unregister_chrdev(SIS1100_major, "SIS1100");
        if (SIS3100sdram_major>0) unregister_chrdev(SIS3100sdram_major, "SIS3100sdram");
        if (SIS3100sharc_major>0) unregister_chrdev(SIS3100sharc_major, "SIS3100sharc");
        return -EBUSY;
    }
    return (0);
}
#endif

#ifdef USE_DEVFS
void __exit
SIS1100_linux_drvdone(void)
{}
#else
void __exit
SIS1100_linux_drvdone(void)
{
    unregister_chrdev(SIS1100_major, "SIS1100");
    unregister_chrdev(SIS3100sdram_major, "SIS3100sdram");
    unregister_chrdev(SIS3100sharc_major, "SIS3100sharc");
}
#endif
