/* $ZEL: sis1100_read.c,v 1.10.2.3 2003/08/07 11:51:41 wuestner Exp $ */

/*
 * Copyright (c) 2001-2003
 * 	Peter Wuestner.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/wrapper.h>
#include <linux/pci.h>
#include <asm/uaccess.h>

#include "sis1100_sc.h"

static int
check_access(loff_t ppos, loff_t maxsize, size_t count, int datasize,
        const char* buf)
{
    if (ppos>=maxsize) {
    	printk(KERN_INFO "sis1100_r/w: start addr out of range\n");
    	return -EINVAL;
    }

    if ((ppos+count>maxsize)||(ppos+count<ppos)) {
    	printk(KERN_INFO "sis1100_r/w: end addr out of range\n");
    	return -EINVAL;
    }
    
    if ((ppos|(off_t)buf|count) & (datasize-1)) {
    /* == (ppos&(datasize-1))||(buf&(datasize-1))...*/
    	printk(KERN_INFO "sis1100_r/w: unaligned access\n");
    	return -EINVAL;
    }
    return 0;
}

ssize_t sis1100_read(struct file* file, char* buf, size_t count,
    loff_t* ppos)
{
    struct SIS1100_softc* sc=SIS1100SC(file);
    struct SIS1100_fdata* fd=SIS1100FD(file);
    int res;
    int32_t am;
    u_int32_t datasize;
    int space, fifo;
    loff_t maxsize;

    if (!sc->remote_ok) return -ENXIO;
    if (!count) return 0;

    if (fd->subdev==sis1100_subdev_ram) {
        am=-1;
        datasize=4;
        space=6;
        fifo=0;
        maxsize=sc->sdram_size;
    } else {
        am=fd->vmespace_am;
        datasize=fd->vmespace_datasize;
        space=1;
        fifo=fd->fifo_mode;
        maxsize=0xffffffffU;
    }

    if (check_access(*ppos, maxsize, count, datasize, buf))
        return -EINVAL;
    if (!access_ok(VERIFY_WRITE, buf, count))
        return -EPERM;

#ifndef ALLWAYS_DMA
    if (count==datasize) {
        u_int32_t var;
        if ((res=sis1100_tmp_read(sc, *ppos, am, datasize, space, &var))!=0) {
            fd->last_prot_err=res;
            res=-EIO;
        } else {
            switch (datasize) {
	    case 4: __put_user(var, (u_int32_t*)buf); break;
	    case 2: __put_user(var, (u_int16_t*)buf); break;
	    case 1: __put_user(var, (u_int8_t*)buf); break;
            }
            res=count;
        }
    } else {
        if ((datasize==4) && fd->mindmalen_r && (count>=fd->mindmalen_r)) {
#endif
            res=sis1100_read_dma(fd, *ppos, am, datasize,
                    space, fifo, count, buf, &fd->last_prot_err);
#ifndef ALLWAYS_DMA
        } else {
            res=sis1100_read_loop(fd, *ppos, am, datasize,
                    space, fifo, count, buf, &fd->last_prot_err);
        }
    }
#endif

    if (res>=0) *ppos+=res;
    return res;
}

ssize_t sis1100_write(struct file* file, const char* buf, size_t count,
    loff_t* ppos)
{
    struct SIS1100_softc* sc=SIS1100SC(file);
    struct SIS1100_fdata* fd=SIS1100FD(file);
    int res;
    int32_t am;
    u_int32_t datasize;
    int space, fifo;
    loff_t maxsize;

    if (!sc->remote_ok) return -ENXIO;
    if (!count) return 0;

    if (fd->subdev==sis1100_subdev_ram) {
        am=-1;
        datasize=4;
        space=6;
        fifo=0;
        maxsize=sc->sdram_size;
    } else {
        am=fd->vmespace_am;
        datasize=fd->vmespace_datasize;
        space=1;
        fifo=fd->fifo_mode;
        maxsize=0xffffffffU;
    }

    if (check_access(*ppos, maxsize, count, datasize, buf))
        return -EINVAL;
    if (!access_ok(VERIFY_READ, buf, count))
        return -EPERM;

#ifndef ALLWAYS_DMA
    if (count==datasize) {
        u_int32_t data;
        switch (datasize) {
        case 1: __get_user(data, (u_int8_t*)buf); break;
        case 2: __get_user(data, (u_int16_t*)buf); break;
        case 4: __get_user(data, (u_int32_t*)buf); break;
        default: data=0;
        }
        if ((res=sis1100_tmp_write(sc, *ppos, am, datasize,
                space, data))!=0) {
            fd->last_prot_err=res;
            res=-EIO;
        } else
            res=count;
    } else {
        if ((datasize==4) && fd->mindmalen_w && (count>=fd->mindmalen_w)) {
#endif
            res=sis1100_write_dma(fd, *ppos, am, datasize,
                    space, fifo, count, buf, &fd->last_prot_err);
#ifndef ALLWAYS_DMA
        } else {
            res=sis1100_write_loop(fd, *ppos, am, datasize,
                    space, fifo, count, buf, &fd->last_prot_err);
        }
    }
#endif

    if (res>=0) *ppos+=res;
    return res;
}

/* SEEK_... normally defined in stdio.h, fcntl.h and unistd.h */
#define SEEK_SET 0
#define SEEK_CUR 1
#define SEEK_END 2

loff_t sis1100_llseek(struct file* file, loff_t offset, int orig)
{
    struct SIS1100_softc* sc=SIS1100SC(file);
    struct SIS1100_fdata* fd=SIS1100FD(file);
    loff_t old=file->f_pos;

    switch (orig) {
        case SEEK_SET: file->f_pos=offset&0xffffffff; break;
        case SEEK_CUR: file->f_pos+=offset; break;
        case SEEK_END:
            if (fd->subdev==sis1100_subdev_ram) {
                file->f_pos=sc->sdram_size+offset;
            } else
                return -EINVAL;
            break;
    }
    if ((file->f_pos<0) ||
        (file->f_pos>
            ((fd->subdev==sis1100_subdev_ram)?sc->sdram_size:0xffffffffU))) {
        file->f_pos=old;
        return -EINVAL;
    }
    return file->f_pos;
}
