/********************************************************************\

  Bridge between two networks

  Subscription to top-level server list DIS_DNS/SERVER_LIST not via
  AddService() ensures AddService() only called from infoHandler(),
  thus serialized by DIM and no Mutex is necessary.

  Oliver Grimm, May 2010

\********************************************************************/

#define SERVER_NAME "Bridge"
#include "Evidence.h"

#include <string>
#include <vector>

const int DEFAULT_PORT = 2505;

using namespace std;

// Class declaration
class Bridge: public DimClient, public EvidenceServer {

	struct Item {
	  DimStampedInfo *DataItem;
	  DimService *Service;
	  char *Data;
	};
	vector<struct Item> List;

	DimInfo *ServerList;
	
    void infoHandler();
	void AddService(string, const char *);
	void RemoveService(string);
   
  public:
    Bridge(char *, int);
    ~Bridge();
}; 

//
// Constructor
//
Bridge::Bridge(char *Name, int Port): EvidenceServer(SERVER_NAME) {

  // Set primary DIM network to subscribe from
  DimClient::setDnsNode(Name, Port);

  // Subsribe to top-level server list
  ServerList = new DimInfo((char *) "DIS_DNS/SERVER_LIST", NO_LINK, this);
}

//
// Destructor: Delete all subscriptions and services
//
Bridge::~Bridge() {

  while (List.size() != 0) RemoveService(List[0].DataItem->getName());  
  delete ServerList;
}

//
// Service subscription and repeating
//
void Bridge::infoHandler() {

  DimInfo *I = getInfo();

  // Check if service available
  if (!ServiceOK(I)) return;

  // If service is DIS_DNS/SERVER_LIST, subscribe to all SERVICE_LIST services
  if (strcmp(I->getName(), "DIS_DNS/SERVER_LIST") == 0) {	
	char *Token = strtok(I->getString(), "+-!@");	
	while (Token != NULL) {
	  if (*I->getString() == '-' || *I->getString() == '!') {
		RemoveService(string(Token)+"/SERVICE_LIST");
	  }
	  else AddService(string(Token)+"/SERVICE_LIST", "C");

	  // Skip server IP address and process ID
	  Token = strtok(NULL, "|");
	  Token = strtok(NULL, "@");
	}	
	return;
  }

  // If service is SERVICE_LIST, scan and subscribe/unsubscribe to services
  if (strstr(I->getName(), "/SERVICE_LIST") != NULL) {
	char *Type, *Name = strtok(I->getString(), "+-!|");
	while (Name != NULL) {
	  // Only consider DIM services (not commands and RPCs)
      if (((Type = strtok(NULL, "\n")) != NULL) &&
		  (strstr(Type, "|CMD") == NULL) && (strstr(Type, "|RPC") == NULL)) {
	    if (*I->getString() == '-' || *I->getString() == '!') RemoveService(Name);
		else {
		  Type[strlen(Type)-1] = '\0'; // Isolate service format
		  AddService(Name, Type);
		}
	  }
	  Name = strtok(NULL, "|");
	}
	return;
  }

  // Identify service and repeat to secondary DNS
  for (int Service=0; Service<List.size(); Service++) if (I == List[Service].DataItem) {

	// Ignores repeating DIS_DNS services
	if (List[Service].Service == NULL) break;

    // Copy service data
    delete[] List[Service].Data;
    List[Service].Data = new char [I->getSize()];
    memcpy(List[Service].Data, I->getData(), I->getSize());

	// Set new service properties and update service
    List[Service].Service->setQuality(I->getQuality());
    List[Service].Service->setTimestamp(I->getTimestamp(), I->getTimestampMillisecs());	
	List[Service].Service->updateService(List[Service].Data, I->getSize());
  }
}

//
// Add service subscription
//
void Bridge::AddService(string Name, const char *Format) {

  // Check if already subscribed to this service
  for (int i=0; i<List.size(); i++) {
	if (Name == List[i].DataItem->getName()) return;
  }

  // Create subscription and new service to secondary DNS (do not forward DIS_DNS services)
  struct Item New;
  
  New.Data = NULL;
  if (Name.find("DIS_DNS/") != string::npos) New.Service = NULL;
  else New.Service = new DimService(Name.c_str(), (char *) Format, New.Data, 0);
  New.DataItem = new DimStampedInfo(Name.c_str(), NO_LINK, this);
  List.push_back(New);
}


//
// Remove service from watch list
//
void Bridge::RemoveService(string Name) {

  // Find service index
  vector<struct Item>::iterator E;
  for (E=List.begin(); E<List.end(); ++E) if (Name == (*E).DataItem->getName()) {
	delete (*E).DataItem;
	delete (*E).Service;
	delete[] (*E).Data;
	List.erase(E);
  }
}


//	    
// Main program
//
int main(int argc, char *argv[]) {

  if (argc == 1) {
	printf("Usage: %s <address of primary DNS> [port] (default port %d)\n", argv[0], DEFAULT_PORT);
	exit(EXIT_FAILURE);
  }
          
  // Static ensures calling of destructor by exit()
  static Bridge Class(argv[1], argc>2 ? atoi(argv[2]) : DEFAULT_PORT);
  
  // Sleep until signal caught
  pause();
}
