/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Harald Kornmayer 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// MHexagon                                                                 //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHexagon.h"

#include <fstream>
#include <iostream>

#include <TVirtualPad.h>  // gPad

#include "MGeomPix.h"     // GetX

ClassImp(MHexagon);

using namespace std;

// ------------------------------------------------------------------------
//
//   default constructor for MHexagon
//
MHexagon::MHexagon()
{
}

// ------------------------------------------------------------------------
//
//    normal constructor for MHexagon
//
MHexagon::MHexagon(Float_t x, Float_t y, Float_t d)
: TAttLine(1, 1, 1), TAttFill(0, 1001), fX(x), fY(y), fD(d)
{
}

// ------------------------------------------------------------------------
//
//    normal constructor for MHexagon
//
MHexagon::MHexagon(MGeomPix &pix)
: TAttLine(1, 1, 1), TAttFill(0, 1001)
{
    fX = pix.GetX();
    fY = pix.GetY();
    fD = pix.GetD();
}

// ------------------------------------------------------------------------
//
//    copy constructor for MHexagon
//
MHexagon::MHexagon(const MHexagon &hexagon)
{
    ((MHexagon&) hexagon).Copy(*this);
} 

// ------------------------------------------------------------------------
//
//     copy this hexagon to hexagon
//
void MHexagon::Copy(TObject &obj)
#if ROOT_VERSION_CODE > ROOT_VERSION(3,04,01)
const
#endif

{
    MHexagon &hex = (MHexagon&) obj;

    TObject::Copy(obj);
    TAttLine::Copy(hex);
    TAttFill::Copy(hex);

    hex.fX = fX;
    hex.fY = fY;
    hex.fD = fD;
}

// ------------------------------------------------------------------------
//
// compute the distance of a point (px,py) to the Hexagon
// this functions needed for graphical primitives, that
// means without this function you are not able to interact
// with the graphical primitive with the mouse!!!
//
// All calcutations are running in pixel coordinates
//
Int_t MHexagon::DistancetoPrimitive(Int_t px, Int_t py)
{
    //
    //  compute the distance of the Point to the center of the Hexagon
    //
    const Int_t pxhex = gPad->XtoAbsPixel(fX);
    const Int_t pyhex = gPad->YtoAbsPixel(fY);

    const Double_t x = TMath::Abs(px-pxhex);
    const Double_t y = TMath::Abs(py-pyhex);

    const Double_t disthex = TMath::Sqrt(x*x + y*y);

    if (disthex==0)
        return 0;

    const Double_t cosa = x / disthex;
    const Double_t sina = y / disthex;

    //
    // comput the distance of hexagon center to pixel border
    //
    const Double_t dx = fD/2 * cosa;
    const Double_t dy = fD/2 * sina;

    const Int_t pxborder = gPad->XtoAbsPixel(fX + dx);
    const Int_t pyborder = gPad->YtoAbsPixel(fY + dy);

    const Double_t bx = pxhex-pxborder;
    const Double_t by = pyhex-pyborder;

    const Double_t distborder = TMath::Sqrt(bx*bx + by*by);

    //
    //  compute the distance from the border of Pixel
    //  here in the first implementation is just circle inside
    //
    return distborder < disthex ? (int)(disthex-distborder+1) : 0;
}

// ------------------------------------------------------------------------
//
// compute the distance of a point (px,py) to the Hexagon in world
// coordinates. Return -1 if inside.
//
Float_t MHexagon::DistanceToPrimitive(Float_t px, Float_t py)
{
    //
    //  compute the distance of the Point to the center of the Hexagon
    //
    const Double_t dx = px-fX;
    const Double_t dy = py-fY;

    const Double_t disthex = TMath::Sqrt(dx*dx + dy*dy);

    //
    //  compute the distance from the border of Pixel
    //  here in the first implementation is just circle inside
    //
    return fD*0.5772 < disthex ? disthex-fD*0.5772 : -1;
}

// ------------------------------------------------------------------------
//
//  Draw this ellipse with new coordinate
//
void MHexagon::DrawHexagon(Float_t x, Float_t y, Float_t d)
{ 
    MHexagon *newhexagon = new MHexagon(x, y, d);

    TAttLine::Copy(*newhexagon);
    TAttFill::Copy(*newhexagon);

    newhexagon->SetBit(kCanDelete);
    newhexagon->AppendPad();
}

/*
// ------------------------------------------------------------------------
//
//  This is the first test of implementing a clickable interface
//  for one pixel
//
void MHexagon::ExecuteEvent(Int_t event, Int_t px, Int_t py)
{
    switch (event)
    {
    case kButton1Down:
        cout << endl << "kButton1Down" << endl;
        SetFillColor(2);
        gPad->Modified();
        break;

    case kButton1Double:
        SetFillColor(0);
        gPad->Modified();
        break;
        //  case kMouseEnter:
        //     printf ("\n Mouse inside object \n" ) ;
        //     break;
    }
}
*/

// ------------------------------------------------------------------------
//
//  list this hexagon with its attributes
//
void MHexagon::ls(const Option_t *) const
{
    TROOT::IndentLevel();
    Print();
}

// ------------------------------------------------------------------------
//
//  paint this hexagon with its attribute
//
void MHexagon::Paint(Option_t *)
{
    PaintHexagon(fX, fY, fD);
}

// ------------------------------------------------------------------------
//
//  draw this hexagon with the coordinates
//
void MHexagon::PaintHexagon(Float_t inX, Float_t inY, Float_t inD)
{ 
    const Int_t np = 6;

    const Float_t dx[np+1] = { .5   , 0.    , -.5   , -.5   , 0.    ,  .5   , .5    };
    const Float_t dy[np+1] = { .2886,  .5772,  .2886, -.2886, -.5772, -.2886, .2886 };

    TAttLine::Modify();    // Change line attributes only if neccessary
    TAttFill::Modify();    // Change fill attributes only if neccessary

    //
    //  calculate the positions of the pixel corners
    //
    Float_t x[np+1], y[np+1];
    for (Int_t i=0; i<np+1; i++)
    {
        x[i] = inX + dx[i]*inD;
        y[i] = inY + dy[i]*inD;
    }

    //
    //   paint the pixel
    //
    if (GetFillColor())
        gPad->PaintFillArea(np, x, y);

    if (GetLineStyle())
        gPad->PaintPolyLine(np+1, x, y);
}

// ------------------------------------------------------------------------
//
//  print/dump this hexagon with its attributes
//
void MHexagon::Print(Option_t *) const
{
    cout << "MHexagon: ";
    cout << "x=" << fX << "mm y=" << fY << "mm r=" << fD << "mm" << endl;

    cout << " Line:";
    cout << " Color=" << GetLineColor() << ",";
    cout << " Style=" << GetLineStyle() << ",";
    cout << " Width=" << GetLineWidth() << endl;
    cout << " Fill:";
    cout << " Color=" << GetFillColor() << ",";
    cout << " Style=" << GetFillStyle() << endl;
}

// ------------------------------------------------------------------------
//
// Save primitive as a C++ statement(s) on output stream out
//
void MHexagon::SavePrimitive(ofstream &out, Option_t *)
{
    if (gROOT->ClassSaved(Class()))
       out << "   ";
    else
       out << "   MHexagon *";

    out << "hexagon = new MHexagon(" << fX << "," << fY << "," << fD << ");" << endl;

    SaveFillAttributes(out, "hexagon");
    SaveLineAttributes(out, "hexagon");

    out << "   hexagon->Draw();" << endl;
}

