/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Abelardo Moralejo 7/2002  <mailto:moralejo@pd.infn.it>
!   Author(s): Thomas Bretz 2002  <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2002-2003
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//   MCerPhotCalc
//
//   This is a task which calculates the number of photons from the FADC
//   time slices. It weights the each slice according to the numbers in
//   the array fWeight (default: all slices added up with weight 1).
//
//   The weights are rescaled, such that sum(weigths)=num slices
//
//  Input Containers:
//   MRawRunHeader, MRawEvtData, MPedestalCam
//
//  Output Containers:
//   MCerPhotEvt
//
//////////////////////////////////////////////////////////////////////////////
#include "MCerPhotCalc.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MMcRunHeader.hxx"

#include "MRawRunHeader.h"
#include "MRawEvtData.h"       // MRawEvtData::GetNumPixels
#include "MCerPhotEvt.h"
#include "MPedestalPix.h"
#include "MPedestalCam.h"
#include "MRawEvtPixelIter.h"

ClassImp(MCerPhotCalc);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor.
//
MCerPhotCalc::MCerPhotCalc(const char *name, const char *title)
{
    fName  = name  ? name  : "MCerPhotCalc";
    fTitle = title ? title : "Calculate pixel signal from FADC data";

    AddToBranchList("MRawEvtData.fHiGainPixId");
    AddToBranchList("MRawEvtData.fLoGainPixId");
    AddToBranchList("MRawEvtData.fHiGainFadcSamples");
    AddToBranchList("MRawEvtData.fLoGainFadcSamples");

    SetDefaultWeights();
}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawRunHeader
//  - MRawEvtData
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//  - MCerPhotEvt
//
Int_t MCerPhotCalc::PreProcess(MParList *pList)
{
    fRunHeader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!fRunHeader)
    {
        *fLog << dbginf << "MRawRunHeader not found... aborting." << endl;
        return kFALSE;
    }

    fRawEvt = (MRawEvtData*)pList->FindObject("MRawEvtData");
    if (!fRawEvt)
    {
        *fLog << dbginf << "MRawEvtData not found... aborting." << endl;
        return kFALSE;
    }

    fPedestals = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
    if (!fPedestals)
    {
        *fLog << dbginf << "MPedestalCam not found... aborting." << endl;
        return kFALSE;
    }

    fCerPhotEvt = (MCerPhotEvt*)pList->FindCreateObj("MCerPhotEvt");
    if (!fCerPhotEvt)
        return kFALSE;

    // Calculate quadratic sum of weights:
    Doubla_t sum = 0;

    fSumQuadWeights = 0;
    for (Int_t i=0; i<fWeight.GetSize(); i++)
        sum += fWeight[i];

    for (Int_t i=0; i<fWeight.GetSize(); i++)
    {
        fWeight[i] /= sum;
        fSumQuadWeights += fWeight[i]*fWeight[i];
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Check for the run type and camera version.
// If the file is a MC file and the used camera version is <= 40
// we enable a fix for truncated pedestal means in this version.
//
Bool_t MCerPhotCalc::ReInit(MParList *pList)
{
    const MRawRunHeader *runheader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!runheader)
    {
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;
        return kTRUE;
    }

    if (fRunHeader->GetNumSamplesHiGain() != fWeight.GetSize())
    {
        *fLog << dbginf << "Number of FADC slices (" << fRunHeader->GetNumSamplesHiGain() <<") is different from assumed one (" << fWeight.GetSize() << ")... aborting." << endl;
        return kFALSE;
    }

    Bool_t fIsMcFile = runheader->GetRunType() == kRTMonteCarlo;
    if (!fIsMcFile)
        return kTRUE;

    ScalePedestals();

    MMcRunHeader *mcrunheader = (MMcRunHeader*)pList->FindObject("MMcRunHeader");
    if (!mcrunheader)
    {
        *fLog << warn << dbginf << "Warning - cannot check for camera version, MC run header not found." << endl;
        return kTRUE;
    }

    fEnableFix = kFALSE;
    if (mcrunheader->GetCamVersion() <= 40)
        fEnableFix = kTRUE;

    return kTRUE;
}

void MCerPhotCalc::ScalePedestals()
{
    Int_t n = 577; //FIXME: fPedestals->GetNumPixel();

    for (int idx=0; idx<n; idx++)
    {
        MPedestalPix &ped = (*fPedestals)[idx];

        const Double_t offset = fEnableFix ? ped.GetPedestal()-0.5 : ped.GetPedestal();
        ped.Set(offset, ped.GetPedestalRms()*sqrt(fSumQuadWeights));
    }

    fPedestals->SetReadyToSave();
}

// --------------------------------------------------------------------------
//
// Calculate the integral of the FADC time slices and store them as a new
// pixel in the MCerPhotEvt container.
//
Int_t MCerPhotCalc::Process()
{
    //fCerPhotEvt->InitSize(fRawEvt->GetNumPixels());

//    if (fIsMcFile)
//        ScalePedestals();

    MRawEvtPixelIter pixel(fRawEvt);

    while (pixel.Next())
    {
        const UInt_t idx = pixel.GetPixelId();
        MPedestalPix &ped = (*fPedestals)[idx];

	//
	// sanity check (old MC files sometimes have pixids>577)
	//
        if (!fPedestals->CheckBounds(idx))
        {
	    *fLog << inf << "Pixel Index larger than camera... skipping event." << endl;
	    return kCONTINUE;
	}

        //
        // Calculate pixel signal unless it has all FADC slices empty:
        //
        Byte_t *ptr = pixel.GetHiGainSamples();

        Float_t nphot = 0;
        for(Int_t i=0; i<fWeight.GetSize(); i++)
            nphot += ptr[i]*fWeight[i];

        nphot -= ped.GetPedestal();

        fCerPhotEvt->AddPixel(idx, nphot, 0);

        // FIXME! Handling of Lo Gains is missing!
    }

    fCerPhotEvt->FixSize();
    fCerPhotEvt->SetReadyToSave();

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Set default values for the number of slices and weights:
//
void MCerPhotCalc::SetDefaultWeights()
{
    const Float_t dummy[15] = { 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 };
    fWeight.Set(15, dummy);
}
